/***************************************************************************
    smb4kcustomoptionsdialog  -  With this dialog the user can define
    custom Samba options for hosts or shares.
                             -------------------
    begin                : So Jun 25 2006
    copyright            : (C) 2006 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,   *
 *   MA  02110-1301 USA                                                    *
 ***************************************************************************/

// TQt includes
#include <tqlayout.h>
#include <tqlabel.h>
#include <tqframe.h>

// KDE includes
#include <tdelocale.h>
#include <tdeapplication.h>
#include <klineedit.h>
#include <kdebug.h>

// application specific includes
#include "smb4kcustomoptionsdialog.h"
#include "../core/smb4tdeglobal.h"
#include "../core/smb4kcore.h"
#include "../core/smb4ksambaoptionsinfo.h"
#include "../core/smb4ksambaoptionshandler.h"
#include "../core/smb4knetworkitems.h"
#include "../core/smb4ksettings.h"

using namespace Smb4TDEGlobal;


// FIXME: Maybe introduce a private class here?
static int default_port = -1;
static TQString default_protocol = TQString();
static bool default_kerberos = false;
static TQString default_uid = TQString();
static TQString default_gid = TQString();
#ifndef __FreeBSD__
static TQString default_filesystem = TQString();
static bool default_readwrite = true;
#endif

static int port_value = -1;
static TQString protocol_value = TQString();
static bool kerberos_value = false;
static TQString uid_value = TQString();
static TQString gid_value = TQString();
#ifndef __FreeBSD__
static TQString filesystem_value = TQString();
static bool readwrite_value = true;
#endif

static bool port_changed_ok = false;
static bool protocol_changed_ok = false;
static bool kerberos_changed_ok = false;
static bool uid_changed_ok = false;
static bool gid_changed_ok = false;
#ifndef __FreeBSD__
static bool filesystem_changed_ok = false;
static bool readwrite_changed_ok = false;
#endif

static bool port_changed_default = false;
static bool protocol_changed_default = false;
static bool kerberos_changed_default = false;
static bool uid_changed_default = false;
static bool gid_changed_default = false;
#ifndef __FreeBSD__
static bool filesystem_changed_default = false;
static bool readwrite_changed_default = false;
#endif


Smb4KCustomOptionsDialog::Smb4KCustomOptionsDialog( Smb4KHostItem *host, TQWidget *parent, const char *name )
: KDialogBase( Plain, i18n( "Custom Options" ), User1|Ok|Cancel, Ok, parent, name, true, true ), m_type( Host ), m_host_item( host ), m_share_item( NULL ), m_homes_user( TQString() )
{
  m_initialized = true;

  setButtonGuiItem( User1, KStdGuiItem::defaults() );

  setWFlags( TQt::WDestructiveClose );

  setupDialog();
}


Smb4KCustomOptionsDialog::Smb4KCustomOptionsDialog( Smb4KShareItem *share, TQWidget *parent, const char *name )
: KDialogBase( Plain, i18n( "Custom Options" ), User1|Ok|Cancel, Ok, parent, name, true, true ), m_type( Share ), m_host_item( NULL ), m_share_item( share ), m_homes_user( TQString() )
{
  if ( TQString::compare( share->name(), "homes" ) != 0 )
  {
    m_initialized = true;
  }
  else
  {
    m_homes_user = specifyUser( share->host(), kapp->mainWidget() ? kapp->mainWidget() : 0, "SpecifyUser" );
    m_initialized = m_homes_user.isEmpty() ? false : true;
  }

  setButtonGuiItem( User1, KStdGuiItem::defaults() );

  setWFlags( TQt::WDestructiveClose );

  setupDialog();
}


Smb4KCustomOptionsDialog::~Smb4KCustomOptionsDialog()
{
}


void Smb4KCustomOptionsDialog::setupDialog()
{
  // The Smb4KSambaOptionsInfo object:
  Smb4KSambaOptionsInfo *info = NULL;

  // We need this later to decide if the "Default"
  // button needs to be enabled:
  bool enable_default_button = false;

  // These are the input widgets we need below:
  m_port_input = NULL;
  m_kerberos = NULL;
  m_proto_input = NULL;
  m_uid_input = NULL;
  m_gid_input = NULL;
#ifndef __FreeBSD__
  m_fs_input = NULL;
  m_rw_input = NULL;
#endif

  // Set-up the widget:
  TQFrame *frame = plainPage();

  TQGridLayout *grid = new TQGridLayout( frame );
  grid->setSpacing( 5 );
  grid->setMargin( 0 );

  // The following widgets are independent of the type of
  // the network item:
  TQLabel *location_label = new TQLabel( m_type == Host ? i18n( "Host:" ) : i18n( "Share:" ), frame );
  KLineEdit *location = new KLineEdit( m_type == Host ? m_host_item->name() : "//"+m_share_item->host()+"/"+(TQString::compare( m_share_item->name(), "homes" ) != 0 ? m_share_item->name() : m_homes_user), frame );
  location->setReadOnly( true );

  TQLabel *port_label = new TQLabel( i18n( "Port:" ), frame );
  m_port_input = new KIntNumInput( -1, frame );
  m_port_input->setMinimumWidth( 200 );
  m_port_input->setMinValue( -1 );
  m_port_input->setMaxValue( 65535 );

  // The widgets will be put into the layout below.

  // Here comes the item-dependent stuff:
  switch ( m_type )
  {
    case Host:
    {
      TQLabel *protocol_label = new TQLabel( i18n( "Protocol:" ), frame );
      m_proto_input = new KComboBox( false, frame );
      m_proto_input->setMinimumWidth( 200 );
      m_proto_input->insertItem( i18n( "auto" ), -1 );
      m_proto_input->insertItem( "RPC", -1 );
      m_proto_input->insertItem( "RAP", -1 );
      m_proto_input->insertItem( "ADS", -1 );

      m_kerberos = new TQCheckBox( i18n( "Try to authenticate with Kerberos (Active Directory)" ), frame );

      grid->addWidget( location_label, 0, 0, 0 );
      grid->addWidget( location, 0, 1, 0 );
      grid->addWidget( port_label, 1, 0, 0 );
      grid->addWidget( m_port_input, 1, 1, 0 );
      grid->addWidget( protocol_label, 2, 0, 0 );
      grid->addWidget( m_proto_input, 2, 1, 0 );
      grid->addMultiCellWidget( m_kerberos, 3, 3, 0, 1, 0 );

      info = optionsHandler()->findItem( m_host_item->name() );

      // Get the default values from the config file:
      default_port = Smb4KSettings::remotePort();

      default_kerberos = Smb4KSettings::useKerberos();

      switch ( Smb4KSettings::protocolHint() )
      {
        case Smb4KSettings::EnumProtocolHint::Automatic:
        {
          // In this case the user leaves it to the net
          // command to determine the right protocol.
          default_protocol = "auto";

          break;
        }
        case Smb4KSettings::EnumProtocolHint::RPC:
        {
          default_protocol = "rpc";

          break;
        }
        case Smb4KSettings::EnumProtocolHint::RAP:
        {
          default_protocol = "rap";

          break;
        }
        case Smb4KSettings::EnumProtocolHint::ADS:
        {
          default_protocol = "ads";

          break;
        }
        default:
        {
          default_protocol = TQString();

          break;
        }
      }

      // Define the values that have to be put into the widgets:
      port_value =     (info && info->port() != -1) ?
                       info->port() :
                       default_port;

      protocol_value = (info && !info->protocol().isEmpty()) ?
                       info->protocol() :
                       default_protocol;

      kerberos_value = (info && info->kerberos() != default_kerberos) ?
                       info->kerberos() :
                       default_kerberos;

      // Put the values in the widgets:
      m_port_input->setValue( port_value );
      m_proto_input->setCurrentText( (TQString::compare( protocol_value, "auto" ) == 0 ? i18n( "auto" ) : protocol_value.upper()) );
      m_kerberos->setChecked( kerberos_value );

      // Does the 'Default' button need to be enabled?
      if ( default_port != port_value ||
           TQString::compare( default_protocol, protocol_value ) != 0 ||
           default_kerberos != kerberos_value )
      {
        enable_default_button = true;
      }

      // Connections:
      connect( m_port_input,  TQT_SIGNAL( valueChanged( int ) ),
               this,          TQT_SLOT( slotPortChanged( int ) ) );

      connect( m_kerberos,    TQT_SIGNAL( toggled( bool ) ),
               this,          TQT_SLOT( slotKerberosToggled( bool ) ) );

      connect( m_proto_input, TQT_SIGNAL( activated( const TQString & ) ),
               this,          TQT_SLOT( slotProtocolChanged( const TQString & ) ) );

      break;
    }
    case Share:
    {
#ifndef __FreeBSD__
      TQLabel *filesystem_label = new TQLabel( i18n( "File system:" ), frame );
      m_fs_input = new KComboBox( false, frame );
      m_fs_input->setMinimumWidth( 200 );
      m_fs_input->insertItem( "SMBFS", -1 );
      m_fs_input->insertItem( "CIFS", -1 );
      m_fs_input->setCurrentItem( 0 );

      TQLabel *permission_label = new TQLabel( i18n( "Write access:" ), frame );
      m_rw_input = new KComboBox( false, frame );
      m_rw_input->setMinimumWidth( 200 );
      m_rw_input->insertItem( i18n( "read-write" ) );
      m_rw_input->insertItem( i18n( "read-only" ) );

      TQLabel *uid_label = new TQLabel( i18n( "User ID:" ), frame );
      m_uid_input = new KLineEdit( frame );
      m_uid_input->setMinimumWidth( 200 );
      m_uid_input->setAlignment( TQt::AlignRight );
      TQLabel *gid_label = new TQLabel( i18n( "Group ID:" ), frame );
      m_gid_input = new KLineEdit( frame );
      m_gid_input->setMinimumWidth( 200 );
      m_gid_input->setAlignment( TQt::AlignRight );

      m_kerberos = new TQCheckBox( i18n( "Try to authenticate with Kerberos (Active Directory)" ), frame );

      grid->addWidget( location_label, 0, 0, 0 );
      grid->addWidget( location, 0, 1, 0 );
      grid->addWidget( port_label, 1, 0, 0 );
      grid->addWidget( m_port_input, 1, 1, 0 );
      grid->addWidget( filesystem_label, 2, 0, 0 );
      grid->addWidget( m_fs_input, 2, 1, 0 );
      grid->addWidget( uid_label, 3, 0, 0 );
      grid->addWidget( m_uid_input, 3, 1, 0 );
      grid->addWidget( gid_label, 4, 0, 0 );
      grid->addWidget( m_gid_input, 4, 1, 0 );
      grid->addWidget( permission_label, 5, 0, 0 );
      grid->addWidget( m_rw_input, 5, 1, 0 );
      grid->addMultiCellWidget( m_kerberos, 6, 6, 0, 1, 0 );
#else
      TQLabel *uid_label = new TQLabel( i18n( "User ID:" ), frame );
      m_uid_input = new KLineEdit( frame );
      m_uid_input->setMinimumWidth( 200 );
      m_uid_input->setAlignment( TQt::AlignRight );
      TQLabel *gid_label = new TQLabel( i18n( "Group ID:" ), frame );
      m_gid_input = new KLineEdit( frame );
      m_gid_input->setMinimumWidth( 200 );
      m_gid_input->setAlignment( TQt::AlignRight );

      grid->addWidget( location_label, 0, 0, 0 );
      grid->addWidget( location, 0, 1, 0 );
      grid->addWidget( port_label, 1, 0, 0 );
      grid->addWidget( m_port_input, 1, 1, 0 );
      grid->addWidget( uid_label, 2, 0, 0 );
      grid->addWidget( m_uid_input, 2, 1, 0 );
      grid->addWidget( gid_label, 3, 0, 0 );
      grid->addWidget( m_gid_input, 3, 1, 0 );
#endif

      info = optionsHandler()->findItem( TQString( "//%1/%2" ).arg( m_share_item->host(), m_share_item->name() ) );

      // Get the default values from the config file:
      default_port = Smb4KSettings::remotePort();
      default_uid = Smb4KSettings::userID();
      default_gid = Smb4KSettings::groupID();
#ifndef __FreeBSD__
      default_kerberos = Smb4KSettings::useKerberos();

      switch ( Smb4KSettings::writeAccess() )
      {
        case Smb4KSettings::EnumWriteAccess::ReadWrite:
        {
          default_readwrite = true;

          break;
        }
        case Smb4KSettings::EnumWriteAccess::ReadOnly:
        {
          default_readwrite = false;

          break;
        }
        default:
        {
          break;
        }
      }


      switch( Smb4KSettings::filesystem() )
      {
        case Smb4KSettings::EnumFilesystem::CIFS:
        {
          default_filesystem = "cifs";

          break;
        }
        case Smb4KSettings::EnumFilesystem::SMBFS:
        {
          default_filesystem = "smbfs";

          break;
        }
        default:
        {
          // FIXME: Set default_filesystem to "cifs"?
          break;
        }
      }
#endif

      // Define the values that have to be put into the widgets:
      port_value = (info && info->port() != -1) ?
                   info->port() :
                   default_port;

      uid_value = (info && !info->uid().isEmpty()) ?
                  info->uid() :
                  default_uid;

      gid_value = (info && !info->gid().isEmpty()) ?
                  info->gid() :
                  default_gid;
#ifndef __FreeBSD__
      kerberos_value = (info && info->kerberos() != default_kerberos) ?
                       info->kerberos() :
                       default_kerberos;

      readwrite_value = (info && info->writeAccess() != default_readwrite) ?
                        info->writeAccess() :
                        default_readwrite;

      filesystem_value = (info && !info->filesystem().isEmpty()) ?
                         info->filesystem() :
                         default_filesystem;
#endif

      // Put the values in the widgets:
      m_port_input->setValue( port_value );
      m_uid_input->setText( uid_value );
      m_gid_input->setText( gid_value );
#ifndef __FreeBSD__
      m_kerberos->setChecked( kerberos_value );
      m_fs_input->setCurrentText( filesystem_value.upper() );
      m_rw_input->setCurrentText( (readwrite_value ?
                                   i18n( "read-write" ) :
                                   i18n( "read-only" )) );

      // Because we do not have Kerberos with CIFS, disable this button
      // in that case:
      m_kerberos->setEnabled( !(TQString::compare( filesystem_value, "cifs" ) == 0) );
#endif

      // Does the 'Default' button need to be enabled?
#ifndef __FreeBSD__
      if ( default_port != port_value ||
           TQString::compare( default_filesystem, filesystem_value ) != 0 ||
           default_kerberos != kerberos_value ||
           default_readwrite != readwrite_value ||
           TQString::compare( default_uid, uid_value ) != 0 ||
           TQString::compare( default_gid, gid_value ) != 0 )
      {
        enable_default_button = true;
      }
#else
      if ( default_port != port_value ||
           TQString::compare( default_uid, uid_value ) != 0 ||
           TQString::compare( default_gid, gid_value ) != 0 )
      {
        enable_default_button = true;
      }
#endif

      // Connections:
      connect( m_port_input,  TQT_SIGNAL( valueChanged( int ) ),
               this,          TQT_SLOT( slotPortChanged( int ) ) );

      connect( m_uid_input,   TQT_SIGNAL( textChanged( const TQString & ) ),
               this,          TQT_SLOT( slotUIDChanged( const TQString & ) ) );

      connect( m_gid_input,   TQT_SIGNAL( textChanged( const TQString & ) ),
               this,          TQT_SLOT( slotGIDChanged( const TQString & ) ) );

#ifndef __FreeBSD__
      connect( m_kerberos,    TQT_SIGNAL( toggled( bool ) ),
               this,          TQT_SLOT( slotKerberosToggled( bool ) ) );

      connect( m_rw_input,    TQT_SIGNAL( activated( const TQString & ) ),
               this,          TQT_SLOT( slotWriteAccessChanged( const TQString & ) ) );

      connect( m_fs_input,    TQT_SIGNAL( activated( const TQString & ) ),
               this,          TQT_SLOT( slotFilesystemChanged( const TQString & ) ) );
#endif

      break;
    }
    default:
    {
      // This should not happen...
      break;
    }
  }

  // Enable the buttons:
  enableButton( Ok, false );
  enableButton( User1, enable_default_button );

  // Connect the buttons:
  connect( this,          TQT_SIGNAL( okClicked() ),
           this,          TQT_SLOT( slotOKButtonClicked() ) );

  connect( this,          TQT_SIGNAL( user1Clicked() ),
           this,          TQT_SLOT( slotDefaultButtonClicked() ) );

  setFixedSize( (sizeHint().width() > 350 ? sizeHint().width() : 350), sizeHint().height() );
}

/////////////////////////////////////////////////////////////////////////////
// TQT_SLOT IMPLEMENTATIONS
/////////////////////////////////////////////////////////////////////////////

void Smb4KCustomOptionsDialog::slotPortChanged( int val )
{
  port_changed_ok = (port_value != val);
  port_changed_default = (default_port != val);

  switch ( m_type )
  {
    case Host:
    {
      enableButton( Ok, port_changed_ok ||
                        protocol_changed_ok ||
                        kerberos_changed_ok );

      enableButton( User1, port_changed_default ||
                           protocol_changed_default ||
                           kerberos_changed_default );

      break;
    }
    case Share:
    {
      enableButton( Ok, port_changed_ok ||
#ifndef __FreeBSD__
                        filesystem_changed_ok ||
                        kerberos_changed_ok ||
                        readwrite_changed_ok ||
#endif
                        uid_changed_ok ||
                        gid_changed_ok );

      enableButton( User1, port_changed_default ||
#ifndef __FreeBSD__
                           filesystem_changed_default ||
                           kerberos_changed_default ||
                           readwrite_changed_default ||
#endif
                           uid_changed_default ||
                           gid_changed_default );

      break;
    }
    default:
    {
      break;
    }
  }
}


void Smb4KCustomOptionsDialog::slotFilesystemChanged( const TQString &text )
{
#ifndef __FreeBSD__
  filesystem_changed_ok = (TQString::compare( filesystem_value, text.lower() ) != 0);
  filesystem_changed_default = (TQString::compare( default_filesystem, text.lower() ) != 0);

  // Authentication with Kerberos does not exist with mount.cifs:
  if ( TQString::compare( text, "CIFS" ) == 0 )
  {
    m_kerberos->setEnabled( false );
  }
  else
  {
    m_kerberos->setEnabled( true );
  }

  switch ( m_type )
  {
    case Host:
    {
      enableButton( Ok, port_changed_ok ||
                        protocol_changed_ok ||
                        kerberos_changed_ok );

      enableButton( User1, port_changed_default ||
                           protocol_changed_default ||
                           kerberos_changed_default );

      break;
    }
    case Share:
    {
      enableButton( Ok, port_changed_ok ||
                        filesystem_changed_ok ||
                        kerberos_changed_ok ||
                        readwrite_changed_ok ||
                        uid_changed_ok ||
                        gid_changed_ok );

      enableButton( User1, port_changed_default ||
                           filesystem_changed_default ||
                           kerberos_changed_default ||
                           readwrite_changed_default ||
                           uid_changed_default ||
                           gid_changed_default );
      break;
    }
    default:
    {
      break;
    }
  }
#endif
}


void Smb4KCustomOptionsDialog::slotProtocolChanged( const TQString &protocol )
{
  protocol_changed_ok = (TQString::compare( protocol_value, protocol.lower() ) != 0);
  protocol_changed_default = (TQString::compare( default_protocol, protocol.lower() ) != 0);

  switch ( m_type )
  {
    case Host:
    {
      enableButton( Ok, port_changed_ok ||
                        protocol_changed_ok ||
                        kerberos_changed_ok );

      enableButton( User1, port_changed_default ||
                           protocol_changed_default ||
                           kerberos_changed_default );

      break;
    }
    case Share:
    {
      enableButton( Ok, port_changed_ok ||
#ifndef __FreeBSD__
                        filesystem_changed_ok ||
                        kerberos_changed_ok ||
                        readwrite_changed_ok ||
#endif
                        uid_changed_ok ||
                        gid_changed_ok );

      enableButton( User1, port_changed_default ||
#ifndef __FreeBSD__
                           filesystem_changed_default ||
                           kerberos_changed_default ||
                           readwrite_changed_default ||
#endif
                           uid_changed_default ||
                           gid_changed_default );

      break;
    }
    default:
    {
      break;
    }
  }
}


void Smb4KCustomOptionsDialog::slotKerberosToggled( bool on )
{
  kerberos_changed_ok = (kerberos_value != on);
  kerberos_changed_default = (default_kerberos != on);

  switch ( m_type )
  {
    case Host:
    {
      enableButton( Ok, port_changed_ok ||
                        protocol_changed_ok ||
                        kerberos_changed_ok );

      enableButton( User1, port_changed_default ||
                           protocol_changed_default ||
                           kerberos_changed_default );

      break;
    }
    case Share:
    {
      enableButton( Ok, port_changed_ok ||
#ifndef __FreeBSD__
                        filesystem_changed_ok ||
                        kerberos_changed_ok ||
                        readwrite_changed_ok ||
#endif
                        uid_changed_ok ||
                        gid_changed_ok );

      enableButton( User1, port_changed_default ||
#ifndef __FreeBSD__
                           filesystem_changed_default ||
                           kerberos_changed_default ||
                           readwrite_changed_default ||
#endif
                           uid_changed_default ||
                           gid_changed_default );

      break;
    }
    default:
    {
      break;
    }
  }
}


void Smb4KCustomOptionsDialog::slotWriteAccessChanged( const TQString &rw )
{
#ifndef __FreeBSD__
  bool readwrite = (TQString::compare( rw, i18n( "read-write" ) ) == 0);
  readwrite_changed_ok = (readwrite_value != readwrite);
  readwrite_changed_default = (default_readwrite != readwrite);

  switch ( m_type )
  {
    case Host:
    {
      enableButton( Ok, port_changed_ok ||
                        protocol_changed_ok ||
                        kerberos_changed_ok );

      enableButton( User1, port_changed_default ||
                           protocol_changed_default ||
                           kerberos_changed_default );

      break;
    }
    case Share:
    {
      enableButton( Ok, port_changed_ok ||
                        filesystem_changed_ok ||
                        kerberos_changed_ok ||
                        readwrite_changed_ok ||
                        uid_changed_ok ||
                        gid_changed_ok );

      enableButton( User1, port_changed_default ||
                           filesystem_changed_default ||
                           kerberos_changed_default ||
                           readwrite_changed_default ||
                           uid_changed_default ||
                           gid_changed_default );
      break;
    }
    default:
    {
      break;
    }
  }
#endif
}


void Smb4KCustomOptionsDialog::slotUIDChanged( const TQString &uid )
{
  uid_changed_ok = (TQString::compare( uid_value, uid ) != 0);
  uid_changed_default = (TQString::compare( default_uid, uid ) != 0);

  switch ( m_type )
  {
    case Host:
    {
      enableButton( Ok, port_changed_ok ||
                        protocol_changed_ok ||
                        kerberos_changed_ok );

      enableButton( User1, port_changed_default ||
                           protocol_changed_default ||
                           kerberos_changed_default );

      break;
    }
    case Share:
    {
      enableButton( Ok, port_changed_ok ||
#ifndef __FreeBSD__
                        filesystem_changed_ok ||
                        kerberos_changed_ok ||
                        readwrite_changed_ok ||
#endif
                        uid_changed_ok ||
                        gid_changed_ok );

      enableButton( User1, port_changed_default ||
#ifndef __FreeBSD__
                           filesystem_changed_default ||
                           kerberos_changed_default ||
                           readwrite_changed_default ||
#endif
                           uid_changed_default ||
                           gid_changed_default );

      break;
    }
    default:
    {
      break;
    }
  }
}


void Smb4KCustomOptionsDialog::slotGIDChanged( const TQString &gid )
{
  gid_changed_ok = (TQString::compare( gid_value, gid ) != 0);
  gid_changed_default = (TQString::compare( default_gid, gid ) != 0);

  switch ( m_type )
  {
    case Host:
    {
      enableButton( Ok, port_changed_ok ||
                        protocol_changed_ok ||
                        kerberos_changed_ok );

      enableButton( User1, port_changed_default ||
                           protocol_changed_default ||
                           kerberos_changed_default );

      break;
    }
    case Share:
    {
      enableButton( Ok, port_changed_ok ||
#ifndef __FreeBSD__
                        filesystem_changed_ok ||
                        kerberos_changed_ok ||
                        readwrite_changed_ok ||
#endif
                        uid_changed_ok ||
                        gid_changed_ok );

      enableButton( User1, port_changed_default ||
#ifndef __FreeBSD__
                           filesystem_changed_default ||
                           kerberos_changed_default ||
                           readwrite_changed_default ||
#endif
                           uid_changed_default ||
                           gid_changed_default );

      break;
    }
    default:
    {
      break;
    }
  }
}


void Smb4KCustomOptionsDialog::slotOKButtonClicked()
{
  Smb4KSambaOptionsInfo *info = NULL;

  switch ( m_type )
  {
    case Host:
    {
      TQString item_name = m_host_item->name();

      // Check if we can remove the item:
      if ( !port_changed_default && !protocol_changed_default && !kerberos_changed_default )
      {
        optionsHandler()->removeItem( item_name, true );
      }
      else
      {
        // First search for the item in the custom options list
        // and create a new one only if the info could not be
        // found:
        if ( !(info = optionsHandler()->findItem( item_name, true )) )
        {
          info = new Smb4KSambaOptionsInfo( item_name );
        }

        // Put in the needed information:
        info->setPort( m_port_input->value() );

        info->setProtocol( TQString::compare( m_proto_input->currentText(), i18n( "auto" ) ) == 0 ?
                           "auto" :
                           m_proto_input->currentText().lower() );

        info->setKerberos( m_kerberos->isChecked() );

        // Add the new item.
        optionsHandler()->addItem( info, true );
      }

      break;
    }
    case Share:
    {
      TQString item_name = TQString( "//%1/%2" ).arg( m_share_item->host(),
                                                    TQString::compare( m_share_item->name(), "homes" ) == 0 ?
                                                    m_homes_user :
                                                    m_share_item->name() );

#ifndef __FreeBSD__
      // Check if we can remove the item:
      if ( !port_changed_default && !filesystem_changed_default && !kerberos_changed_default &&
           !readwrite_changed_default && !uid_changed_default && !gid_changed_default )
      {
        optionsHandler()->removeItem( item_name, true );
      }
      else
      {
        // First search for the item in the custom options list
        // and create a new one only if the info could not be
        // found:
        if ( !(info = optionsHandler()->findItem( item_name, true )) )
        {
          info = new Smb4KSambaOptionsInfo( item_name );
        }

        // Put in the needed information:
        info->setPort( m_port_input->value() );
        info->setKerberos( m_kerberos->isChecked() );
        info->setWriteAccess( (TQString::compare( m_rw_input->currentText(), i18n( "read-write" ) ) == 0) );
        info->setFilesystem( m_fs_input->currentText().lower() );
        info->setUID( m_uid_input->text() );
        info->setGID( m_gid_input->text() );

        // Add the new item.
        optionsHandler()->addItem( info, true );
      }
#else
      // Check if we can remove the item:
      if ( !port_changed_default && !kerberos_changed_default && !uid_changed_default &&
           !gid_changed_default )
      {
        optionsHandler()->removeItem( item_name, true );
      }
      else
      {
        // First search for the item in the custom options list
        // and create a new one only if the info could not be
        // found:
        if ( !(info = optionsHandler()->findItem( item_name, true )) )
        {
          info = new Smb4KSambaOptionsInfo( item_name );
        }

        // Put in the needed information:
        info->setPort( m_port_input->value() );
        info->setUID( m_uid_input->text() );

        info->setGID( m_gid_input->text() );

        // Add the new item.
        optionsHandler()->addItem( info, true );
      }
#endif

      break;
    }
    default:
    {
      break;
    }
  }
}


void Smb4KCustomOptionsDialog::slotDefaultButtonClicked()
{
  // Here, we only reset the dialog and enable the OK button
  // if necessary.

  switch ( m_type )
  {
    case Host:
    {
      m_port_input->setValue( default_port );
      m_kerberos->setChecked( default_kerberos );
      TQString protocol = (TQString::compare( default_protocol, "auto" ) == 0 ? i18n( "auto" ) : protocol_value.upper());
      m_proto_input->setCurrentText( protocol );

      // Enable or disable the OK button:
      enableButton( Ok, default_port != port_value ||
                        default_kerberos != kerberos_value ||
                        TQString::compare( default_protocol, protocol_value ) != 0 );

      break;
    }
    case Share:
    {
      m_port_input->setValue( default_port );
      m_uid_input->setText( default_uid );
      m_gid_input->setText( default_gid );

#ifndef __FreeBSD__
      m_kerberos->setChecked( default_kerberos );
      TQString write_access = (default_readwrite ? i18n( "read-write" ) : i18n( "read-only" ));
      m_rw_input->setCurrentText( write_access );
      m_fs_input->setCurrentText( default_filesystem.upper() );

      // Enable or disable the Kerberos check box depending on
      // the (default) file system:
      if ( TQString::compare( default_filesystem, "cifs" ) == 0 )
      {
        m_kerberos->setEnabled( false );
      }
      else
      {
        m_kerberos->setEnabled( true );
      }
#endif

      // Enable or disable the OK button:
      enableButton( Ok, default_port != port_value ||
#ifndef __FreeBSD__
                        default_kerberos != kerberos_value ||
                        TQString::compare( default_filesystem, filesystem_value ) != 0 ||
                        default_readwrite != readwrite_value ||
#endif
                        TQString::compare( default_uid, uid_value ) != 0 ||
                        TQString::compare( default_gid, gid_value ) != 0 );

      break;
    }
    default:
    {
      break;
    }
  }

  // We just put the default values into the dialog.
  // Disable the 'Default' button:
  enableButton( User1, false );
}

#include "smb4kcustomoptionsdialog.moc"
