//
//   File : class_layout.cpp
//   Creation date : Fri Now 22 2002 00:50:01 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 2002 Szymon Stefanek (pragma at kvirc dot net)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
//
#define _KVI_DEBUG_CHECK_RANGE_
#include "kvi_debug.h"

#include "kvi_locale.h"

#include "class_layout.h"

#ifdef COMPILE_USE_QT4
#define TQLAYOUT_AUTO_CONSTRAINT TQLayout::SetDefaultConstraint
#define TQLAYOUT_FIXED TQLayout::SetFixedSize
#define TQLAYOUT_FREE_RESIZE TQLayout::SetNoConstraint
#define TQLAYOUT_MINIMUM TQLayout::SetMinimumSize
#else
#define TQLAYOUT_AUTO_CONSTRAINT TQLayout::Auto
#define TQLAYOUT_FIXED TQLayout::Fixed
#define TQLAYOUT_FREE_RESIZE TQLayout::FreeResize
#define TQLAYOUT_MINIMUM TQLayout::Minimum
#endif 


/*
	@doc: layout
	@keyterms:
		layout object class, child widgets
	@title:
		layout class
	@type:
		class
	@short:
		Manages child widget geometry
	@inherits:
		[class]object[/class]
	@description:
		The layout is a geometry management tool for child widgets.
		You create a layout , give it some widgets to manage and it will layout them
		automatically.[br]
		The parent of the layout must be the widget for which child widget geometries have to be managed.
		A layout is a grid of NxM cells in which you insert child widgets with [classfnc:layout]$addWidget[/classfnc]().[br]
		Widgets that must span multiple cells can be added to the layout with [classfnc:layout]$addMultiCellWidget[/classfnc]().[br]
	@functions:
		!fn: $addWidget(<widget:object widget>,<row:uint>,<column:uint>)
		Adds a widget to this layout placing it at position <row>,<column> in the grid
		!fn: $addMultiCellWidget(<widget:object widget>,<start_row:uint>,<end_row:uint>,<start_col:uint>,<end_col:uint>)
		Adds a widget to this layout spanning multiple grid cells
		!fn: $setRowStretch(<row:uint>,<stretch:uint>)
		Sets the stretch value for a particular row of this layout. The <stretch_value>
		must be a positive integer. The rows with bigger stretch values will take more space
		in the layout.
		!fn: $setColStretch(<column:uint>,<stretch:uint>)
		Sets the stretch value for a particular column in this layout. The <stretch_value>
		must be a positive integer. The rows with bigger stretch values will take more space
		in the layout.
		!fn: $addRowSpacing(<row:uint>,<spacing:uint>)
		Sets the minimum height of the specified <row> to <spacing> which must be a positive integer
		!fn: $addColSpacing(<column:uint>,<spacing:uint>)
		Sets the minimum width of the specigfied <column> to <spacing> which must be a positive integer
		!fn: $setSpacing(<spacing:uint>)
		Sets the default spacing of the widgets in pixels
		!fn: $setMargin(<margin:uint>)
		Sets the dimension of the layout margin : the distance from the border to the outermost child widget edges.
		!fn: $setResizeMode(<resize_mode:string>)
		Sets the resize mode of the parent widget in relation to this layout.
		<mode> can be one of:[br]
		-Auto: this is the default[br]
		-Fixed: the parent widget of this layout is resized to the "sizeHint" value and it cannot be resized by the user.[br]
		-Minimum: the minimum size of the parent widget of this layout is set to minimumSize() and it cannot be smaller[br]
		-FreeResize: the parent widget of this layout is not constrained at all[br]
*/


KVSO_BEGIN_REGISTERCLASS(KviKvsObject_layout,"layout","object")
	KVSO_REGISTER_HANDLER(KviKvsObject_layout,"addWidget", functionAddWidget)
	KVSO_REGISTER_HANDLER(KviKvsObject_layout,"addMultiCellWidget", functionAddMultiCellWidget)
	KVSO_REGISTER_HANDLER(KviKvsObject_layout,"setRowStretch", functionSetRowStretch)
	KVSO_REGISTER_HANDLER(KviKvsObject_layout,"setColStretch", functionSetColStretch)
	KVSO_REGISTER_HANDLER(KviKvsObject_layout,"addRowSpacing", functionAddRowSpacing)
	KVSO_REGISTER_HANDLER(KviKvsObject_layout,"addColSpacing", functionAddColSpacing)
	KVSO_REGISTER_HANDLER(KviKvsObject_layout,"setMargin", functionSetMargin)
	KVSO_REGISTER_HANDLER(KviKvsObject_layout,"setSpacing", functionSetSpacing)
	KVSO_REGISTER_HANDLER(KviKvsObject_layout,"setResizeMode", functionSetResizeMode)
KVSO_END_REGISTERCLASS(KviKvsObject_layout)

KVSO_BEGIN_CONSTRUCTOR(KviKvsObject_layout,KviKvsObject)

KVSO_END_CONSTRUCTOR(KviKvsObject_layout)


KVSO_BEGIN_DESTRUCTOR(KviKvsObject_layout)

KVSO_END_CONSTRUCTOR(KviKvsObject_layout)

bool KviKvsObject_layout::init(KviKvsRunTimeContext * pContext,KviKvsVariantList *pParams)
{
	TQWidget * w = parentScriptWidget();

	if(!w)
	{
		pContext->warning(__tr2qs("The parent of a layout must be a widget!"));
		return false;
	}
	setObject(TQT_TQOBJECT(new TQGridLayout(w)));
	return true;
}

bool KviKvsObject_layout::functionAddWidget(KviKvsObjectFunctionCall *c)
{
	KviKvsObject * pObject;
	kvs_hobject_t hObject;
	kvs_uint_t uCol,uRow;
	KVSO_PARAMETERS_BEGIN(c)
		KVSO_PARAMETER("widget",KVS_PT_HOBJECT,0,hObject)
		KVSO_PARAMETER("row",KVS_PT_UNSIGNEDINTEGER,0,uRow)
		KVSO_PARAMETER("col",KVS_PT_UNSIGNEDINTEGER,0,uCol)
	KVSO_PARAMETERS_END(c)
	pObject=KviKvsKernel::instance()->objectController()->lookupObject(hObject);
	if(!widget())return true;
	if (!pObject)
	{
		c->warning(__tr2qs("Widget parameter is not an object"));
		return true;
	}
	if (!pObject->object())
	{
		c->warning(__tr2qs("Widget parameter is not a valid object"));
		return true;
	}
	if(!pObject->object()->isWidgetType())
	{
		c->warning(__tr2qs("Can't add a non-widget object"));
		return true;
	}
	((TQGridLayout *)object())->addWidget(((TQWidget *)(pObject->object())),uRow,uCol);
	return true;
}
bool KviKvsObject_layout::functionAddMultiCellWidget(KviKvsObjectFunctionCall *c)
{
	KviKvsObject * pObject;
	kvs_hobject_t hObject;
	kvs_uint_t uStartCol,uStartRow,uEndCol,uEndRow;
	KVSO_PARAMETERS_BEGIN(c)
		KVSO_PARAMETER("widget",KVS_PT_HOBJECT,0,hObject)
		KVSO_PARAMETER("start_row",KVS_PT_UNSIGNEDINTEGER,0,uStartRow)
		KVSO_PARAMETER("end_row",KVS_PT_UNSIGNEDINTEGER,0,uEndRow)
		KVSO_PARAMETER("start_column",KVS_PT_UNSIGNEDINTEGER,0,uStartCol)
		KVSO_PARAMETER("end_column",KVS_PT_UNSIGNEDINTEGER,0,uEndCol)
	KVSO_PARAMETERS_END(c)
	pObject=KviKvsKernel::instance()->objectController()->lookupObject(hObject);
	if(!widget())return true;
	if (!pObject)
	{
		c->warning(__tr2qs("Widget parameter is not an object"));
		return true;
	}
	if (!pObject->object())
	{
		c->warning(__tr2qs("Widget parameter is not a valid object"));
		return true;
	}
	if(!pObject->object()->isWidgetType())
	{
		c->warning(__tr2qs("Can't add a non-widget object"));
		return true;
	}
	((TQGridLayout *)object())->addMultiCellWidget(((TQWidget *)(pObject->object())),uStartRow,uEndRow,uStartCol,uEndCol);
	return true;
}
bool KviKvsObject_layout::functionSetRowStretch(KviKvsObjectFunctionCall *c)
{
	kvs_uint_t uRow,uStretch;
	KVSO_PARAMETERS_BEGIN(c)
		KVSO_PARAMETER("row",KVS_PT_UNSIGNEDINTEGER,0,uRow)
		KVSO_PARAMETER("stretch",KVS_PT_UNSIGNEDINTEGER,0,uStretch)
	KVSO_PARAMETERS_END(c)
	if(!widget())return true;
	((TQGridLayout *)object())->setRowStretch(uRow,uStretch);
	return true;
}
bool KviKvsObject_layout::functionSetColStretch(KviKvsObjectFunctionCall *c)
{
	kvs_uint_t uCol,uStretch;
	KVSO_PARAMETERS_BEGIN(c)
		KVSO_PARAMETER("column",KVS_PT_UNSIGNEDINTEGER,0,uCol)
		KVSO_PARAMETER("stretch",KVS_PT_UNSIGNEDINTEGER,0,uStretch)
	KVSO_PARAMETERS_END(c)
	if(!widget())return true;
	((TQGridLayout *)object())->setColStretch(uCol,uStretch);
	return true;
}

bool KviKvsObject_layout::functionSetMargin(KviKvsObjectFunctionCall *c)
{
	kvs_uint_t uMargin;
	KVSO_PARAMETERS_BEGIN(c)
		KVSO_PARAMETER("margin",KVS_PT_UNSIGNEDINTEGER,0,uMargin)
	KVSO_PARAMETERS_END(c)
    if (widget()) ((TQGridLayout *)object())->setMargin(uMargin);
	return true;
}

bool KviKvsObject_layout::functionSetSpacing(KviKvsObjectFunctionCall *c)
{
	kvs_uint_t uSpacing;
	KVSO_PARAMETERS_BEGIN(c)
		KVSO_PARAMETER("spacing",KVS_PT_UNSIGNEDINTEGER,0,uSpacing)
	KVSO_PARAMETERS_END(c)
    if (widget()) ((TQGridLayout *)object())->setSpacing(uSpacing);
	return true;
}


bool KviKvsObject_layout::functionAddRowSpacing(KviKvsObjectFunctionCall *c)
{
	kvs_uint_t uSpacing,uRow;
	KVSO_PARAMETERS_BEGIN(c)
		KVSO_PARAMETER("row",KVS_PT_UNSIGNEDINTEGER,0,uRow)
		KVSO_PARAMETER("spacing",KVS_PT_UNSIGNEDINTEGER,0,uSpacing)
	KVSO_PARAMETERS_END(c)
    if (widget()) ((TQGridLayout *)object())->addRowSpacing(uRow,uSpacing);
	return true;
}

bool KviKvsObject_layout::functionAddColSpacing(KviKvsObjectFunctionCall *c)
{
	kvs_uint_t uSpacing,uCol;
	KVSO_PARAMETERS_BEGIN(c)
		KVSO_PARAMETER("column",KVS_PT_UNSIGNEDINTEGER,0,uCol)
		KVSO_PARAMETER("spacing",KVS_PT_UNSIGNEDINTEGER,0,uSpacing)
	KVSO_PARAMETERS_END(c)
    if (widget()) ((TQGridLayout *)object())->addColSpacing(uCol,uSpacing);
	return true;
}

bool KviKvsObject_layout::functionSetResizeMode(KviKvsObjectFunctionCall *c)
{
	TQString szMode;
	KVSO_PARAMETERS_BEGIN(c)
		KVSO_PARAMETER("resize_mode",KVS_PT_STRING,0,szMode)
	KVSO_PARAMETERS_END(c)
	if(!widget())return true;
#ifdef COMPILE_USE_QT4
	TQLayout::SizeConstraint r = TQLAYOUT_AUTO_CONSTRAINT;
#else
	TQLayout::ResizeMode r = TQLAYOUT_AUTO_CONSTRAINT;
#endif
	if(KviTQString::equalCI(szMode,"FreeResize")) r = TQLAYOUT_FREE_RESIZE;
	else if(KviTQString::equalCI(szMode,"Minimum")) r = TQLAYOUT_MINIMUM;
	else if(KviTQString::equalCI(szMode,"Fixed"))r = TQLAYOUT_FIXED;
	else c->warning(__tr2qs("Invalid resize mode defaulting to Auto"));
	((TQGridLayout *)object())->setResizeMode(r);
	return true;

}

