// This module implements the QextScintillaCommandSet class.
//
// Copyright (c) 2006
// 	Riverbank Computing Limited <info@riverbankcomputing.co.uk>
// 
// This file is part of TQScintilla.
// 
// This copy of TQScintilla is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option) any
// later version.
// 
// TQScintilla is supplied in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
// details.
// 
// You should have received a copy of the GNU General Public License along with
// TQScintilla; see the file LICENSE.  If not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.


#include <tqsettings.h>

#include "qextscintillacommandset.h"
#include "qextscintillacommand.h"
#include "qextscintilla.h"


// The ctor.
QextScintillaCommandSet::QextScintillaCommandSet(QextScintilla *qs) : qsci(qs)
{
	struct sci_cmd {
		int msg;
		int key;
		int altkey;
		const char *desc;
	};

	// This is based on the default table in src/KeyMap.cxx.
	static struct sci_cmd cmd_table[] = {
		{
			QextScintillaBase::SCI_LINEDOWN,
			TQt::Key_Down,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Move down one line")
		},
		{
			QextScintillaBase::SCI_LINEDOWNEXTEND,
			TQt::Key_Down | TQt::SHIFT,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Extend selection down one line")
		},
		{
			QextScintillaBase::SCI_LINESCROLLDOWN,
			TQt::Key_Down | TQt::CTRL,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Scroll view down one line")
		},
		{
			QextScintillaBase::SCI_LINEDOWNRECTEXTEND,
			TQt::Key_Down | TQt::ALT | TQt::SHIFT,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Extend rectangular selection down one line")
		},
		{
			QextScintillaBase::SCI_LINEUP,
			TQt::Key_Up,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Move up one line")
		},
		{
			QextScintillaBase::SCI_LINEUPEXTEND,
			TQt::Key_Up | TQt::SHIFT,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Extend selection up one line")
		},
		{
			QextScintillaBase::SCI_LINESCROLLUP,
			TQt::Key_Up | TQt::CTRL,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Scroll view up one line")
		},
		{
			QextScintillaBase::SCI_LINEUPRECTEXTEND,
			TQt::Key_Up | TQt::ALT | TQt::SHIFT,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Extend rectangular selection up one line")
		},
		{
			QextScintillaBase::SCI_PARAUP,
			TQt::Key_BracketLeft | TQt::CTRL,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Move up one paragraph")
		},
		{
			QextScintillaBase::SCI_PARAUPEXTEND,
			TQt::Key_BracketLeft | TQt::CTRL | TQt::SHIFT,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Extend selection up one paragraph")
		},
		{
			QextScintillaBase::SCI_PARADOWN,
			TQt::Key_BracketRight | TQt::CTRL,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Move down one paragraph")
		},
		{
			QextScintillaBase::SCI_PARADOWNEXTEND,
			TQt::Key_BracketRight | TQt::CTRL | TQt::SHIFT,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Extend selection down one paragraph")
		},
		{
			QextScintillaBase::SCI_CHARLEFT,
			TQt::Key_Left,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Move left one character")
		},
		{
			QextScintillaBase::SCI_CHARLEFTEXTEND,
			TQt::Key_Left | TQt::SHIFT,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Extend selection left one character")
		},
		{
			QextScintillaBase::SCI_WORDLEFT,
			TQt::Key_Left | TQt::CTRL,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Move left one word")
		},
		{
			QextScintillaBase::SCI_WORDLEFTEXTEND,
			TQt::Key_Left | TQt::SHIFT | TQt::CTRL,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Extend selection left one word")
		},
		{
			QextScintillaBase::SCI_CHARLEFTRECTEXTEND,
			TQt::Key_Left | TQt::ALT | TQt::SHIFT,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Extend rectangular selection left one character")
		},
		{
			QextScintillaBase::SCI_CHARRIGHT,
			TQt::Key_Right,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Move right one character")
		},
		{
			QextScintillaBase::SCI_CHARRIGHTEXTEND,
			TQt::Key_Right | TQt::SHIFT,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Extend selection right one character")
		},
		{
			QextScintillaBase::SCI_WORDRIGHT,
			TQt::Key_Right | TQt::CTRL,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Move right one word")
		},
		{
			QextScintillaBase::SCI_WORDRIGHTEXTEND,
			TQt::Key_Right | TQt::CTRL | TQt::SHIFT,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Extend selection right one word")
		},
		{
			QextScintillaBase::SCI_CHARRIGHTRECTEXTEND,
			TQt::Key_Right | TQt::ALT | TQt::SHIFT,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Extend rectangular selection right one character")
		},
		{
			QextScintillaBase::SCI_WORDPARTLEFT,
			TQt::Key_Slash | TQt::CTRL,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Move left one word part")
		},
		{
			QextScintillaBase::SCI_WORDPARTLEFTEXTEND,
			TQt::Key_Slash | TQt::CTRL | TQt::SHIFT,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Extend selection left one word part")
		},
		{
			QextScintillaBase::SCI_WORDPARTRIGHT,
			TQt::Key_Backslash | TQt::CTRL,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Move right one word part")
		},
		{
			QextScintillaBase::SCI_WORDPARTRIGHTEXTEND,
			TQt::Key_Backslash | TQt::CTRL | TQt::SHIFT,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Extend selection right one word part")
		},
		{
			QextScintillaBase::SCI_VCHOME,
			TQt::Key_Home,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Move to first visible character in line")
		},
		{
			QextScintillaBase::SCI_VCHOMEEXTEND,
			TQt::Key_Home | TQt::SHIFT,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Extend selection to first visible character in line")
		},
		{
			QextScintillaBase::SCI_DOCUMENTSTART,
			TQt::Key_Home | TQt::CTRL,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Move to start of text")
		},
		{
			QextScintillaBase::SCI_DOCUMENTSTARTEXTEND,
			TQt::Key_Home | TQt::CTRL | TQt::SHIFT,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Extend selection to start of text")
		},
		{
			QextScintillaBase::SCI_HOMEDISPLAY,
			TQt::Key_Home | TQt::ALT,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Move to start of displayed line")
		},
		{
			QextScintillaBase::SCI_HOMEDISPLAYEXTEND,
			0,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Extend selection to start of line")
		},
		{
			QextScintillaBase::SCI_VCHOMERECTEXTEND,
			TQt::Key_Home | TQt::ALT | TQt::SHIFT,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Extend rectangular selection to first visible character in line")
		},
		{
			QextScintillaBase::SCI_LINEEND,
			TQt::Key_End,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Move to end of line")
		},
		{
			QextScintillaBase::SCI_LINEENDEXTEND,
			TQt::Key_End | TQt::SHIFT,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Extend selection to end of line")
		},
		{
			QextScintillaBase::SCI_DOCUMENTEND,
			TQt::Key_End | TQt::CTRL,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Move to end of text")
		},
		{
			QextScintillaBase::SCI_DOCUMENTENDEXTEND,
			TQt::Key_End | TQt::CTRL | TQt::SHIFT,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Extend selection to end of text")
		},
		{
			QextScintillaBase::SCI_LINEENDDISPLAY,
			TQt::Key_End | TQt::ALT,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Move to end of displayed line")
		},
		{
			QextScintillaBase::SCI_LINEENDDISPLAYEXTEND,
			0,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Extend selection to end of displayed line")
		},
		{
			QextScintillaBase::SCI_LINEENDRECTEXTEND,
			TQt::Key_End | TQt::ALT | TQt::SHIFT,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Extend rectangular selection to end of line")
		},
		{
			QextScintillaBase::SCI_PAGEUP,
			TQt::Key_Prior,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Move up one page")
		},
		{
			QextScintillaBase::SCI_PAGEUPEXTEND,
			TQt::Key_Prior | TQt::SHIFT,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Extend selection up one page")
		},
		{
			QextScintillaBase::SCI_PAGEUPRECTEXTEND,
			TQt::Key_Prior | TQt::ALT | TQt::SHIFT,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Extend rectangular selection up one page")
		},
		{
			QextScintillaBase::SCI_PAGEDOWN,
			TQt::Key_Next,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Move down one page")
		},
		{
			QextScintillaBase::SCI_PAGEDOWNEXTEND,
			TQt::Key_Next | TQt::SHIFT,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Extend selection down one page")
		},
		{
			QextScintillaBase::SCI_PAGEDOWNRECTEXTEND,
			TQt::Key_Next | TQt::ALT | TQt::SHIFT,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Extend rectangular selection down one page")
		},
		{
			QextScintillaBase::SCI_CLEAR,
			TQt::Key_Delete,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Delete current character")
		},
		{
			QextScintillaBase::SCI_CUT,
			TQt::Key_X | TQt::CTRL,
			TQt::Key_Delete | TQt::SHIFT,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Cut selection")
		},
		{
			QextScintillaBase::SCI_DELWORDRIGHT,
			TQt::Key_Delete | TQt::CTRL,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Delete word to right")
		},
		{
			QextScintillaBase::SCI_DELLINERIGHT,
			TQt::Key_Delete | TQt::CTRL | TQt::SHIFT,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Delete line to right")
		},
		{
			QextScintillaBase::SCI_EDITTOGGLEOVERTYPE,
			TQt::Key_Insert,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Toggle insert/overtype")
		},
		{
			QextScintillaBase::SCI_PASTE,
			TQt::Key_V | TQt::CTRL,
			TQt::Key_Insert | TQt::SHIFT,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Paste")
		},
		{
			QextScintillaBase::SCI_COPY,
			TQt::Key_C | TQt::CTRL,
			TQt::Key_Insert | TQt::CTRL,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Copy selection")
		},
		{
			QextScintillaBase::SCI_CANCEL,
			TQt::Key_Escape,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Cancel")
		},
		{
			QextScintillaBase::SCI_DELETEBACK,
			TQt::Key_Backspace,
			TQt::Key_Backspace | TQt::SHIFT,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Delete previous character")
		},
		{
			QextScintillaBase::SCI_DELWORDLEFT,
			TQt::Key_Backspace | TQt::CTRL,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Delete word to left")
		},
		{
			QextScintillaBase::SCI_UNDO,
			TQt::Key_Z | TQt::CTRL,
			TQt::Key_Backspace | TQt::ALT,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Undo the last command")
		},
		{
			QextScintillaBase::SCI_DELLINELEFT,
			TQt::Key_Backspace | TQt::CTRL | TQt::SHIFT,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Delete line to left")
		},
		{
			QextScintillaBase::SCI_REDO,
			TQt::Key_Y | TQt::CTRL,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Redo last command")
		},
		{
			QextScintillaBase::SCI_SELECTALL,
			TQt::Key_A | TQt::CTRL,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Select all text")
		},
		{
			QextScintillaBase::SCI_TAB,
			TQt::Key_Tab,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Indent one level")
		},
		{
			QextScintillaBase::SCI_BACKTAB,
			TQt::Key_Tab | TQt::SHIFT,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Move back one indentation level")
		},
		{
			QextScintillaBase::SCI_NEWLINE,
			TQt::Key_Return,
			TQt::Key_Return | TQt::SHIFT,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Insert new line")
		},
		{
			QextScintillaBase::SCI_ZOOMIN,
			TQt::Key_Plus | TQt::CTRL,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Zoom in")
		},
		{
			QextScintillaBase::SCI_ZOOMOUT,
			TQt::Key_Minus | TQt::CTRL,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Zoom out")
		},
		{
			QextScintillaBase::SCI_SETZOOM,
			0,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Set zoom")
		},
		{
			QextScintillaBase::SCI_FORMFEED,
			0,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Formfeed")
		},
		{
			QextScintillaBase::SCI_LINECUT,
			TQt::Key_L | TQt::CTRL,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Cut current line")
		},
		{
			QextScintillaBase::SCI_LINEDELETE,
			TQt::Key_L | TQt::CTRL | TQt::SHIFT,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Delete current line")
		},
		{
			QextScintillaBase::SCI_LINECOPY,
			TQt::Key_T | TQt::CTRL | TQt::SHIFT,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Copy current line")
		},
		{
			QextScintillaBase::SCI_LINETRANSPOSE,
			TQt::Key_T | TQt::CTRL,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Swap current and previous lines")
		},
		{
			QextScintillaBase::SCI_SELECTIONDUPLICATE,
			TQt::Key_D | TQt::CTRL,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Duplicate selection")
		},
		{
			QextScintillaBase::SCI_LOWERCASE,
			TQt::Key_U | TQt::CTRL,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Convert selection to lower case")
		},
		{
			QextScintillaBase::SCI_UPPERCASE,
			TQt::Key_U | TQt::CTRL | TQt::SHIFT,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Convert selection to upper case")
		},
		{
			QextScintillaBase::SCI_DELETEBACKNOTLINE,
			0,
			0,
			TQT_TRANSLATE_NOOP("QextScintillaCommand",
				"Delete previous character if not at line start")
		},
	};

	cmds.setAutoDelete(TRUE);

	// Clear the default map.
	qsci -> SendScintilla(QextScintillaBase::SCI_CLEARALLCMDKEYS);

	for (int i = 0; i < sizeof (cmd_table) / sizeof (cmd_table[0]); ++i)
		cmds.append(new QextScintillaCommand(qsci,cmd_table[i].msg,cmd_table[i].key,cmd_table[i].altkey,cmd_table[i].desc));
}


// Read the command set from settings.
bool QextScintillaCommandSet::readSettings(TQSettings &qs,const char *prefix)
{
	bool rc = TRUE;
	TQString skey;

	for (QextScintillaCommand *cmd = cmds.first(); cmd; cmd = cmds.next())
	{
		skey.sprintf("%s/keymap/c%d/",prefix,cmd -> msgId());

		int key;
		bool ok;

		// Read the key.
		key = qs.readNumEntry(skey + "key",0,&ok);

		if (ok)
			cmd -> setKey(key);
		else
			rc = FALSE;

		// Read the alternate key.
		key = qs.readNumEntry(skey + "alt",0,&ok);

		if (ok)
			cmd -> setAlternateKey(key);
		else
			rc = FALSE;
	}

	return rc;
}


// Write the command set to settings.
bool QextScintillaCommandSet::writeSettings(TQSettings &qs,const char *prefix)
{
	bool rc = TRUE;
	TQString skey;

	for (const QextScintillaCommand *cmd = cmds.first(); cmd; cmd = cmds.next())
	{
		skey.sprintf("%s/keymap/c%d/",prefix,cmd -> msgId());

		// Write the key.
		if (!qs.writeEntry(skey + "key",cmd -> key()))
			rc = FALSE;

		// Write the alternate key.
		if (!qs.writeEntry(skey + "alt",cmd -> alternateKey()))
			rc = FALSE;
	}

	return rc;
}


// Clear the key bindings.
void QextScintillaCommandSet::clearKeys()
{
	TQPtrList<QextScintillaCommand> &set = commands();

	for (QextScintillaCommand *cmd = set.first(); cmd; cmd = set.next())
		cmd -> setKey(0);
}


// Clear the alternate key bindings.
void QextScintillaCommandSet::clearAlternateKeys()
{
	TQPtrList<QextScintillaCommand> &set = commands();

	for (QextScintillaCommand *cmd = set.first(); cmd; cmd = set.next())
		cmd -> setAlternateKey(0);
}
