/* This file is part of the KDE project
   Copyright (C) 2005 Dag Andersen <danders@get2net.dk>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation;
   version 2 of the License.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
*/

#include "kptwbsdefinition.h"


#include <tdelocale.h>
#include <kdebug.h>

#include <tqstring.h>
#include <tqstringlist.h>
#include <tqpair.h>

namespace KPlato
{


WBSDefinition::WBSDefinition() {
    m_levelsEnabled = false;
    
    m_defaultDef.code = "Number";
    m_defaultDef.separator = ".";
    
    m_codeLists.append(tqMakePair(TQString("Number"), i18n("Number")));
    m_codeLists.append(tqMakePair(TQString("Roman, upper case"), i18n("Roman, Upper Case")));
    m_codeLists.append(tqMakePair(TQString("Roman, lower case"), i18n("Roman, Lower Case")));
    m_codeLists.append(tqMakePair(TQString("Letter, upper case"), i18n("Letter, Upper Case")));
    m_codeLists.append(tqMakePair(TQString("Letter, lower case"), i18n("Letter, Lower Case")));
}

WBSDefinition::~WBSDefinition() {
}

void WBSDefinition::clear() {
    m_defaultDef.clear();
    m_levelsDef.clear();
}
    
TQString WBSDefinition::wbs(uint index, int level) {
    if (isLevelsDefEnabled()) {
        CodeDef def = levelsDef(level);
        if (!def.isEmpty()) {
            return code(def, index) + def.separator;
        }
    }
    return code(m_defaultDef, index) + m_defaultDef.separator;
}


TQString WBSDefinition::code(uint index, int level) {
    if (isLevelsDefEnabled()) {
        CodeDef def = levelsDef(level);
        if (!def.isEmpty()) {
            return code(def, index);
        }
    }
    return code(m_defaultDef, index);
}

TQString WBSDefinition::separator(int level) {
    if (isLevelsDefEnabled()) {
        CodeDef def = levelsDef(level);
        if (!def.isEmpty()) {
            return def.separator;
        }
    }
    return m_defaultDef.separator;
}

void WBSDefinition::setLevelsDef(TQMap<int, CodeDef> def) { 
    m_levelsDef.clear();
    m_levelsDef = def; 
}

WBSDefinition::CodeDef WBSDefinition::levelsDef(int level) const { 
    return m_levelsDef.contains(level) ? m_levelsDef[level] : CodeDef(); 
}
    
void WBSDefinition::setLevelsDef(int level, CodeDef def) {
    m_levelsDef.insert(level, def);
}

void WBSDefinition::setLevelsDef(int level, TQString c, TQString s) {
    m_levelsDef.insert(level, CodeDef(c, s));
}

bool WBSDefinition::level0Enabled() {
    return m_levelsEnabled && !levelsDef(0).isEmpty();
}

const TQChar Letters[] = { '?','a','b','c','d','e','f','g','h','i','j','k','l','m','n','o','p','q','r','s','t','u','v','w','x','y','z' };

TQString WBSDefinition::code(CodeDef &def, uint index) {
    if (def.code == "Number") {
        return TQString("%1").arg(index);
    }
    if (def.code == "Roman, lower case") {
        return TQString("%1").arg(toRoman(index));
    }
    if (def.code == "Roman, upper case") {
        return TQString("%1").arg(toRoman(index, true));
    }
    if (def.code == "Letter, lower case") {
        if (index > 26) {
            index = 0;
        }
        return TQString("%1").arg(Letters[index]);
    }
    if (def.code == "Letter, upper case") {
        if (index > 26) {
            index = 0;
        }
        return TQString("%1").arg(Letters[index].upper());
    }
    return TQString();
}

// Nicked from koparagcounter.cc
const TQCString RNUnits[] = {"", "i", "ii", "iii", "iv", "v", "vi", "vii", "viii", "ix"};
const TQCString RNTens[] = {"", "x", "xx", "xxx", "xl", "l", "lx", "lxx", "lxxx", "xc"};
const TQCString RNHundreds[] = {"", "c", "cc", "ccc", "cd", "d", "dc", "dcc", "dccc", "cm"};
const TQCString RNThousands[] = {"", "m", "mm", "mmm"};

TQString WBSDefinition::toRoman( int n, bool upper )
{
    if ( n >= 0 ) {
        TQString s = TQString::fromLatin1( RNThousands[ ( n / 1000 ) ] +
                                         RNHundreds[ ( n / 100 ) % 10 ] +
                                         RNTens[ ( n / 10 ) % 10 ] +
                                         RNUnits[ ( n ) % 10 ] );
        return upper ? s.upper() : s;
        
    } else { // should never happen, but better not crash if it does
        kdWarning()<< k_funcinfo << " n=" << n << endl;
        return TQString::number( n );
    }
}

TQStringList WBSDefinition::codeList() {
    TQStringList cl;
    TQValueList<TQPair<TQString, TQString> >::Iterator it;
    for (it = m_codeLists.begin(); it != m_codeLists.end(); ++it) {
        cl.append((*it).second);
    }
    return cl;
}

int WBSDefinition::defaultCodeIndex() const {
    TQValueList<TQPair<TQString, TQString> >::const_iterator it;
    int i = -1;
    for(it = m_codeLists.begin(); it != m_codeLists.end(); ++it) {
        ++i;
        if (m_defaultDef.code == (*it).first)
            break;
    }
    return i;
}

bool WBSDefinition::setDefaultCode(uint index) {
    TQValueList<TQPair<TQString, TQString> >::const_iterator it = m_codeLists.at(index);
    if (it == m_codeLists.end()) {
        return false;
    }
    m_defaultDef.code = (*it).first;
    return true;
}

void WBSDefinition::setDefaultSeparator(TQString s) {
    m_defaultDef.separator = s;
}

} //namespace KPlato
