/***************************************************************************
                         richtexteditor.cpp - Widget providing simple rich text editing 
                             -------------------
    copyright            : (C) 2002 by Marc Britton
    email                : consume@optusnet.com.au
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
/* KDE INCLUDES */
#include <tdelocale.h>

/* QT INCLUDES */
#include <tqlayout.h>
#include <tqevent.h>
#include <tqtextedit.h>
#include <tqframe.h>
#include <tqwidget.h>
#include <tqlayout.h>
#include <tqtoolbutton.h>
#include <tqpixmap.h>
#include <tqhbuttongroup.h>
#include <tqfont.h>
#include <tqstringlist.h>
#include <tqevent.h>

/* OTHER INCLUDES */
#include <specials.h>
#include "richtexteditor.h"

/* Pixmaps */
#include "pixmaps/textbold.xpm"
#include "pixmaps/textunder.xpm"
#include "pixmaps/textitalic.xpm"
#include "pixmaps/textcenter.xpm"
#include "pixmaps/textleft.xpm"
#include "pixmaps/textright.xpm"

RichTextEditor::RichTextEditor(TQWidget *a_parent, const char *a_name)
	: TQWidget(a_parent, a_name), KommanderWidget(TQT_TQOBJECT(this))
{

	TQStringList states;
	states << "default";
	setStates(states);
	setDisplayStates(states);

	// setup toolbar
	m_toolbar = new TQFrame(this, "buttonBar");
	m_toolbar->setMinimumSize(0, 32);
	m_toolbar->setFrameShape(TQFrame::NoFrame);
	m_toolbar->setFrameShadow(TQFrame::Plain);

	//setup textedit
	m_textedit = new TQTextEdit(this, "editor");
	m_textedit->setTextFormat(RichText);

	// layout the widgets
	TQVBoxLayout *layout = new TQVBoxLayout(this);
	layout->addWidget(m_toolbar);
	layout->addWidget(m_textedit);
	layout->setSpacing(1);

	// setup buttons
	TQHBoxLayout *tbLayout = new TQHBoxLayout(m_toolbar);

	//bold italic underline left right center link
	m_formatGroup = new TQHButtonGroup(m_toolbar, "formatGroup");
	//m_formatGroup->setFlat(true);
	m_alignGroup = new TQHButtonGroup(m_toolbar, "alignGroup");
	//m_alignGroup->setFlat(true);
	m_alignGroup->setExclusive(true);
	tbLayout->insertStretch(0);
	tbLayout->addWidget(m_formatGroup);
	tbLayout->addWidget(m_alignGroup);
	tbLayout->insertStretch(3);

	m_buttonTextBold = new TQToolButton(m_formatGroup, "textBold");
	m_buttonTextBold->setPixmap(TQPixmap((const char **)bold_xpm));
	m_buttonTextBold->setToggleButton(true);
	connect(m_buttonTextBold, TQT_SIGNAL(toggled(bool)), this, TQT_SLOT(textBold(bool)));
	m_buttonTextItalic = new TQToolButton(m_formatGroup, "textItalic");
	m_buttonTextItalic->setPixmap(TQPixmap((const char **)italic_xpm));
	m_buttonTextItalic->setToggleButton(true);
	connect(m_buttonTextItalic, TQT_SIGNAL(toggled(bool)), this, TQT_SLOT(textItalic(bool)));
	m_buttonTextUnder = new TQToolButton(m_formatGroup, "textUnder");
	m_buttonTextUnder->setPixmap(TQPixmap((const char **)under_xpm));
	m_buttonTextUnder->setToggleButton(true);
	connect(m_buttonTextUnder, TQT_SIGNAL(toggled(bool)), this, TQT_SLOT(textUnder(bool)));

	m_buttonTextLeft = new TQToolButton(m_alignGroup, "textLeft");
	m_buttonTextLeft->setPixmap(TQPixmap((const char **)left_xpm));
	m_buttonTextLeft->setToggleButton(true);
	m_buttonTextCenter = new TQToolButton(m_alignGroup, "textCenter");
	m_buttonTextCenter->setPixmap(TQPixmap((const char **)center_xpm));
	m_buttonTextCenter->setToggleButton(true);
	m_buttonTextRight = new TQToolButton(m_alignGroup, "textRight");
	m_buttonTextRight->setPixmap(TQPixmap((const char **)right_xpm));
	m_buttonTextRight->setToggleButton(true);

	connect(m_alignGroup, TQT_SIGNAL(clicked(int)), this, TQT_SLOT(textAlign(int)));
	connect(m_textedit, TQT_SIGNAL(currentFontChanged(const TQFont &)), this, TQT_SLOT(fontChanged(const TQFont &)));
	connect(m_textedit, TQT_SIGNAL(currentAlignmentChanged(int)), this, TQT_SLOT(alignmentChanged(int)));
	
	connect(m_textedit, TQT_SIGNAL(textChanged()), this, TQT_SLOT(setTextChanged()));

}

TQString RichTextEditor::currentState() const
{
	return TQString("default");
}

RichTextEditor::~RichTextEditor()
{
}

bool RichTextEditor::isKommanderWidget() const
{
	return true;
}

TQStringList RichTextEditor::associatedText() const
{
	return KommanderWidget::associatedText();
}

void RichTextEditor::setAssociatedText(const TQStringList& a_at)
{
	KommanderWidget::setAssociatedText(a_at);
}

void RichTextEditor::setPopulationText(const TQString& a_text)
{
    KommanderWidget::setPopulationText( a_text );
}

TQString RichTextEditor::populationText() const
{
    return KommanderWidget::populationText();
}

void RichTextEditor::populate()
{
    TQString txt = KommanderWidget::evalAssociatedText( populationText() );
    setWidgetText( txt );
}

void RichTextEditor::setWidgetText(const TQString &a_text)
{
  m_textedit->setText(a_text);
  emit widgetTextChanged(a_text);
}

void RichTextEditor::setTextChanged()
{
  emit widgetTextChanged(m_textedit->text());
}

void RichTextEditor::textBold(bool a_isOn)
{
  m_textedit->setBold(a_isOn);
}

void RichTextEditor::textUnder(bool a_isOn)
{
  m_textedit->setUnderline(a_isOn);
}

void RichTextEditor::textItalic(bool a_isOn)
{
  m_textedit->setItalic(a_isOn);
}

void RichTextEditor::textAlign(int a_id)
{
  TQToolButton *b = (TQToolButton *)m_alignGroup->find(a_id);
  if(b == m_buttonTextLeft)
    m_textedit->setAlignment(TQt::AlignLeft);
  else if(b == m_buttonTextCenter)
    m_textedit->setAlignment(TQt::AlignCenter);
  else if(b == m_buttonTextRight)
    m_textedit->setAlignment(TQt::AlignRight);
}

void RichTextEditor::fontChanged(const TQFont &a_font)
{
  m_buttonTextBold->setOn(a_font.bold());
  m_buttonTextItalic->setOn(a_font.italic());
  m_buttonTextUnder->setOn(a_font.underline());
}

void RichTextEditor::alignmentChanged(int a_alignment)
{
  if((a_alignment == AlignAuto) || (a_alignment & AlignLeft)) 
    m_buttonTextLeft->setOn(true);
  else if(a_alignment & AlignHCenter) 
    m_buttonTextCenter->setOn(true);
  else if(a_alignment & AlignRight) 
    m_buttonTextRight->setOn(true);
}

void RichTextEditor::showEvent( TQShowEvent *e )
{
    TQWidget::showEvent(e);
    emit widgetOpened();
}

void RichTextEditor::contextMenuEvent( TQContextMenuEvent * e )
{
  e->accept();
  TQPoint p = e->globalPos();
  emit contextMenuRequested(p.x(), p.y());
}

bool RichTextEditor::isFunctionSupported(int f)
{
  return f == DCOP::text || f == DCOP::setText || f == DCOP::clear || f == DCOP::selection || f == DCOP::setEditable;
}

TQString RichTextEditor::handleDCOP(int function, const TQStringList& args)
{
  switch (function) {
    case DCOP::text:
      return m_textedit->text();
    case DCOP::setText:
      setWidgetText(args[0]);
      break;
    case DCOP::selection:
      return m_textedit->selectedText();
    case DCOP::clear:
      setWidgetText("");
      break;
    case DCOP::setEditable:
      m_textedit->setReadOnly(args[0] == "false" || args[0] == "0");
      break;
    default:
      return KommanderWidget::handleDCOP(function, args);
  }      
  return TQString();
}

#include "richtexteditor.moc"
