/***************************************************************************
*   Copyright (C) 1999 by Jonas Nordin                                    *
*   jonas.nordin@syncom.se                                                *
*   Copyright (C) 2000-2001 by Bernd Gehrmann                             *
*   bernd@kdevelop.org                                                    *
*   Copyright (C) 2002-2003 by Roberto Raggi                              *
*   roberto@kdevelop.org                                                  *
*   Copyright (C) 2003-2004 by Alexander Dymo                             *
*   adymo@mksat.net                                                       *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
***************************************************************************/

#ifndef _CPPSUPPORTPART_H_
#define _CPPSUPPORTPART_H_

#include <kdevcore.h>
#include <kdevlanguagesupport.h>

#include <tqthread.h>
#include <tqmutex.h>
#include <tqtimer.h>
#include <kdialogbase.h>
#include <tqguardedptr.h>
#include <tqstring.h>
#include <tqwaitcondition.h>
#include <tqdatetime.h>
#include <tqdir.h>
#include <tqprogressbar.h>
#include <kdebug.h>
#include <ext/hash_map>
#include "driver.h"


///A class that helps detecting what exactly makes the UI block. To use it, just place a breakpoint on UIBlockTester::lockup() and inspect the execution-position of the main thread
class UIBlockTester : public TQObject {
	Q_OBJECT
  
    class UIBlockTesterThread : public TQThread {
    public:
      UIBlockTesterThread( UIBlockTester& parent );
      void run();
      void stop();
    private:
      UIBlockTester& m_parent;
      bool m_stop;
    };
  friend class UIBlockTesterThread;
public:
    
  ///@param milliseconds when the ui locks for .. milliseconds, lockup() is called
  UIBlockTester( uint milliseconds );
  ~UIBlockTester();
	
private slots:
  void timer();
	
protected:
  virtual void lockup();

private:
    UIBlockTesterThread m_thread;
	TQDateTime m_lastTime;
	TQMutex m_timeMutex;
	TQTimer * m_timer;
	uint m_msecs;
};

class Context;
class CppCodeCompletion;
class CppCodeCompletionConfig;
class CppSplitHeaderSourceConfig;
class CreateGetterSetterConfiguration;
class QtBuildConfig;
class ProblemReporter;
class BackgroundParser;
class Catalog;
class TQLabel;
class TQProgressBar;
class TQStringList;
class TQListViewItem;
class TranslationUnitAST;
class TQTimer;
class TDEListView;
class Driver;
class TDEPopupMenu;
class BackgroundParserConfig;
class TDEAction;

namespace KParts
{
	class Part;
}
namespace KTextEditor
{
	class Document;
	class View;
	class EditInterface;
	class SelectionInterface;
	class ViewCursorInterface;
}


class SynchronizedFileSet
{
public:
    typedef __gnu_cxx::hash_set< HashedString > SetType;
    SynchronizedFileSet()
	{}
	
	bool isEmpty() const
	{
		TQMutexLocker locker( &m_mutex );
		return m_fileSet.empty();
	}
	
	uint count() const
	{
		TQMutexLocker locker( &m_mutex );
		return m_fileSet.size();
	}
	
	void clear()
	{
		TQMutexLocker locker( &m_mutex );
		m_fileSet.clear();
	}

    void setFiles( const SetType& files ) {
		TQMutexLocker locker( &m_mutex );
        m_fileSet = files;
    }
	
	void insert( const HashedString& str )
	{
        HashedString s( TQString::fromUtf8(str.str().utf8()) );
		TQMutexLocker locker( &m_mutex );
          
		m_fileSet.insert( s );
	}

    bool contains( const HashedString& str ) const {
		TQMutexLocker locker( &m_mutex );
        return m_fileSet.find( str ) != m_fileSet.end();
    }

private:
	mutable TQMutex m_mutex;
	SetType m_fileSet;
};

class CppSupportPart : public KDevLanguageSupport
{
	Q_OBJECT
  

public:
	CppSupportPart( TQObject *parent, const char *name, const TQStringList &args );
	virtual ~CppSupportPart();

	bool isValid() const
	{
		return m_valid;
	}

	TQString specialHeaderName( bool local = false ) const;
	void updateParserConfiguration();
	void updateBackgroundParserConfig();
	
	// @fixme - isValid is used to avoid using the problem reporter
	// when a project is first parsed. This because the problem reporter
	// is currently a great slowdown for large projects (see bug #73671)
    ProblemReporter* problemReporter() const
	{
 		return isValid() ? static_cast<ProblemReporter *>( m_problemReporter ) : 0;
	}

    /** parses the file and all files that belong to it using the background-parser */
    int parseFileAndDependencies( const TQString& fileName, bool background = true, bool parseFirst = false, bool silent = false );
    int parseFilesAndDependencies( TQStringList files, bool background = true, bool parseFirst = false, bool silent = false );
    
    BackgroundParser* backgroundParser() const
	{
		return m_backgroundParser;
	}
    CppCodeCompletion* codeCompletion() const
	{
		return m_pCompletion;
	}
    CppCodeCompletionConfig* codeCompletionConfig() const
	{
		return m_pCompletionConfig;
	}
    CppSplitHeaderSourceConfig* splitHeaderSourceConfig() const
    {
        return m_pSplitHeaderSourceConfig;
    }
    CreateGetterSetterConfiguration* createGetterSetterConfiguration() const
	{
		return m_pCreateGetterSetterConfiguration;
	}
	
	/**
		Get a pointer to the QtBuildConfig object
		@return A pointer to the QtBuildConfig object.
	*/
	inline QtBuildConfig* qtBuildConfig() const { return m_qtBuildConfig; }

    const TQPtrList<Catalog>& catalogList() const
	{
		return m_catalogList;
	}
	void addCatalog( Catalog* catalog );
	void removeCatalog( const TQString& dbName );

	bool isValidSource( const TQString& fileName ) const;

	virtual void customEvent( TQCustomEvent* ev );

	virtual TQStringList subclassWidget( const TQString& formName );
	virtual TQStringList updateWidget( const TQString& formName, const TQString& fileName );

	FunctionDefinitionDom currentFunctionDefinition();
	FunctionDefinitionDom functionDefinitionAt( int line, int column );


	KTextEditor::Document* findDocument( const KURL& url );
	static TDEConfig *config();

	virtual TQString formatTag( const Tag& tag );
	virtual TQString formatModelItem( const CodeModelItem *item, bool shortDescription = false );
	virtual void addClass();

	TQString extractInterface( const ClassDom& klass );

	bool isHeader( const TQString& fileName ) const;
	bool isSource( const TQString& fileName ) const;

    //uses the old simple algorithm to find the header
    TQString findHeaderSimple( const TQString &header );
  
	virtual KDevDesignerIntegration *designer( KInterfaceDesigner::DesignerType type );

	void setTyping( bool typing );
	
	/**
	 * Add a new method to a class.
	 * @param aClass The class to which the method should be added.
	 * @param name The name of the method.
	 * @param type The return type of the method.
	 * @param parameters A string containing the parameters
	 * (including names, default values, but no '(' , ')', e.g.: "int, const TQString& aString").
	 * @param accessType The access specifier e.g. CodeModelItem::PUBLIC.
	 * @param isConst true if method is const.
	 * @param isInline true if method should be declared inline.
	 * @param isVirtual true if method is virtual(this is ignored if isPureVirtual is true)
	 * @param isPureVirtual true if method is pure virtual (this overrides any value of isVirtual)
	 * @param implementation a optional implementation, if this is not set the method body will be empty.
	 * @author Jonas Jacobi <j.jacobi@gmx.de>
	 */
	virtual void addMethod( ClassDom aClass, const TQString& name, const TQString type, const TQString& parameters, CodeModelItem::Access accessType, bool isConst, bool isInline, bool isVirtual, bool isPureVirtual, const TQString& implementation = "" );

	void createAccessMethods( ClassDom theClass, VariableDom theVariable );

    bool isQueued( const TQString& file ) const;
    bool switchHeaderImpl( const TQString& file, int line, int col, bool scrollOnly = false );

    const Driver* driver() const;

    Driver* driver();

    ///thread-safe, returns the thread-safe set of all files that do not need to be parsed when being included, either because they are part of the project and parsed anyway, or because they are already in the code-repository
    const SynchronizedFileSet& safeFileSet() const;
    SynchronizedFileSet& safeFileSet();
signals:
	void fileParsed( const TQString& fileName );
    ///Emitted whenever a file was parsed, but the code-model could be updated(the file in the code-model did not have to be replaced)
	void codeModelUpdated( const TQString& fileName );
	///Emitted whenever a translation-unit was parsed in the main thread
	void synchronousParseReady( const TQString& file, ParsedFilePointer unit );
    
protected:
	virtual KDevLanguageSupport::Features features();
	virtual KMimeType::List mimeTypes();
	virtual TQString formatClassName( const TQString &name );
	virtual TQString unformatClassName( const TQString &name );
    virtual bool shouldSplitDocument( const KURL &url );
    virtual Qt::Orientation splitOrientation() const;
	virtual void addMethod( ClassDom klass );
	virtual void addAttribute( ClassDom klass );

private slots:
	void activePartChanged( KParts::Part *part );
	void partRemoved( KParts::Part* part );
	void projectOpened();
	void projectClosed();
	void savedFile( const KURL &fileName );
	void configWidget( KDialogBase *dlg );
	void projectConfigWidget( KDialogBase *dlg );
	void contextMenu( TQPopupMenu *popup, const Context *context );
	void addedFilesToProject( const TQStringList &fileList );
	void removedFilesFromProject( const TQStringList &fileList );
	void changedFilesInProject( const TQStringList & fileList );
	void slotProjectCompiled();
	void setupCatalog();
    void codeCompletionConfigStored();
    void splitHeaderSourceConfigStored();
    //	void recomputeCodeModel( const TQString& fileName );
	void slotNavigate();
	void slotNewClass();
	void slotSwitchHeader( bool scrollOnly = false );
	void slotCompleteText();
	void slotMakeMember();
	void slotExtractInterface();
	void slotCursorPositionChanged();
// 	void slotFunctionHint();
	void gotoLine( int line );
	void gotoDeclarationLine( int line );
    void emitFileParsed( TQStringList l );
	void slotParseFiles();
	void slotCreateSubclass();
	void slotCreateAccessMethods();
	void slotDeleteParserStore();
    void slotSaveMemory();
	void slotTextChanged();
	void slotCursorMoved();
	void slotParseCurrentFile();
	void embedProblemReporter( bool force = false );
	void removeProblemReporter();
	

	void slotNeedTextHint( int, int, TQString& );

	/**
	 * loads, parses and creates both classstores needed
	 */
	void initialParse( );

	/**
	 * only parses the current project
	 */
	bool parseProject( bool force = false );

private:

	void resetParserStoreTimer();
	/**
	 * Get a linenumber in which a new method with a specific access specifier can be inserted.
	 * If there isn't a "section" with access, such a "section" gets inserted and the resulting place is returned.
	 * @param aClass the class one wants to insert a method to.
	 * @param access the access specifier the new method should have.
	 * @return A linenumber where the new method can be inserted 
	 * or -1 if partController()->activePart() is no KTextEditorInterface.
	 * @author Jonas Jacobi <j.jacobi@gmx.de>
	 */
	int findInsertionLineMethod( ClassDom aClass, CodeModelItem::Access access );
	/**
	 * Same as above, just returns a insertion line for a variable instead of a method
	 */
	int findInsertionLineVariable( ClassDom aClass, CodeModelItem::Access access );


	/**
	 * Get a class declaration which is "around" the current cursor position.
	 * @return The class declaration which is "around" the current cursor position,
	 * in the case of nested classes this is the innermost fitting class. If there is no
	 * class declared at the current cursor position, 0 is returned.
	 * @author Jonas Jacobi <j.jacobi@gmx.de>
	 */
	ClassDom currentClass() const;
	/**
	 * Get the class attribute of curClass, which is declared at the current cursor position.
	 * @param curClass the class to search for attributes.
	 * @return the attribute declared at the current cursor position or 0, if no attribute is declared there.
	 * @author Jonas Jacobi <j.jacobi@gmx.de>
	 */
	VariableDom currentAttribute( ClassDom curClass ) const;

	/**
	 * checks if a file has to be parsed
	 */
    FileDom fileByName( const TQString& name);
	void maybeParse( const TQString& fileName, bool background = true );
	void removeWithReferences( const TQString& fileName );
	void createIgnorePCSFile();

	void MakeMemberHelper( TQString& text, int& atline, int& atcol );

    TQString sourceOrHeaderCandidate( const KURL &url = KURL() );

	FunctionDom findFunction( const FunctionDom& def );
	FunctionDom findFunctionInNamespace( const NamespaceDom& ns, const FunctionDom& def, const std::set<NamespaceImportModel>& nsImports,
	                                     const TQString& candidateFile, int scopeIndex, FunctionDom& bestMatch );
	FunctionDom findFunctionInClass( const ClassDom& cs, const FunctionDom& def, const std::set<NamespaceImportModel>& nsImports,
	                                 const TQString& candidateFile, int scopeIndex, FunctionDom& bestMatch );
	FunctionDom findFunctionDefinition( const FunctionDom& decl );

	void jumpToCodeModelItem( const ItemDom& item, bool scrollOnly );

	TQStringList modifiedFileList();
	TQString findSourceFile();
	int pcsVersion();
	void setPcsVersion( int version );

	void saveProjectSourceInfo();
	TQStringList reorder( const TQStringList& list );

	CppCodeCompletion* m_pCompletion;
	CppCodeCompletionConfig* m_pCompletionConfig;
    CppSplitHeaderSourceConfig* m_pSplitHeaderSourceConfig;

	CreateGetterSetterConfiguration* m_pCreateGetterSetterConfiguration;
	TDEAction * m_createGetterSetterAction;
	TDEAction * m_switchHeaderSourceAction;
	
	QtBuildConfig* m_qtBuildConfig;

	bool withcpp;
	TQString m_contextFileName;

	VariableDom m_curAttribute;
	ClassDom m_curClass;
	TQGuardedPtr< ProblemReporter > m_problemReporter;
	BackgroundParser* m_backgroundParser;
    UIBlockTester* m_lockupTester;

	KTextEditor::Document* m_activeDocument;
	KTextEditor::View* m_activeView;
	KTextEditor::SelectionInterface* m_activeSelection;
	KTextEditor::EditInterface* m_activeEditor;
	KTextEditor::ViewCursorInterface* m_activeViewCursor;
	TQString m_activeFileName;

	TQMap<KInterfaceDesigner::DesignerType, KDevDesignerIntegration*> m_designers;

	TQWaitCondition m_eventConsumed;
	bool m_projectClosed;
	bool m_projectClosing;

	TQMap<TQString, TQDateTime> m_timestamp;
	bool m_valid;
	bool m_isTyping;
	bool m_hadErrors; ///Whether there were already errors when the user started typing
	
	TQPtrList<Catalog> m_catalogList;
	Driver* m_driver;
	TQString m_projectDirectory;
	TQStringList m_projectFileList;

	ClassDom m_activeClass;
	FunctionDom m_activeFunction;
	VariableDom m_activeVariable;

	TQGuardedPtr<TDEPopupMenu> m_navigationMenu;

// 	TQTimer* m_functionHintTimer;
// 	TQTimer* m_deleteParserStoreTimer;
	TQTimer* m_saveMemoryTimer;
	TQTimer * m_textChangedTimer;
	TQTimer * m_cursorMovedTimer;
    TQTimer* m_buildSafeFileSetTimer;
    
    class ParseEmitWaiting {
    public:
	    enum Flags {
		    None = 0,
			HadErrors = 1,
			HadQueueProblem = 2,
            Silent = 4
	    };
    private:
	    struct Item {
		    TQStringList first;
		    TQStringList second;
		    Flags flags;
	    	Item() : flags(None) {
		    }
			Item( TQStringList f, TQStringList s, Flags fl = None ) : first( f ), second( s ), flags( fl )
		    {
		    }
	    };
	    //typedef TQPair<TQStringList, TQStringList> Item; ///The files we are waiting fore, and the files we already got
        typedef TQValueList< Item > List;
        List m_waiting;
        
        
        TQStringList harvestUntil( List::iterator targIt ) {
            List::iterator it = m_waiting.begin();
            TQStringList ret;
            while( it != targIt && it != m_waiting.end() ) {
                ret += (*it).first;
                it = m_waiting.erase( it );
            }
            return ret;
        }
        
    public:
        void addGroup( TQStringList& files, Flags flag = None ) {
            m_waiting << Item(files, TQStringList(), flag);
        }
        void addGroupFront( TQStringList& files, Flags flag = None ) {
            m_waiting.push_front( Item(files, TQStringList(), flag) );
        }
        void clear() {
            m_waiting.clear();
        }
        
        ///files that were not requested must not be processed, since they maybe do not respect the group-relationships.
        bool reject( TQString file ) {
            for( List::iterator it = m_waiting.begin(); it != m_waiting.end(); ++it) {
                if( (*it).first.find( file ) !=  (*it).first.end() ) {
                    return false;
                }
            }
            return true;
        }

        bool waiting( TQString file, Flags forbidFlags = None, int count = 1 ) const {
            int hits = 0;
            for( List::const_iterator it = m_waiting.begin(); it != m_waiting.end(); ++it) {
                if( (*it).first.find( file ) !=  (*it).first.end() ) {
                    if( ((Flags)((*it).flags & forbidFlags )) == None ) {
                        hits++;
                        if( hits >= count ) return true;
                    }
                }
            }
            return false;
        }
        
	    struct Processed {
		    TQStringList res;
		    Flags flag;
			Processed() : flag(None) {
			}
			Processed(const TQStringList& l , Flags f = None ) : res( l ), flag( f ) {
			}
		    operator TQStringList() {
			    return res;
		    }

		    bool hadQueueProblem() {
			    return flag & HadQueueProblem;
		    }

		    bool hasFlag( Flags f ) const {
			    return f & flag;
		    }
		    // surely a copy paste implementation?
/*		    
    	    bool hasFlag( Flags flag ) const {
        	    return flag & HadQueueProblem;
    	    }
*/
	    };


    private:
        ///Just return all files that have been parsed
	    Processed errorRecover( TQString currentFile ) {
		    TQStringList ret;
	    kdDebug( 9007 ) << "ParseEmitWaiting: error in the waiting-chain" << endl;
		    for( List::iterator it = m_waiting.begin(); it != m_waiting.end(); ++it) {
			    ret += (*it).second;
		    }
		    if( !currentFile.isEmpty() ) ret << currentFile;
		    m_waiting.clear();
		    return Processed( ret, HadQueueProblem );
	    }
    public:
	    
        ///returns the parsed-messages that should be emitted
		Processed processFile( TQString file, Flags flag = None ) {
            TQStringList ret;
            for( List::iterator it = m_waiting.begin(); it != m_waiting.end(); ++it) {
                if( (*it).first.find( file ) !=  (*it).first.end() ) {
                    if( (*it).second.find( file ) == (*it).second.end() ) {
	                    (*it).flags = (Flags) ((*it).flags | flag);
                        (*it).second << file;
                        if( (*it).second.count() == (*it).first.count() ) {
	                        Flags f = (*it).flags;
                            if( it != m_waiting.begin() ) {
                                kdDebug( 9007 ) << "ParseEmitWaiting: the chain has multiple groups waiting, they are flushed" << endl;
	                            f = (Flags)(f | HadQueueProblem);
                            }
	                        return Processed( harvestUntil( ++it ), f );
                        } else {
                            ///The file was registered, now wait for the next
                            return TQStringList();
                        }
                    } else {
                        ///The file has already been parsed
                    	kdDebug( 9007 ) << "ParseEmitWaiting: file has been parsed twice" << endl;
	                    return errorRecover( file );
                    }
                }
            }
            
            kdDebug( 9007 ) << "ParseEmitWaiting: file \"" << file << "\" has no group waiting for it" << endl;
            ret << file;
			return Processed( ret, HadQueueProblem );
        }
    };
    
    ParseEmitWaiting m_parseEmitWaiting;
    ParseEmitWaiting m_fileParsedEmitWaiting;

private slots:
	void parseEmit( ParseEmitWaiting::Processed files );
    void  buildSafeFileSet();
private:

    SynchronizedFileSet m_safeProjectFiles;
	BackgroundParserConfig * m_backgroundParserConfig;
	
	static TQStringList m_sourceMimeTypes;
	static TQStringList m_headerMimeTypes;

	static TQStringList m_sourceExtensions;
	static TQStringList m_headerExtensions;

	friend class KDevCppSupportIface;
	friend class CppDriver;

	// we need something to plug actions that are not in any menu 
	// into in order for their shortcuts to work
	TQWidget m_DummyActionWidget;

    void addToRepository( ParsedFilePointer );
	void emitSynchronousParseReady( const TQString& file, ParsedFilePointer unit );

	struct JobData
	{
		TQDir dir;
		TQGuardedPtr<TQProgressBar> progressBar;
		TQStringList::Iterator it;
		TQStringList files;
		int cycle;
		int backgroundCount;
		int lastBackgroundState;
		int backgroundState;
		TQStringList reparseList;
		TQMap< TQString, TQPair<uint, uint> > pcs;
		TQDataStream stream;
		TQFile file;
		TQTime lastParse;

		~JobData()
		{
			delete progressBar;
		}
	};

	JobData * _jd;
};

#endif 
// kate: indent-mode csands; tab-width 4;

