/* This file is part of the KDE project
   Copyright (C) 2006 David Nolden <david.nolden.tdevelop@art-master.de>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/

#include <tqpixmap.h>
#include <tqapplication.h>
#include <tqnamespace.h>
#include <tdeversion.h>

#include "fancylistviewitem.h"


void FancyListViewItem::init(const TQString& label1, const TQString& /*label2*/) {
    if( !label1.isEmpty() ) {
        setText(0, label1);
    }
}


void FancyListViewItem::setItem(int column, TextPaintItem item) {
    if( column >= int(m_items.size()) ) {
        m_items.append( item );
    }else{
        m_items[column] = item;
    }
}


void FancyListViewItem::setText ( int column, const TQString & text ) {
    if( column >= int(m_items.size()) ) {
        m_items.append( TextPaintItem(text) );
    }else{
        m_items[column] = TextPaintItem(text);
    }
}


TQString FancyListViewItem::text(int column) const {
    if( m_items.isEmpty() ) return "";
    return (TQString)m_items[column];
}


TQColor FancyListViewItem::backgroundColor(int col) {

    return TDEListViewItem::backgroundColor(col);
}


///this is a modified version of the original TQListViewItem::paintCell from the qt source
///multiline is not supported!

void FancyListViewItem::paintCell( TQPainter *painter, const TQColorGroup &cg,
                                                int column, int width, int align)
{
    if(column < 0 || column >= int(m_items.size()) || m_items[column].items().isEmpty()) {
        TQListViewItem::paintCell(painter, cg, column, width, align);
        return;
    }
    
    painter->save();
    TQColorGroup grp(cg);
    
    int styleNum = m_items[column].items()[0].style;
    TextPaintStyleStore::Item& style = m_styles.getStyle( styleNum );
    ///currently only the first background-color is used
    if( style.bgValid() ) {
        grp.setColor( TQColorGroup::Base, style.background );
    }else{
        if(backgroundColor(column).isValid())
            grp.setColor( TQColorGroup::Base, backgroundColor(column) ); ///use the nice kde background-color
    }

    TQListView *lv = listView();
    if ( !lv )
        return;
    TQPainter* p = painter;
    TQFontMetrics fm( p->fontMetrics() );

    TQString t = text( column );

    int marg = lv->itemMargin();
    int r = marg;
    const TQPixmap * icon = pixmap( column );

    const BackgroundMode bgmode = lv->viewport()->backgroundMode();
    const TQColorGroup::ColorRole crole = TQPalette::backgroundRoleFromMode( bgmode );
    p->fillRect( 0, 0, width, height(), grp.brush( crole ) );
    
    if ( isSelected() && (column == 0 || lv->allColumnsShowFocus()) ) {
        p->fillRect( r - marg, 0, width - r + marg, height(), cg.brush( TQColorGroup::Highlight ) );
         
        if ( isEnabled() || !lv ) 
            p->setPen( cg.highlightedText() );
        else if ( !isEnabled() && lv)
            p->setPen( lv->palette().disabled().highlightedText() );
    } 
    {
        if ( isEnabled() || !lv )
            p->setPen( cg.text() );
        else if ( !isEnabled() && lv)
            p->setPen( lv->palette().disabled().text() );


        int iconWidth = 0;

        if ( icon ) {
            iconWidth = icon->width() + lv->itemMargin();
            int xo = r;
            int yo = ( height() - icon->height() ) / 2;

            if ( align & AlignBottom )
                yo = height() - icon->height();
            else if ( align & AlignTop )
                yo = 0;

// respect horizontal alignment when there is no text for an item.
            if ( t.isEmpty() ) {
                if ( align & AlignRight )
                    xo = width - 2 * marg - iconWidth;
                else if ( align & AlignHCenter )
                    xo = ( width - iconWidth ) / 2;
            }
            p->drawPixmap( xo, yo, *icon );
        }


        if ( !t.isEmpty() ) {
            if ( !(align & AlignTop || align & AlignBottom) )
                align |= AlignVCenter;

            r += iconWidth;
    
            TextPaintItem::Chain::iterator it = m_items[column].items().begin();
            while(it != m_items[column].items().end()) 
            {
                int styleNum = (*it).style;
                TextPaintStyleStore::Item& style = m_styles.getStyle( styleNum );
                
                painter->setFont(style.font);
                p->drawText( r, 0, width-marg-r, height(), align, (*it).text );
                r += textWidth( style.font, (*it).text );
                ++it;
            }
        }
    }
    
    painter->restore();
}


int FancyListViewItem::textWidth(const TQFont& font, const TQString& text) 
{
    TQFontMetrics fm( font );
    if ( multiLinesEnabled() )
        return fm.size( AlignVCenter, text ).width();
    else
        return fm.width( text );
    
}


int FancyListViewItem::width(const TQFontMetrics &fm, const TQListView *lv, int column)
{
    int width = 0;
    if (column >= 0 && column < (int)m_items.size() && !multiLinesEnabled()) {
        TextPaintItem::Chain::iterator it = m_items[column].items().begin();
        while(it != m_items[column].items().end()) {
            int styleNum = (*it).style;
            TextPaintStyleStore::Item& style = m_styles.getStyle( styleNum );
            
            width += textWidth( style.font, (*it).text);
            ++it;
        }
        width += lv->itemMargin() * 2;// - lv->d->minLeftBearing - lv->d->minRightBearing;
        
        const TQPixmap * pm = pixmap( column );
        if ( pm )
            width += pm->width() + lv->itemMargin();
        
        width =  TQMAX( width, TQApplication::globalStrut().width() );
    }
    else
        width = TQListViewItem::width(fm, lv, column);
    return width;
}


