/*
  broadcaststatus.cpp

  This file is part of KDEPIM.

  Author: Don Sanders <sanders@kde.org>

  Copyright (C) 2000 Don Sanders <sanders@kde.org>

  License GPL
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <tqdatetime.h>

#include <klocale.h>
#include <kglobal.h>
#include <kstaticdeleter.h>

#include "broadcaststatus.h"
#include "progressmanager.h"

KPIM::BroadcastStatus* KPIM::BroadcastStatus::instance_ = 0;
static KStaticDeleter<KPIM::BroadcastStatus> broadcasStatusDeleter;

namespace KPIM {

BroadcastStatus* BroadcastStatus::instance()
{
  if (!instance_)
    broadcasStatusDeleter.setObject( instance_, new BroadcastStatus() );

  return instance_;
}

BroadcastStatus::BroadcastStatus()
  :mTransientActive( false )
{
}

BroadcastStatus::~BroadcastStatus()
{
  instance_ = 0;
}

void BroadcastStatus::seStatusMsg( const TQString& message )
{
  mStatusMsg = message;
  if( !mTransientActive )
    emit statusMsg( message );
}

void BroadcastStatus::seStatusMsgWithTimestamp( const TQString& message )
{
  KLocale* locale = KGlobal::locale();
  seStatusMsg( i18n( "%1 is a time, %2 is a status message", "[%1] %2" )
                .tqarg( locale->formatTime( TQTime::currentTime(),
                                          true /* with seconds */ ) )
                .tqarg( message ) );
}

void BroadcastStatus::seStatusMsgTransmissionCompleted( int numMessages,
                                                           int numBytes,
                                                           int numBytesRead,
                                                           int numBytesToRead,
                                                           bool mLeaveOnServer,
                                                           KPIM::ProgressItem* item )
{
  TQString statusMsg;
  if( numMessages > 0 ) {
    if( numBytes != -1 ) {
      if( ( numBytesToRead != numBytes ) && mLeaveOnServer )
        statusMsg = i18n( "Transmission complete. %n new message in %1 KB "
                          "(%2 KB remaining on the server).",
                          "Transmission complete. %n new messages in %1 KB "
                          "(%2 KB remaining on the server).",
                          numMessages )
                    .tqarg( numBytesRead / 1024 )
                    .tqarg( numBytes / 1024 );
      else
        statusMsg = i18n( "Transmission complete. %n message in %1 KB.",
                         "Transmission complete. %n messages in %1 KB.",
                          numMessages )
                    .tqarg( numBytesRead / 1024 );
    }
    else
      statusMsg = i18n( "Transmission complete. %n new message.",
                        "Transmission complete. %n new messages.",
                        numMessages );
  }
  else
    statusMsg = i18n( "Transmission complete. No new messages." );

  seStatusMsgWithTimestamp( statusMsg );
  if ( item )
    item->setqStatus( statusMsg );
}

void BroadcastStatus::seStatusMsgTransmissionCompleted( const TQString& account,
                                                           int numMessages,
                                                           int numBytes,
                                                           int numBytesRead,
                                                           int numBytesToRead,
                                                           bool mLeaveOnServer,
                                                           KPIM::ProgressItem* item )
{
  TQString statusMsg;
  if( numMessages > 0 ) {
    if( numBytes != -1 ) {
      if( ( numBytesToRead != numBytes ) && mLeaveOnServer )
        statusMsg = i18n( "Transmission for account %3 complete. "
                          "%n new message in %1 KB "
                          "(%2 KB remaining on the server).",
                          "Transmission for account %3 complete. "
                          "%n new messages in %1 KB "
                          "(%2 KB remaining on the server).",
                          numMessages )
                    .tqarg( numBytesRead / 1024 )
                    .tqarg( numBytes / 1024 )
                    .tqarg( account );
      else
        statusMsg = i18n( "Transmission for account %2 complete. "
                          "%n message in %1 KB.",
                          "Transmission for account %2 complete. "
                          "%n messages in %1 KB.",
                          numMessages )
                    .tqarg( numBytesRead / 1024 )
                    .tqarg( account );
    }
    else
      statusMsg = i18n( "Transmission for account %1 complete. "
                        "%n new message.",
                        "Transmission for account %1 complete. "
                        "%n new messages.",
                        numMessages )
                  .tqarg( account );
  }
  else
    statusMsg = i18n( "Transmission for account %1 complete. No new messages.")
                .tqarg( account );

  seStatusMsgWithTimestamp( statusMsg );
  if ( item )
    item->setqStatus( statusMsg );
}

void BroadcastStatus::setTransienStatusMsg( const TQString& msg )
{
  mTransientActive = true;
  emit statusMsg( msg );
}

void BroadcastStatus::reset()
{
  mTransientActive = false;
  // restore
  emit statusMsg( mStatusMsg );
}

}

#include "broadcaststatus.moc"
