/*
    This file is part of libkcal.

    Copyright (c) 2003,2004 Cornelius Schumacher <schumacher@kde.org>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/
#ifndef KCAL_RESOURCECACHED_H
#define KCAL_RESOURCECACHED_H

#include "resourcecalendar.h"

#include "libemailfunctions/idmapper.h"
#include "incidence.h"
#include "calendarlocal.h"

#include <kconfig.h>

#include <tqptrlist.h>
#include <tqstring.h>
#include <tqdatetime.h>
#include <tqtimer.h>

#include <kdepimmacros.h>

namespace KCal {

/**
  This class provides a calendar resource using a local CalendarLocal object to
  cache the calendar data.
*/
class KDE_EXPORT ResourceCached : public ResourceCalendar,
                       public KCal::Calendar::Observer
{
    Q_OBJECT
  TQ_OBJECT
  public:
    /**
      Reload policy.

      @see setReloadPolicy(), reloadPolicy()
    */
    enum { ReloadNever, ReloadOnStartup, ReloadInterval };
    /**
      Save policy.

      @see setSavePolicy(), savePolicy()
    */
    enum { SaveNever, SaveOnExit, SaveInterval, SaveDelayed, SaveAlways };

    ResourceCached( const KConfig * );
    virtual ~ResourceCached();

    void readConfig( const KConfig *config );
    void writeConfig( KConfig *config );

    static bool editorWindowOpen();
    static void setEditorWindowOpen(bool open);

    /**
      Set reload policy. This controls when the cache is refreshed.

      ReloadNever     never reload
      ReloadOnStartup reload when resource is started
      ReloadInterval  reload regularly after given interval
    */
    void setReloadPolicy( int policy );
    /**
      Return reload policy.

      @see setReloadPolicy()
    */
    int reloadPolicy() const;

    /**
      Set reload interval in minutes which is used when reload policy is
      ReloadInterval.
    */
    void setReloadInterval( int minutes );

    /**
      Return reload interval in minutes.
    */
    int reloadInterval() const;

    /**
      Set save policy. This controls when the cache is refreshed.

      SaveNever     never save
      SaveOnExit    save when resource is exited
      SaveInterval  save regularly after given interval
      SaveDelayed   save after small delay
      SaveAlways    save on every change
    */
    void setSavePolicy( int policy );
    /**
      Return save policy.

      @see setsavePolicy()
    */
    int savePolicy() const;

    /**
      Set save interval in minutes which is used when save policy is
      SaveInterval.
    */
    void setSaveInterval( int minutes );

    /**
      Return save interval in minutes.
    */
    int saveInterval() const;

    /**
      Return time of last load.
    */
    TQDateTime lastLoad() const;

    /**
      Return time of last save.
    */
    TQDateTime lastSave() const;

    /**
      Add event to calendar.
    */
    KDE_DEPRECATED bool addEvent( Event *event );
    bool addEvent( Event *event, const TQString &subresource );

    /**
      Deletes an event from this calendar.
    */
    bool deleteEvent(Event *);

    /**
      Retrieves an event on the basis of the unique string ID.
    */
    Event *event(const TQString &UniqueStr);
    /**
      Return filtered list of all events in calendar.
    */
    Event::List events();
    /**
      Return unfiltered list of all events in calendar.
    */
    Event::List rawEvents( EventSortField sortField = EventSortUnsorted, SortDirection sortDirection = SortDirectionAscending );
    /**
      Builds and then returns a list of all events that match for the
      date specified. useful for dayView, etc. etc.
    */
    Event::List rawEventsForDate( const TQDate &date, EventSortField sortField = EventSortUnsorted, SortDirection sortDirection = SortDirectionAscending );

    /**
      Get unfiltered events for date \a qdt.
    */
    Event::List rawEventsForDate( const TQDateTime &qdt );
    /**
      Get unfiltered events in a range of dates. If inclusive is set to true,
      only events are returned, which are completely included in the range.
    */
    Event::List rawEvents( const TQDate &start, const TQDate &end,
                               bool inclusive = false );

    /**
      Add a todo to the todolist.
    */
    KDE_DEPRECATED bool addTodo( Todo *todo );
    bool addTodo( Todo *todo, const TQString &subresource );

    /**
      Remove a todo from the todolist.
    */
    bool deleteTodo( Todo * );
    /**
      Searches todolist for an event with this unique string identifier,
      returns a pointer or null.
    */
    Todo *todo( const TQString &uid );
    /**
      Return list of all todos.
    */
    Todo::List rawTodos( TodoSortField sortField = TodoSortUnsorted, SortDirection sortDirection = SortDirectionAscending );
    /**
      Returns list of todos due on the specified date.
    */
    Todo::List rawTodosForDate( const TQDate &date );
    /**
      Add a Journal entry to calendar
    */
    KDE_DEPRECATED bool addJournal( Journal *journal );
    bool addJournal( Journal *journal, const TQString &subresource );

    /**
      Remove a Journal from the calendar
    */
    bool deleteJournal( Journal * );
    /**
      Return Journal with given unique id.
    */
    Journal *journal( const TQString &uid );
    /**
      Return list of all journals.
    */
    Journal::List rawJournals( JournalSortField sortField = JournalSortUnsorted,SortDirection sortDirection = SortDirectionAscending );
    /**
      Return list of journals for the given date.
    */
    Journal::List rawJournalsForDate( const TQDate &date );

    /**
      Return all alarms, which ocur in the given time interval.
    */
    Alarm::List alarms( const TQDateTime &from, const TQDateTime &to );

    /**
      Return all alarms, which ocur before given date.
    */
    Alarm::List alarmsTo( const TQDateTime &to );

    /**
      Set id of timezone, e.g. "Europe/Berlin"
    */
    void setTimeZoneId( const TQString &timeZoneId );

    TQString timeZoneId() const;

    /**
      Return the owner of the calendar's full name.
    */
    const Person &getOwner() const;
    /**
      Set the owner of the calendar. Should be owner's full name.
    */
    void setOwner( const Person &owner );

    void enableChangeNotification();
    void disableChangeNotification();

    void clearChange( Incidence * );
    void clearChange( const TQString &uid );

    void clearChanges();

    bool hasChanges() const;

    Incidence::List allChanges() const;

    Incidence::List addedIncidences() const;
    Incidence::List changedIncidences() const;
    Incidence::List deletedIncidences() const;

    /**
      Loads the cache, this method should be called on load.
     */
    void loadCache();

    /**
      Saves the cache back.
     */
    void saveCache();

    /**
      Clear cache.
    */
    void clearCache();

    /**
      Clear events cache.
    */
    void clearEventsCache();

    /**
      Clear todos cache.
    */
    void clearTodosCache();

    /**
      Clear journals cache.
    */
    void clearJournalsCache();

    void cleanUpEventCache( const KCal::Event::List &eventList );
    void cleanUpTodoCache( const KCal::Todo::List &todoList );

    /**
      Returns a reference to the id mapper.
     */
    KPIM::IdMapper& idMapper();

  protected:
    // From Calendar::Observer
    void calendarIncidenceAdded( KCal::Incidence *incidence );
    void calendarIncidenceChanged( KCal::Incidence *incidence );
    void calendarIncidenceDeleted( KCal::Incidence *incidence );

    CalendarLocal mCalendar;

    /**
      Virtual method from KRES::Resource, called when the last instace of the
      resource is closed
     */
    virtual void doClose();
    /**
      Opens the resource. Dummy implementation, so child classes don't have to
      reimplement this method. By default, this does not do anything, but can be reimplemented in child classes
     */
    virtual bool doOpen();
    /**
      Check if reload required according to reload policy.
    */
    bool checkForReload();
    /**
      Check if save required according to save policy.
    */
    bool checkForSave();

    void checkForAutomaticSave();

    void addInfoText( TQString & ) const;

    void setupSaveTimer();
    void setupReloadTimer();

    /**
      This method is used by loadCache() and saveCache(), reimplement
      it to change the location of the cache.
     */
    virtual TQString cacheFile() const;

    /**
      Functions for keeping the changes persistent.
     */
    virtual TQString changesCacheFile( const TQString& ) const;
    void loadChangesCache( TQMap<Incidence*, bool>&, const TQString& );
    void loadChangesCache();
    void saveChangesCache( const TQMap<Incidence*, bool>&, const TQString& );
    void saveChangesCache();

  protected slots:
    void slotReload();
    void slotSave();

    void setIdMapperIdentifier();

  private:
    int mReloadPolicy;
    int mReloadInterval;
    TQTimer mReloadTimer;
    bool mReloaded;

    int mSavePolicy;
    int mSaveInterval;
    TQTimer mSaveTimer;

    TQDateTime mLastLoad;
    TQDateTime mLastSave;

    TQMap<KCal::Incidence *,bool> mAddedIncidences;
    TQMap<KCal::Incidence *,bool> mChangedIncidences;
    TQMap<KCal::Incidence *,bool> mDeletedIncidences;

    KPIM::IdMapper mIdMapper;

    class Private;
    Private *d;
};

}

#endif
