/*
    This file is part of libkcal.

    Copyright (c) 1998 Preston Brown <pbrown@kde.org>
    Copyright (c) 2001,2002 Cornelius Schumacher <schumacher@kde.org>
    Copyright (C) 2003-2004 Reinhold Kainhofer <reinhold@kainhofer.com>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/

#include <tqapplication.h>
#include <tqclipboard.h>
#include <tqmap.h>

#include <kiconloader.h>
#include <kdebug.h>
#include <kmessagebox.h>
#include <klocale.h>

#include "vcaldrag.h"
#include "icaldrag.h"
#include "calendar.h"
#include "vcalformat.h"
#include "icalformat.h"
#include "calendarlocal.h"

#include "dndfactory.h"

using namespace KCal;

class DndFactory::Private
{
  public:
    Incidence * pasteIncidence( Incidence *inc,
                                const TQDate &newDate,
                                const TQTime *newTime = 0 )
    {
      if ( inc ) {
        inc = inc->clone();
        inc->recreate();
      }

      if ( inc && newDate.isValid() ) {
        if ( inc->type() == "Event" ) {
          Event *anEvent = static_cast<Event*>( inc );
          // Calculate length of event
          int daysOffset = anEvent->dtStart().date().daysTo(
            anEvent->dtEnd().date() );
          // new end date if event starts at the same time on the new day
          TQDateTime endDate( newDate.addDays(daysOffset), anEvent->dtEnd().time() );

          if ( newTime ) {
            // additional offset for new time of day
            int addSecsOffset( anEvent->dtStart().time().secsTo( *newTime ));
            endDate=endDate.addSecs( addSecsOffset );
            anEvent->setDtStart( TQDateTime( newDate, *newTime ) );
          } else {
            anEvent->setDtStart( TQDateTime( newDate, anEvent->dtStart().time() ) );
          }
          anEvent->setDtEnd( endDate );
        } else if ( inc->type() == "Todo" ) {
          Todo *anTodo = static_cast<Todo*>( inc );
          if ( newTime ) {
            anTodo->setDtDue( TQDateTime( newDate, *newTime ) );
          } else {
            anTodo->setDtDue( TQDateTime( newDate, anTodo->dtDue().time() ) );
          }
        } else if ( inc->type() == "Journal" ) {
          Journal *anJournal = static_cast<Journal*>( inc );
          if ( newTime ) {
            anJournal->setDtStart( TQDateTime( newDate, *newTime ) );
          } else {
            anJournal->setDtStart( TQDateTime( newDate ) );
          }
        } else {
          kdDebug(5850) << "Trying to paste unknown incidence of type " << inc->type() << endl;
        }
      }
      return inc;
    }
};

DndFactory::DndFactory( Calendar *cal ) :
  mCalendar( cal ), d( new Private )
{
}

DndFactory::~DndFactory()
{
  delete d;
}

ICalDrag *DndFactory::createDrag( Incidence *incidence, TQWidget *owner )
{
  CalendarLocal cal( mCalendar->timeZoneId() );
  Incidence *i = incidence->clone();
  cal.addIncidence( i );

  ICalDrag *icd = new ICalDrag( &cal, owner );
  if ( i->type() == "Event" )
    icd->setPixmap( BarIcon( "appointment" ) );
  else if ( i->type() == "Todo" )
    icd->setPixmap( BarIcon( "todo" ) );

  return icd;
}

Event *DndFactory::createDrop(TQDropEvent *de)
{
  kdDebug(5800) << "DndFactory::createDrop()" << endl;

  CalendarLocal cal( mCalendar->timeZoneId() );

  if ( ICalDrag::decode( de, &cal ) || VCalDrag::decode( de, &cal ) ) {
    de->accept();

    Event::List events = cal.events();
    if ( !events.isEmpty() ) {
      Event *event = new Event( *events.first() );
      return event;
    }
  }

  return 0;
}

Todo *DndFactory::createDropTodo(TQDropEvent *de)
{
  kdDebug(5800) << "VCalFormat::createDropTodo()" << endl;

  CalendarLocal cal( mCalendar->timeZoneId() );

  if ( ICalDrag::decode( de, &cal ) || VCalDrag::decode( de, &cal ) ) {
    de->accept();

    Todo::List todos = cal.todos();
    if ( !todos.isEmpty() ) {
      Todo *todo = new Todo( *todos.first() );
      return todo;
    }
  }

  return 0;
}

void DndFactory::cutIncidence( Incidence *selectedInc )
{
  Incidence::List list;
  list.append( selectedInc );
  cutIncidences( list );
}

bool DndFactory::cutIncidences( const Incidence::List &incidences )
{
  if ( copyIncidences( incidences ) ) {
    Incidence::List::ConstIterator it;
    for ( it = incidences.constBegin(); it != incidences.constEnd(); ++it ) {
      mCalendar->deleteIncidence( *it );
    }
    return true;
  } else {
    return false;
  }
}

bool DndFactory::copyIncidences( const Incidence::List &incidences )
{
  TQClipboard *cb = TQApplication::tqclipboard();
  CalendarLocal cal( mCalendar->timeZoneId() );
  Incidence::List::ConstIterator it;

  for ( it = incidences.constBegin(); it != incidences.constEnd(); ++it ) {
    if ( *it ) {
      cal.addIncidence( ( *it )->clone() );
    }
  }

  if ( cal.incidences().isEmpty() ) {
    return false;
  } else {
    cb->setData( new ICalDrag( &cal ) );
    return true;
  }
}

bool DndFactory::copyIncidence( Incidence *selectedInc )
{
  Incidence::List list;
  list.append( selectedInc );
  return copyIncidences( list );
}

Incidence::List DndFactory::pasteIncidences( const TQDate &newDate, const TQTime *newTime )
{
  CalendarLocal cal( mCalendar->timeZoneId() );
  TQClipboard *cb = TQApplication::tqclipboard();
  Incidence::List list;

  if ( !ICalDrag::decode( cb->data(), &cal ) &&
       !VCalDrag::decode( cb->data(), &cal ) ) {
    kdDebug(5800) << "Can't parse clipboard" << endl;
    return list;
  }

  // All pasted incidences get new uids, must keep track of old uids,
  // so we can update child's parents
  TQMap<TQString,Incidence*> oldUidToNewInc;

  Incidence::List::ConstIterator it;
  const Incidence::List incs = cal.incidences();
  for ( it = incs.constBegin(); it != incs.constEnd(); ++it ) {
    Incidence *inc = d->pasteIncidence( *it, newDate, newTime );
    if ( inc ) {
      list.append( inc );
      oldUidToNewInc[( *it )->uid()] = inc;
    }
  }

  // update relations
  for ( it = list.constBegin(); it != list.constEnd(); ++it ) {
    Incidence *inc = *it;
    if ( oldUidToNewInc.contains( inc->relatedToUid() ) ) {
      Incidence *parentInc = oldUidToNewInc[inc->relatedToUid()];
      inc->setRelatedToUid( parentInc->uid() );
      inc->setRelatedTo( parentInc );
    } else {
      // not related to anything in the clipboard
      inc->setRelatedToUid( TQString() );
      inc->setRelatedTo( 0 );
    }
  }

  return list;
}

Incidence *DndFactory::pasteIncidence( const TQDate &newDate, const TQTime *newTime )
{
  CalendarLocal cal( mCalendar->timeZoneId() );
  TQClipboard *cb = TQApplication::tqclipboard();

  if ( !ICalDrag::decode( cb->data(), &cal ) &&
       !VCalDrag::decode( cb->data(), &cal ) ) {
    kdDebug(5800) << "Can't parse clipboard" << endl;
    return 0;
  }

  Incidence::List incList = cal.incidences();
  Incidence *inc = incList.isEmpty() ? 0 : incList.first();

  Incidence *newInc = d->pasteIncidence( inc, newDate, newTime );
  newInc->setRelatedTo( 0 );
  return newInc;
}
