/*  This file is part of the KDE KMobile library
    Copyright (C) 2003 Helge Deller <deller@kde.org>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License version 2 as published by the Free Software Foundation.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.

*/

#include <tqiconview.h>
#include <tqstringlist.h>

#include <ktrader.h>
#include <klibloader.h>
#include <klocale.h>
#include <kabc/vcardconverter.h>
#include <kprocess.h>
#include <kdebug.h>
#include <kconfig.h>

#include "kmobileview.h"
#include "kmobileitem.h"


KMobileView::KMobileView(TQWidget *parent, KConfig *_config)
    : DCOPObject("kmobileIface"), TQIconView(parent)
{
   m_config = _config;
   setSelectionMode(TQIconView::Single);
   setResizeMode(TQIconView::Adjust);
   setAutoArrange(true);
   connect(this, TQT_SIGNAL(doubleClicked(TQIconViewItem *)),
                 TQT_SLOT(slotDoubleClicked(TQIconViewItem *)));
}

KMobileView::~KMobileView()
{
}

bool KMobileView::addNewDevice(KConfig *config, KService::Ptr service)
{
   kdDebug() << "New mobile tqdevice item:\n";
   kdDebug() << TQString("LIBRARY: '%1', NAME: '%2', ICON: '%3', COMMENT: '%4'\n")
		.tqarg(service->library()).tqarg(service->name()).tqarg(service->icon())
		.tqarg(service->comment());

   KMobileItem *it;
   it = new KMobileItem(this, config, service);
   bool available = it->driverAvailable();
   it->configSave();
   it->writeKonquMimeFile();
   return available;
}

void KMobileView::saveAll()
{
   m_config->setGroup( "Main" );
   m_config->writeEntry( "Entries", count() );
   for ( TQIconViewItem *item = firstItem(); item; item = item->nextItem() ) {
	KMobileItem *it = static_cast<KMobileItem *>(item);
	it->driverAvailable();
	it->configSave();
	it->writeKonquMimeFile();
   }
   m_config->sync();
   emit signalChangeStatusbar( i18n("Configuration saved") );
}

void KMobileView::restoreAll()
{
   m_config->setGroup( "Main" );
   int num = m_config->readNumEntry( "Entries" );
   for (int i=0; i<num; ++i) {
	KMobileItem *it;
	it = new KMobileItem(this, m_config, i);
	it->driverAvailable();
	it->writeKonquMimeFile();
   }
   emit signalChangeStatusbar( i18n("Configuration restored") );
}

KMobileItem *KMobileView::findDevice( const TQString &tqdeviceName ) const
{
   for ( TQIconViewItem *item = firstItem(); item; item = item->nextItem() ) {
	if (item->text() == tqdeviceName)
		return static_cast<KMobileItem *>(item);
   }
   return 0L;
}

bool KMobileView::startKonqueror( const TQString &devName )
{
   KProcess *proc = new KProcess;
   *proc << "kfmclient" << "openProfile" << "webbrowsing" << "mobile:/"+devName;
   return proc->start();
}

void KMobileView::slotDoubleClicked( TQIconViewItem * item )
{
   startKonqueror(item->text());
}


/**
 * DCOP - Implementation
 */

#define MUTEX_LOCK(dev)    { dev->m_mutex.lock()
#define MUTEX_UNLOCK(dev)  dev->m_mutex.unlock(); }


TQStringList KMobileView::tqdeviceNames()
{
   TQStringList names;
   for ( TQIconViewItem *item = firstItem(); item; item = item->nextItem() ) {
	names.append(item->text());
   }
   return names;
}

void KMobileView::removeDevice( TQString tqdeviceName )
{
   delete findDevice(tqdeviceName);
   emit signalChangeStatusbar( i18n("%1 removed").tqarg(tqdeviceName) );
}

void KMobileView::configDevice( TQString tqdeviceName )
{
   KMobileItem *dev = findDevice(tqdeviceName);
   if (!dev || !dev->driverAvailable())
	return;
   MUTEX_LOCK(dev->m_dev);
   dev->m_dev->configDialog(this);
   MUTEX_UNLOCK(dev->m_dev);
}


bool KMobileView::connectDevice( TQString tqdeviceName )
{
   KMobileItem *dev = findDevice(tqdeviceName);
   if (!dev || !dev->driverAvailable())
	return false;
   bool connected;
   MUTEX_LOCK(dev->m_dev);
   connected = dev->m_dev->connectDevice();
   MUTEX_UNLOCK(dev->m_dev);
   emit signalChangeStatusbar(
	connected ? i18n("Connection to %1 established").tqarg(tqdeviceName)
	          : i18n("Connection to %1 failed").tqarg(tqdeviceName) );
   return connected;
}

bool KMobileView::disconnectDevice( TQString tqdeviceName )
{
   KMobileItem *dev = findDevice(tqdeviceName);
   if (!dev || !dev->driverAvailable())
	return true;
   bool disconnected;
   MUTEX_LOCK(dev->m_dev);
   disconnected = dev->m_dev->disconnectDevice();
   MUTEX_UNLOCK(dev->m_dev);
   emit signalChangeStatusbar(
	disconnected ? i18n("%1 disconnected").tqarg(tqdeviceName)
	             : i18n("Disconnection of %1 failed").tqarg(tqdeviceName) );
   return disconnected;
}

bool KMobileView::connected( TQString tqdeviceName )
{
   KMobileItem *dev = findDevice(tqdeviceName);
   if (!dev || !dev->driverAvailable())
	return false;
   bool conn;
   MUTEX_LOCK(dev->m_dev);
   conn = dev->m_dev->connected();
   MUTEX_UNLOCK(dev->m_dev);
   return conn;
}


TQString KMobileView::tqdeviceClassName( TQString tqdeviceName )
{
   KMobileItem *dev = findDevice(tqdeviceName);
   if (!dev || !dev->driverAvailable())
	return TQString();
   TQString cn;
   MUTEX_LOCK(dev->m_dev);
   cn = dev->m_dev->tqdeviceClassName();
   MUTEX_UNLOCK(dev->m_dev);
   return cn;
}

TQString KMobileView::tqdeviceName( TQString tqdeviceName )
{
   KMobileItem *dev = findDevice(tqdeviceName);
   if (!dev || !dev->driverAvailable())
	return TQString();
   TQString dn;
   MUTEX_LOCK(dev->m_dev);
   dn = dev->m_dev->tqdeviceName();
   MUTEX_UNLOCK(dev->m_dev);
   return dn;
}

TQString KMobileView::revision( TQString tqdeviceName )
{
   KMobileItem *dev = findDevice(tqdeviceName);
   if (!dev || !dev->driverAvailable())
	return TQString();
   TQString rev;
   MUTEX_LOCK(dev->m_dev);
   rev = dev->m_dev->revision();
   MUTEX_UNLOCK(dev->m_dev);
   return rev;
}

int KMobileView::classType( TQString tqdeviceName )
{
   KMobileItem *dev = findDevice(tqdeviceName);
   if (!dev || !dev->driverAvailable())
	return KMobileDevice::Unclassified;
   int ct;
   MUTEX_LOCK(dev->m_dev);
   ct = dev->m_dev->classType();
   MUTEX_UNLOCK(dev->m_dev);
   return ct;
}

int KMobileView::capabilities( TQString tqdeviceName )
{
   KMobileItem *dev = findDevice(tqdeviceName);
   if (!dev || !dev->driverAvailable())
	return KMobileDevice::hasNothing;
   int cap;
   MUTEX_LOCK(dev->m_dev);
   cap = dev->m_dev->capabilities();
   MUTEX_UNLOCK(dev->m_dev);
   return cap;
}

TQString KMobileView::nameForCap( TQString tqdeviceName, int cap )
{
   KMobileItem *dev = findDevice(tqdeviceName);
   if (!dev || !dev->driverAvailable())
	return TQString();
   TQString nc;
   MUTEX_LOCK(dev->m_dev);
   nc = dev->m_dev->nameForCap(cap);
   MUTEX_UNLOCK(dev->m_dev);
   return nc;
}

TQString KMobileView::iconFileName( TQString tqdeviceName )
{
   KMobileItem *dev = findDevice(tqdeviceName);
   if (!dev || !dev->driverAvailable())
	return TQString();
   TQString fn;
   MUTEX_LOCK(dev->m_dev);
   fn = dev->m_dev->iconFileName();
   MUTEX_UNLOCK(dev->m_dev);
   return fn;
}

int KMobileView::numAddresses( TQString tqdeviceName )
{
   KMobileItem *dev = findDevice(tqdeviceName);
   if (!dev || !dev->driverAvailable())
	return 0;
   int num;
   MUTEX_LOCK(dev->m_dev);
   num = dev->m_dev->numAddresses();
   MUTEX_UNLOCK(dev->m_dev);
   return num;
}

TQString KMobileView::readAddress( TQString tqdeviceName, int index )
{
   KMobileItem *dev = findDevice(tqdeviceName);
   if (!dev || !dev->driverAvailable())
	return TQString();

   int err;
   KABC::Addressee adr;
   MUTEX_LOCK(dev->m_dev);
   err = dev->m_dev->readAddress(index, adr);
   MUTEX_UNLOCK(dev->m_dev);
   if (err)
	return TQString();

   KABC::VCardConverter converter;
   TQString str = converter.createVCard(adr);
   if (str.isEmpty())
        return TQString();

   emit signalChangeStatusbar( i18n("Read addressbook entry %1 from %2")
		.tqarg(index).tqarg(tqdeviceName) );

   return str;
}

bool KMobileView::storeAddress( TQString tqdeviceName, int index, TQString vcard, bool append )
{
   KMobileItem *dev = findDevice(tqdeviceName);
   if (!dev || !dev->driverAvailable())
	return false;

   KABC::VCardConverter converter;
   KABC::Addressee adr = converter.parseVCard(vcard);
   if (adr.isEmpty())
        return false;

   int err;
   MUTEX_LOCK(dev->m_dev);
   err = dev->m_dev->storeAddress(index, adr, append);
   MUTEX_UNLOCK(dev->m_dev);
   emit signalChangeStatusbar(
	err ? i18n("Storing contact %1 on %2 failed").tqarg(index).tqarg(tqdeviceName)
	    : i18n("Contact %1 stored on %2").tqarg(index).tqarg(tqdeviceName) );
   return (err == 0);
}

int KMobileView::numCalendarEntries( TQString tqdeviceName )
{
   KMobileItem *dev = findDevice(tqdeviceName);
   if (!dev || !dev->driverAvailable())
	return 0;
   int num;
   MUTEX_LOCK(dev->m_dev);
   num = dev->m_dev->numCalendarEntries();
   MUTEX_UNLOCK(dev->m_dev);
   return num;
}

int KMobileView::numNotes( TQString tqdeviceName )
{
   KMobileItem *dev = findDevice(tqdeviceName);
   if (!dev || !dev->driverAvailable())
	return 0;
   int num;
   MUTEX_LOCK(dev->m_dev);
   num = dev->m_dev->numNotes();
   MUTEX_UNLOCK(dev->m_dev);
   return num;
}

TQString KMobileView::readNote( TQString tqdeviceName, int index )
{
   KMobileItem *dev = findDevice(tqdeviceName);
   if (!dev || !dev->driverAvailable())
	return TQString();

   TQString note;
   int err;
   MUTEX_LOCK(dev->m_dev);
   err = dev->m_dev->readNote(index, note);
   MUTEX_UNLOCK(dev->m_dev);
   if (err)
	return TQString();
   emit signalChangeStatusbar( i18n("Read note %1 from %2")
		.tqarg(index).tqarg(tqdeviceName) );
   return note;
}

bool KMobileView::storeNote( TQString tqdeviceName, int index, TQString note )
{
   KMobileItem *dev = findDevice(tqdeviceName);
   if (!dev || !dev->driverAvailable())
	return false;

   int err;
   MUTEX_LOCK(dev->m_dev);
   err = dev->m_dev->storeNote(index, note);
   MUTEX_UNLOCK(dev->m_dev);
   if (err)
	return false;
   emit signalChangeStatusbar( i18n("Stored note %1 to %2")
		.tqarg(index).tqarg(tqdeviceName) );
   return true;
}



/*
 * DCOP Implementation for the tqdevices:/ kioslave
 */

/*
 * returns the information for the given tqdeviceName for usage in the
 * the tqdevices kioslave. The TQStringList returned is comparable to the
 * format of /etc/fstab file. Please refer to the tqdevices kioslave for
 * further information.
 * If tqdeviceName is empty, this functions returns information for all
 * active mobile tqdevices.
 * (function is only used by the tqdevices kioslave - don't use elsewhere !)
 */
TQStringList KMobileView::kio_tqdevices_tqdeviceInfo(TQString tqdeviceName)
{
   TQStringList mobiles = tqdeviceNames();
   if (mobiles.count() == 0)
	return mobiles;

   TQStringList mountList;
   for ( TQStringList::Iterator it = mobiles.begin(); it != mobiles.end(); ++it ) {
	TQString name = *it;

	if (tqdeviceName.isEmpty())
		mountList << name;
	else
		if (tqdeviceName!=name)
			continue;

	KMobileItem *dev = findDevice(name);
	TQString mime = dev ? dev->getKonquMimeType() : KMOBILE_MIMETYPE_DEVICE;

	mountList << name;
	mountList << " ";
	mountList << TQString("mobile:/%1").tqarg(name); // KIO::encodeFileName()
	mountList << mime;
	mountList << "true"; // mountState
	mountList << "---";
	if (!tqdeviceName.isEmpty())
		break;
   }
   return mountList;
}


#include "kmobileview.moc"
