/*
    This file is part of KMail.

    Copyright (c) 2003 Steffen Hansen <steffen@klaralvdalens-datakonsult.se>
    Copyright (c) 2003 - 2004 Bo Thorsen <bo@sonofthor.dk>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.

    In addition, as a special exception, the copyright holders give
    permission to link the code of this program with any edition of
    the TQt library by Trolltech AS, Norway (or with modified versions
    of TQt that use the same license as TQt), and distribute linked
    combinations including the two.  You must obey the GNU General
    Public License in all respects for all of the code used other than
    TQt.  If you modify this file, you may extend this exception to
    your version of the file, but you are not obligated to do so.  If
    you do not wish to do so, delete this exception statement from
    your version.
*/

/** @file This file contains the class KMailICalIfaceImpl which actually
* implements the ICal (DCOP) interface for KMail.
*/

#ifndef KMAILICALIFACEIMPL_H
#define KMAILICALIFACEIMPL_H

#include "kmailicalIface.h"
#include "kmfoldertype.h"
#include <kfoldertree.h>

#include <tqdict.h>
#include <tqguardedptr.h>
#include <tqmap.h>

class KMFolder;
class KMMessage;
class KMFolderDir;
class KMFolderTreeItem;

namespace KMail {

  // Local helper class
class ExtraFolder {
public:
  ExtraFolder( KMFolder* f );
  ~ExtraFolder();
  TQGuardedPtr<KMFolder> folder;
};

class Accumulator {
public:
  Accumulator( const TQString& t, const TQString& f, int c )
  :type( t ), folder( f ), count( c ) {}

  void add( const TQString& incidence ) {
    incidences << incidence;
    count--;
  }
  bool isFull() { return count == 0; }

  const TQString type;
  const TQString folder;
  TQStringList incidences;
  int count;
};

}

/** The implementation of the interface. */
class KMailICalIfaceImpl : public TQObject, virtual public KMailICalIface {
  Q_OBJECT
  TQ_OBJECT
public:
  KMailICalIfaceImpl();

  bool isWritableFolder( const TQString& type, const TQString& resource );

  StorageFormat storageFormat( const TQString &resource );

  /// Update a kolab storage entry.
  /// If message is not there, it is added and
  /// given the subject as Subject: header.
  /// Returns the new mail serial number,
  /// or 0 if something went wrong,
  TQ_UINT32 update( const TQString& resource,
                   TQ_UINT32 sernum,
                   const TQString& subject,
                   const TQString& plainTextBody,
                   const TQMap<TQCString, TQString>& customHeaders,
                   const TQStringList& attachmentURLs,
                   const TQStringList& attachmentMimetypes,
                   const TQStringList& attachmentNames,
                   const TQStringList& deletedAttachments );

  bool deleteIncidenceKolab( const TQString& resource,
                             TQ_UINT32 sernum );
  int incidencesKolabCount( const TQString& mimetype,
                            const TQString& resource );
  TQMap<TQ_UINT32, TQString> incidencesKolab( const TQString& mimetype,
                                           const TQString& resource,
                                           int startIndex,
                                           int nbMessages );
  int dimapAccounts();

  TQValueList<SubResource> subresourcesKolab( const TQString& contentsType );

  bool triggerSync( const TQString& contentsType );

  // "Get" an attachment. This actually saves the attachment in a file
  // and returns a URL to it
  KURL getAttachment( const TQString& resource,
                      TQ_UINT32 sernum,
                      const TQString& filename );

  TQString attachmentMimetype( const TQString &resource,
                              TQ_UINT32 sernum,
                              const TQString &filename );

  TQStringList listAttachments( const TQString &resource, TQ_UINT32 sernum );


  bool removeSubresource( const TQString& );

  bool addSubresource( const TQString& resource,
                       const TQString& parent,
                       const TQString& contentsType );

  // tell KOrganizer about messages to be deleted
  void msgRemoved( KMFolder*, KMMessage* );

  /** Initialize all folders. */
  void initFolders();

  /** Disconnect all slots and close the dirs. */
  void cleanup();

  /**
   * Returns true if resource mode is enabled and folder is one of the
   * resource folders.
   */
  bool isResourceFolder( KMFolder* folder ) const;

  /* Returns true if the folder is one of the standard resource folders, as
   * opposed to an extra folder. */
  bool isStandardResourceFolder( KMFolder* folder ) const;

  /**
   * Returns true if isResourceFolder( folder ) returns true, and
   * imap folders should be hidden.
   */
  bool hideResourceFolder( KMFolder* folder ) const;

  /**
   * Returns true if the given folder is the root of the groupware account,
   * groupware folders are hidden, and only groupware folders shown in this
   * account.
   */
  bool hideResourceAccountRoot( KMFolder* folder ) const;

  /**
   * Returns the resource folder type. Other is returned if resource
   * isn't enabled or it isn't a resource folder.
   */
  KFolderTreeItem::Type folderType( KMFolder* folder ) const;

  /**
   * Returns the name of the standard icon for a folder of given type or
   * TQString() if the type is no groupware type.
   */
  TQString folderPixmap( KFolderTreeItem::Type type ) const;

  /** Returns the localized name of a folder of given type.
   */
  TQString folderName( KFolderTreeItem::Type type, int language = -1 ) const;

  /** Get the folder that holds *type* entries */
  KMFolder* folderFromType( const TQString& type, const TQString& folder );

  /** Return the ical type of a folder */
  TQString icalFolderType( KMFolder* folder ) const;

  /** Find message matching a given UID. */
  KMMessage* findMessageByUID( const TQString& uid, KMFolder* folder );
  /** Find message matching a given serial number. */
  static KMMessage* findMessageBySerNum( TQ_UINT32 serNum, KMFolder* folder );

  /** Convenience function to delete a message. */
  void deleteMsg( KMMessage* msg );

  bool isEnabled() const { return mUseResourceIMAP; }

  /** Called when a folders contents have changed */
  void folderContentsTypeChanged( KMFolder*, KMail::FolderContentsType );

  /// @return the storage format of a given folder
  StorageFormat storageFormat( KMFolder* folder ) const;
  /// Set the storage format of a given folder. Called when seeing the kolab annotation.
  void setStorageFormat( KMFolder* folder, StorageFormat format );


  static const char* annotationForContentsType( KMail::FolderContentsType type );

  // Called after a folder was synced with the server
  void folderSynced( KMFolder* folder, const KURL& folderURL );
  // Called when deletion of a folder from the server suceeded,
  // triggers fb re-generation
  void folderDeletedOnServer( const KURL& folderURL );
  void addFolderChange( KMFolder* folder, FolderChanges changes );

  // See CachedImapJob::slotPutMessageResult
  bool isResourceQuiet() const;
  void setResourceQuiet(bool q);

  static TQMap<TQString, TQString>* getResourceMap() { return mSubResourceUINamesMap; }

public slots:
  /* (Re-)Read configuration file */
  void readConfig();
  void slotFolderRemoved( KMFolder* folder );

  void slotIncidenceAdded( KMFolder* folder, TQ_UINT32 sernum );
  void slotIncidenceDeleted( KMFolder* folder, TQ_UINT32 sernum );
  void slotRefresh( const TQString& type);

  // Called when a folder is made readonly or readwrite, or renamed,
  // or any other similar change that affects the resources
  void slotFolderPropertiesChanged( KMFolder* folder );
  void changeResourceUIName( const TQString &folderPath, const TQString &newName );

private slots:
  void slotRefreshFolder( KMFolder* );
  void slotCheckDone();
  void slotFolderLocationChanged( const TQString&, const TQString& );
  void slotFolderRenamed();
  void slotMessageRetrieved( KMMessage* );

private:
  /** Helper function for initFolders. Initializes a single folder. */
  KMFolder* initFolder( KMail::FolderContentsType contentsType );
  KMFolder* initScalixFolder( KMail::FolderContentsType contentsType );

  void connectFolder( KMFolder* folder );

  KMFolder* extraFolder( const TQString& type, const TQString& folder );

  void syncFolder( KMFolder* folder ) const;

  struct StandardFolderSearchResult
  {
    enum FoundEnum { FoundAndStandard, NotFound, FoundByType, FoundByName };
    StandardFolderSearchResult() : folder( 0 ) {}
    StandardFolderSearchResult( KMFolder* f, FoundEnum e ) : folder( f ), found( e ) {}
    StandardFolderSearchResult( const TQValueList<KMFolder*> &f, FoundEnum e ) :
        folder( f.first() ), folders( f ), found( e ) {}
    KMFolder* folder; // NotFound implies folder==0 of course.
    TQValueList<KMFolder*> folders; // in case we found multiple default folders (which should not happen)
    FoundEnum found;
  };

  StandardFolderSearchResult findStandardResourceFolder( KMFolderDir* folderParentDir, KMail::FolderContentsType contentsType );
  KMFolder* findResourceFolder( const TQString& resource );


  bool updateAttachment( KMMessage& msg,
                         const TQString& attachmentURL,
                         const TQString& attachmentName,
                         const TQString& attachmentMimetype,
                         bool lookupByName );
  bool deleteAttachment( KMMessage& msg,
                         const TQString& attachmentURL );
  TQ_UINT32 addIncidenceKolab( KMFolder& folder,
                              const TQString& subject,
                              const TQString& plainTextBody,
                              const TQMap<TQCString, TQString>& customHeaders,
                              const TQStringList& attachmentURLs,
                              const TQStringList& attachmentNames,
                              const TQStringList& attachmentMimetypes );
  static bool kolabXMLFoundAndDecoded( const KMMessage& msg, const TQString& mimetype, TQString& s );

  void handleFolderSynced( KMFolder* folder,
                           const KURL& folderURL,
                           int _changes );
  void triggerKolabFreeBusy( const KURL& folderURL );

  StorageFormat globalStorageFormat() const;

  static bool folderIsAlarmRelevant( const KMFolder * );

private:
  TQGuardedPtr<KMFolder> mContacts;
  TQGuardedPtr<KMFolder> mCalendar;
  TQGuardedPtr<KMFolder> mNotes;
  TQGuardedPtr<KMFolder> mTasks;
  TQGuardedPtr<KMFolder> mJournals;

  // The extra IMAP resource folders
  // Key: folder location. Data: folder.
  TQDict<KMail::ExtraFolder> mExtraFolders;
  // used for collecting incidences during async loading
  TQDict<KMail::Accumulator> mAccumulators;
  // More info for each folder we care about (mContacts etc. as well as the extra folders)
  // The reason for storing it here is that it can be shared between
  // kmfoldercachedimap and kmfolderimap, and that it's groupware data anyway.
  struct FolderInfo {
    FolderInfo() {} // for TQMap
    FolderInfo( StorageFormat f, FolderChanges c ) :
      mStorageFormat( f ), mChanges( c ) {}
    StorageFormat mStorageFormat;
    FolderChanges mChanges;
  };
  // The storage format used for each folder that we care about
  typedef TQMap<KMFolder*, FolderInfo> FolderInfoMap;
  // helper for reading the FolderInfo from the config file
  FolderInfo readFolderInfo( const KMFolder * const folder ) const;

  FolderInfoMap mFolderInfoMap;

  unsigned int mFolderLanguage;

  KMFolderDir* mFolderParentDir;
  KMFolder*    mFolderParent;
  KMFolderType mFolderType;

  bool mUseResourceIMAP;
  bool mResourceQuiet;
  bool mHideFolders;

  /*
   * Bunch of maps to keep track of incidents currently in transfer, ones
   * which need to be ungotten, once we are done, once with updates pending.
   * Since these are transient attributes of only a small but changing number
   * of incidences they are not encapsulated in a struct or somesuch.
   */
  TQMap<TQString, TQ_UINT32> mUIDToSerNum;
  TQMap<TQ_UINT32, bool> mTheUnGetMes;
  TQMap<TQString, TQString> mPendingUpdates;
  TQMap<TQString, bool> mInTransit;
  static TQMap<TQString, TQString> *mSubResourceUINamesMap;

};

#endif // KMAILICALIFACEIMPL_H
