/*
    This file is part of KitchenSync.

    Copyright (c) 2005 Tobias Koenig <tokoe@kde.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

#include <kapplication.h>
#include <kdialog.h>
#include <kglobal.h>
#include <kglobalsettings.h>
#include <kiconloader.h>
#include <klocale.h>
#include <kpassivepopup.h>
#include <kurllabel.h>

#include <tqlabel.h>
#include <tqlayout.h>
#include <tqpixmap.h>
#include <tqprogressbar.h>
#include <tqvbox.h>

#include "memberinfo.h"
#include "multiconflictdialog.h"
#include "singleconflictdialog.h"
#include "syncprocessmanager.h"

#include "groupitem.h"

GroupItem::GroupItem( KWidgetList *parent, SyncProcess *process )
  : KWidgetListItem( parent ), mSyncProcess( process ),
    mCallbackHandler( new QSync::CallbackHandler ),
    mProcessedItems( 0 ), mMaxProcessedItems( 0 ),
    mSynchronizing( false )
{
  TQFont boldFont;
  boldFont.setBold( true );
  boldFont.setPointSize( boldFont.pointSize() + 2 );

  TQGridLayout *tqlayout = new TQGridLayout( this, 4, 4, KDialog::marginHint(), KDialog::spacingHint() );

  mBox = new TQVBox( this );
  mBox->setMargin( 5 );
  mProgressBar = new TQProgressBar( this );
  mProgressBar->setTotalSteps( 100 );

  mTime = new TQLabel( this );
  mSyncAction = new KURLLabel( "exec:/sync", i18n( "Synchronize Now" ), this );
  mConfigureAction = new KURLLabel( "exec:/config", i18n( "Configure" ), this );

  // header
  TQHBox* hbox = new TQHBox( this );
  hbox->setMargin( 2 );

  static TQPixmap icon;
  if ( icon.isNull() )
    icon = KGlobal::iconLoader()->loadIcon( "kontact_summary", KIcon::Desktop );

  mIcon = new TQLabel( hbox );
  mIcon->setPixmap( icon );
  mIcon->setFixedSize( mIcon->tqsizeHint() );
  mIcon->setPaletteBackgroundColor( tqcolorGroup().mid() );

  mGroupName = new TQLabel( hbox );
  mGroupName->tqsetAlignment( AlignLeft | AlignVCenter );
  mGroupName->setIndent( KDialog::spacingHint() );
  mGroupName->setFont( boldFont );
  mGroupName->setPaletteForegroundColor( tqcolorGroup().light() );
  mGroupName->setPaletteBackgroundColor( tqcolorGroup().mid() );

  mtqStatus = new TQLabel( hbox );
  mtqStatus->tqsetAlignment( TQt::AlignRight );
  mtqStatus->tqsetAlignment( AlignRight | AlignVCenter );
  mtqStatus->setIndent( KDialog::spacingHint() );
  mtqStatus->setFont( boldFont );
  mtqStatus->setPaletteForegroundColor( tqcolorGroup().light() );
  mtqStatus->setPaletteBackgroundColor( tqcolorGroup().mid() );
  mtqStatus->setText( i18n( "Ready" ) );

  hbox->setPaletteBackgroundColor( tqcolorGroup().mid() );
  hbox->setMaximumHeight( hbox->tqminimumSizeHint().height() );

  tqlayout->addMultiCellWidget( hbox, 0, 0, 0, 3 );
  tqlayout->addMultiCellWidget( mBox, 1, 1, 0, 3 );
  tqlayout->addWidget( mTime, 2, 0 );
  tqlayout->addWidget( mSyncAction, 2, 1 );
  tqlayout->addWidget( mConfigureAction, 2, 2 );
  tqlayout->addWidget( mProgressBar, 2, 3 );
  tqlayout->setColStretch( 0, 1 );
  tqlayout->setRowStretch( 3, 1 );

  setPaletteBackgroundColor( kapp->tqpalette().active().base() );

  connect( mCallbackHandler, TQT_SIGNAL( conflict( QSync::SyncMapping ) ),
           this, TQT_SLOT( conflict( QSync::SyncMapping ) ) );
  connect( mCallbackHandler, TQT_SIGNAL( change( const QSync::SyncChangeUpdate& ) ),
           this, TQT_SLOT( change( const QSync::SyncChangeUpdate& ) ) );
  connect( mCallbackHandler, TQT_SIGNAL( mapping( const QSync::SyncMappingUpdate& ) ),
           this, TQT_SLOT( mapping( const QSync::SyncMappingUpdate& ) ) );
  connect( mCallbackHandler, TQT_SIGNAL( engine( const QSync::SyncEngineUpdate& ) ),
           this, TQT_SLOT( engine( const QSync::SyncEngineUpdate& ) ) );
  connect( mCallbackHandler, TQT_SIGNAL( member( const QSync::SyncMemberUpdate& ) ),
           this, TQT_SLOT( member( const QSync::SyncMemberUpdate& ) ) );
  connect( mSyncAction, TQT_SIGNAL( leftClickedURL() ),
           this, TQT_SLOT( synchronize() ) );
  connect( mConfigureAction, TQT_SIGNAL( leftClickedURL() ),
           this, TQT_SLOT( configure() ) );
  connect( mSyncProcess, TQT_SIGNAL( engineChanged( QSync::Engine* ) ),
           this, TQT_SLOT( engineChanged( QSync::Engine* ) ) );

  mCallbackHandler->setEngine( mSyncProcess->engine() );

  setSelectionForegroundColor( KGlobalSettings::textColor() );
  setSelectionBackgroundColor( KGlobalSettings::alternateBackgroundColor() );

  update();
}

GroupItem::~GroupItem()
{
  delete mCallbackHandler;
  mCallbackHandler = 0;
}

void GroupItem::update()
{
  clear();

  mGroupName->setText( i18n( "Group: %1" ).tqarg( mSyncProcess->group().name() ) );

  TQDateTime dateTime = mSyncProcess->group().lastSynchronization();
  if ( dateTime.isValid() )
    mTime->setText( i18n( "Last synchronized on: %1" ).tqarg( KGlobal::locale()->formatDateTime( dateTime ) ) );
  else
    mTime->setText( i18n( "Not synchronized yet" ) );

  mProgressBar->reset();
  mProgressBar->hide();

  QSync::Group group = mSyncProcess->group();
  QSync::Group::Iterator memberIt( group.begin() );
  QSync::Group::Iterator memberEndIt( group.end() );

  for ( ; memberIt != memberEndIt; ++memberIt ) {
    MemberItem *item = new MemberItem( mBox, mSyncProcess, *memberIt );
    item->show();
    item->seStatusMessage( i18n( "Ready" ) );
    mMemberItems.append( item );
  }
}

void GroupItem::clear()
{
  mGroupName->setText( TQString() );

  TQValueList<MemberItem*>::Iterator it;
  for ( it = mMemberItems.begin(); it != mMemberItems.end(); ++it )
    delete *it;

  mMemberItems.clear();
}

void GroupItem::conflict( QSync::SyncMapping mapping )
{
  if ( mapping.changesCount() == 2 ) {
    SingleConflictDialog dlg( mapping, this );
    dlg.exec();
  } else {
    MultiConflictDialog dlg( mapping, this );
    dlg.exec();
  }
}

void GroupItem::change( const QSync::SyncChangeUpdate &update )
{
  switch ( update.type() ) {
    case QSync::SyncChangeUpdate::Received:
      mProcessedItems++;
      mtqStatus->setText( i18n( "%1 entries read" ).tqarg( mProcessedItems ) );
      break;
    case QSync::SyncChangeUpdate::ReceivedInfo:
      mtqStatus->setText( i18n( "Receive information" ) );
      break;
    case QSync::SyncChangeUpdate::Sent:
      mProcessedItems--;
      mtqStatus->setText( i18n( "%1 entries written" ).tqarg( mMaxProcessedItems - mProcessedItems ) );

      mProgressBar->show();

      {
        int progress = 100;
        if ( mMaxProcessedItems != 0 )
          progress = (mProcessedItems * 100) / mMaxProcessedItems;

        if ( progress < 0 )
          progress = 0;

        mProgressBar->setProgress( 100 - progress );
      }
      break;
    case QSync::SyncChangeUpdate::WriteError:
      mtqStatus->setText( i18n( "Error" ) );
      KPassivePopup::message( update.result().message(), this );
      break;
    case QSync::SyncChangeUpdate::ReceiveError:
      mtqStatus->setText( i18n( "Error" ) );
      KPassivePopup::message( update.result().message(), this );
      break;
    default:
      mtqStatus->setText( TQString() );
      break;
  }
}

void GroupItem::mapping( const QSync::SyncMappingUpdate& )
{
}

void GroupItem::engine( const QSync::SyncEngineUpdate &update )
{
  switch ( update.type() ) {
    case QSync::SyncEngineUpdate::EndPhaseConnected:
      mtqStatus->setText( i18n( "Connected" ) );
      mProgressBar->setProgress( 0 );
      mSynchronizing = true;
      mSyncAction->setText( "Abort Synchronization" );
      break;
    case QSync::SyncEngineUpdate::EndPhaseRead:
      mtqStatus->setText( i18n( "Data read" ) );
      break;
    case QSync::SyncEngineUpdate::EndPhaseWrite:
      mtqStatus->setText( i18n( "Data written" ) );
      mProgressBar->setProgress( 100 );
      mProcessedItems = mMaxProcessedItems = 0;
      break;
    case QSync::SyncEngineUpdate::EndPhaseDisconnected:
      mtqStatus->setText( i18n( "Disconnected" ) );
      break;
    case QSync::SyncEngineUpdate::Error:
      mtqStatus->setText( i18n( "Synchronization failed" ) );
      KPassivePopup::message( update.result().message(), this );
      this->update();

      mSynchronizing = false;
      mSyncAction->setText( i18n( "Synchronize Now" ) );
      break;
    case QSync::SyncEngineUpdate::SyncSuccessfull:
      mtqStatus->setText( i18n( "Successfully synchronized" ) );
      mSyncProcess->group().setLastSynchronization( TQDateTime::tqcurrentDateTime() );
      mSyncProcess->group().save();
      this->update();

      mSynchronizing = false;
      mSyncAction->setText( i18n( "Synchronize Now" ) );
      break;
    case QSync::SyncEngineUpdate::PrevUnclean:
      mtqStatus->setText( i18n( "Previous synchronization failed" ) );
      break;
    case QSync::SyncEngineUpdate::EndConflicts:
      mtqStatus->setText( i18n( "Conflicts solved" ) );
      mMaxProcessedItems = mProcessedItems;
      break;
    default:
      mtqStatus->setText( TQString() );
      break;
  }
}

void GroupItem::member( const QSync::SyncMemberUpdate &update )
{
  TQValueList<MemberItem*>::Iterator it;
  for ( it = mMemberItems.begin(); it != mMemberItems.end(); ++it ) {
    if ( (*it)->member() == update.member() ) {
      switch ( update.type() ) {
        case QSync::SyncMemberUpdate::Connected:
          (*it)->seStatusMessage( i18n( "Connected" ) );
          break;
        case QSync::SyncMemberUpdate::SentChanges:
          (*it)->seStatusMessage( i18n( "Changes read" ) );
          break;
        case QSync::SyncMemberUpdate::CommittedAll:
          (*it)->seStatusMessage( i18n( "Changes written" ) );
          break;
        case QSync::SyncMemberUpdate::Disconnected:
          (*it)->seStatusMessage( i18n( "Disconnected" ) );
          break;
        case QSync::SyncMemberUpdate::ConnectError:
          (*it)->seStatusMessage( i18n( "Error: %1" ).tqarg( update.result().message() ) );
          break;
        case QSync::SyncMemberUpdate::GetChangesError:
          (*it)->seStatusMessage( i18n( "Error: %1" ).tqarg( update.result().message() ) );
          break;
        case QSync::SyncMemberUpdate::CommittedAllError:
          (*it)->seStatusMessage( i18n( "Error: %1" ).tqarg( update.result().message() ) );
          break;
        case QSync::SyncMemberUpdate::SyncDoneError:
          (*it)->seStatusMessage( i18n( "Error: %1" ).tqarg( update.result().message() ) );
          break;
        case QSync::SyncMemberUpdate::DisconnectedError:
          (*it)->seStatusMessage( i18n( "Error: %1" ).tqarg( update.result().message() ) );
          break;
        default:
          break;
      }

      return;
    }
  }
}

void GroupItem::synchronize()
{
  if ( !mSynchronizing )
    emit synchronizeGroup( mSyncProcess );
  else
    emit abortSynchronizeGroup( mSyncProcess );
}

void GroupItem::configure()
{
  emit configureGroup( mSyncProcess );

  this->update();
}

void GroupItem::engineChanged( QSync::Engine *engine )
{
  Q_ASSERT( engine );

  mCallbackHandler->setEngine( engine );

  this->update();
}

MemberItem::MemberItem( TQWidget *parent, SyncProcess *process,
                        const QSync::Member &member )
  : TQWidget( parent ), mSyncProcess( process ), mMember( member )
{
  TQFont boldFont;
  boldFont.setBold( true );

  MemberInfo mi( member );

  TQPixmap icon = mi.smallIcon();

  QSync::Plugin plugin = member.plugin();

  TQVBoxLayout *tqlayout = new TQVBoxLayout( this );

  TQHBox* box = new TQHBox( this );
  box->setMargin( 5 );
  box->setSpacing( 6 );
  tqlayout->addWidget( box );

  mIcon = new TQLabel( box );
  mIcon->setPixmap( icon );
  mIcon->tqsetAlignment( TQt::AlignTop );
  mIcon->setFixedWidth( mIcon->tqsizeHint().width() );

  TQVBox *nameBox = new TQVBox( box );
  mMemberName = new TQLabel( nameBox );
  mMemberName->setFont( boldFont );
  mDescription = new TQLabel( nameBox );

  mtqStatus = new TQLabel( box );

  mMemberName->setText( member.name() );
  mDescription->setText( plugin.longName() );
}

void MemberItem::seStatusMessage( const TQString &msg )
{
  mtqStatus->setText( msg );
}

#include "groupitem.moc"
