/* ****************************************************************************
  This file is part of KBabel

  Copyright (C) 2000 by Matthias Kiefer
                            <matthias.kiefer@gmx.de>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

  In addition, as a special exception, the copyright holders give
  permission to link the code of this program with any edition of
  the TQt library by Trolltech AS, Norway (or with modified versions
  of TQt that use the same license as TQt), and distribute linked
  combinations including the two.  You must obey the GNU General
  Public License in all respects for all of the code used other than
  TQt. If you modify this file, you may extend this exception to
  your version of the file, but you are not obligated to do so.  If
  you do not wish to do so, delete this exception statement from
  your version.

**************************************************************************** */
#ifndef KBABELDICTBOX_H
#define KBABELDICTBOX_H

#include "kbabeldictiface.h"
#include "searchengine.h"
#include "catalog.h"

#include <tqdict.h>
#include <tqguardedptr.h>
#include <tqptrlist.h>
#include <tqstringlist.h>

class KListView;
class TQLabel;
class TQListViewItem;
class TQPopupMenu;
class TQPushButton;
class TQSplitter;
class TQTextView;
class TQToolButton;

struct ModuleInfo
{
   TQString id;
   TQString name;
   
   bool editable;
};


class KDE_EXPORT KBabelDictBox : public TQWidget, virtual public KBabelDictIFace
{ 
    Q_OBJECT
  TQ_OBJECT

public:
   KBabelDictBox( TQWidget* parent = 0, const char* name = 0
                  , WFlags fl = 0 );
   ~KBabelDictBox();

   /** @returns ids of available modules */
   virtual TQStringList modules();

   /** @returns (translated) names of available modules */
   TQStringList moduleNames();

   /**
    * @returns a list with information about the available modules
    * Take care about, that you have to delete the items by yourself.
    */
   TQPtrList<ModuleInfo> moduleInfos();
   
   /** @returns preferencesWidgets of modules */
   TQPtrList<PrefWidget> modPrefWidgets(TQWidget* parent);
   
   int activeModule();
   bool activeModuleEditable();

   /**
    * lets the modules always update their options before
    * starting a new search
    */
   void setAutoUpdateOptions(bool on);

   void nextResult();
   void prevResult();
   void startSearch(const TQString);
   void startTranslationSearch(const TQString);
   void startDelayedSearch(const TQString);
   void startDelayedTranslationSearch(const TQString);
   TQString translate(const TQString);
   TQString searchTranslation(const TQString, int& score);
   TQString fuzzyTranslation(const TQString, int& score);
   void stopSearch();
   void setActiveModule(TQString name);

   /**
    * @returns true, if a search is active at the moment
    */
   bool isSearching();

   /**
    * Saves the settings including the settings of the different modules
    * in the given config object.
    */
   void saveSettings(KConfigBase *config);
   
   /**
    * Saves the settings of the given module
    * in the given config object.
    */
   void saveSettings(const TQString& moduleId, KConfigBase *config);
   
   /**
    * Reads the settings including the settings of the different modules
    * from the given config object.
    */
   void readSettings(KConfigBase *config);

   /**
    * Reads the settings of the given module
    * from the given config object.
    */
   void readSettings(const TQString& moduleId, KConfigBase *config);
   


   /** @returns the translation of the current search result */
   TQString translation();

   bool hasSelectedText() const;
   TQString selectedText() const;

   void setRMBMenu(TQPopupMenu *popup);
   

   virtual bool eventFilter(TQObject *, TQEvent*);
   

   /**
    * A hack for kbabel to get a list of messages for a specific package
    * from dbsearchengine. When dbsearchengine allows multiple access to
    * the database this will get removed.
    */
   bool messagesForPackage(const TQString& package
           , TQValueList<KBabel::DiffEntry>& resultList, TQString& error); 
   
public slots:
   virtual void setActiveModule(int);
   void slotStartSearch(const TQString&);
   void slotStopSearch();
   void slotNextResult();
   void slotPrevResult();

   void about();
   void aboutModule(const TQString& moduleID);
   void aboutActiveModule();

   /**
    * sets the name of the package currently edited
    */
   void setEditedPackage(const TQString& packageName);
   
   /**
    * sets the filepath of the package currently edited
    */
   void setEditedFile(const TQString& path);

   /** sets the language code to use */
   void setLanguage(const TQString& languageCode
           , const TQString& languageName);


   void setTextChanged(const TQStringList& orig, const TQString& translation,
           uint pluralForm, const TQString& description);
   
   /**
    * if text is marked, copy this into the clipboard, otherwise
    * copy the current translation into the clipboard
    */
   void copy();
   
   void configure(const TQString& moduleID, bool modal=false);
   void edit(const TQString& moduleID);
   void edit();

   void clear();
   

signals:
   void searchStarted();
   void searchStopped();
   void progressed(int);
   void progressStarts(const TQString&);
   void progressEnds();
   void activeModuleChanged(int);
   void activeModuleChanged(bool editable);
   void errorInModule(const TQString& error);

   /** 
    * emitted when either the order 
    * or the number of modules was changed
    */
   void modulesChanged();

protected slots:
   void showResult(TQListViewItem*);
   void addResult(const SearchResult*);
   void nextInfo();
   void showDetailsOnly();
   void showListOnly();
   void clearModuleResults();
   void editFile();
   void showContextMenu(KListView *, TQListViewItem *, const TQPoint&);

   /**
    * This slots gets connected to a SearchEngine's searchStopped() signal,
    * when a new search is requested although a search is currently active.
    */
   void startDelayedSearch();
   void startDelayedTranslationSearch();
   

   /**
    * This slot is connected to the finished signal of @ref KDialogBase
    * to destroy this dialog after the user has closed it.
    */
   void destroyConfigDialog();
   
protected:
   virtual void wheelEvent(TQWheelEvent*);
   
private:
   /**
   * Register the module and connect the slots
   */
   void registerModule( SearchEngine* module);
   
   TQPtrList<SearchEngine> moduleList;
   int active;
   int currentResult;
   int currentInfo;
   int total;

   /** stores a string to be used with delayed search */
   TQString searchText;

   TQDict<TQWidget> prefDialogs;

   TQLabel *translatorLabel;
   TQLabel *locationLabel;
   TQLabel *totalResultsLabel;
   TQLabel *currentLabel;
   TQLabel *dateLabel;
   TQPushButton *moreButton;
   TQPushButton *nextButton;
   TQPushButton *prevButton;
   TQTextView *origView;
   TQTextView *translationView;
   TQSplitter *viewContainer;
   KListView *resultListView;
   TQSplitter *resultSplitter;

   TQToolButton *listButton;
   TQToolButton *detailButton;

   int editFileIndex;
   TQGuardedPtr<TQPopupMenu> rmbPopup;
};

#endif // KBABELDICTBOX_H
