/* This file is part of the KDE project
   Copyright (C) 2003 Fabian Wolf <fabianw@gmx.net>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; version 2
   of the License.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/

#include <kfilemetainfo.h>
#include <klocale.h>
#include <kmimetype.h>
#include <kurl.h>
#include <tqvbox.h>
#include <tqlabel.h>
#include <tqpixmap.h>
#include <kio/netaccess.h>
#include <kurllabel.h>
#include <kmimetype.h>
#include <kmediaplayer/player.h>
#include <kparts/componentfactory.h>
#include <ksqueezedtextlabel.h>

#include "audiopreview.h"

AudioPreview::AudioPreview( TQWidget *parent, const char *name, const TQString &fileName, const TQString &mimeType)
  : TQVBox( parent, name )
{
  m_isTempFile = false;
  pic = 0;
  m_player = 0L;
  description = 0;
  // fileName is created by KUrl::prettyURL()
  KURL url( fileName );
  setSpacing( 0 );
  if( url.isValid() && url.isLocalFile() ) {
    m_localFile = url.path();
    pic = new TQLabel(this);
    pic->setPixmap(KMimeType::pixmapForURL( url ));
    pic->adjustSize();
    initView( mimeType );
  } else if( !url.isLocalFile() ) {
    KURLLabel *label = new KURLLabel( this );
    label->setText(i18n("This audio file isn't stored\non the local host.\nClick on this label to load it.\n" ) );
    label->setURL( url.prettyURL() );
    connect(label, TQT_SIGNAL(leftClickedURL(const TQString&)), TQT_SLOT(downloadFile(const TQString&)));
    pic = label;
  } else {
    description = new TQLabel(this );
    description->setText(i18n("Unable to load audio file") );
  }
}

AudioPreview::~AudioPreview()
{
  if ( m_isTempFile )
    KIO::NetAccess::removeTempFile( m_localFile );

  delete m_player;
}

void AudioPreview::initView( const TQString& mimeType )
{
  KURL url = KURL::fromPathOrURL( m_localFile );
  pic->setText( TQString() );
  pic->setPixmap(KMimeType::pixmapForURL( url ));
  pic->adjustSize();
 
  KFileMetaInfo info(m_localFile);
  KMimeType::Ptr mimeptr = KMimeType::mimeType(mimeType);

  TQString desc;
  if (info.isValid()) 
  {
    if (mimeptr->is("audio/x-mp3") || mimeptr->is("application/ogg"))
    {
      // following 3 infos might be very long; make sure they get squeezed
      // TODO: when string-freeze is over, eliminate trailing '\n' as it's no longer needed
      KSqueezedTextLabel *sl;

      sl = new KSqueezedTextLabel(this);
      sl->setText(i18n("Artist: %1\n").tqarg(info.item("Artist").value().toString()));

      sl = new KSqueezedTextLabel(this);
      sl->setText(i18n("Title: %1\n").tqarg(info.item("Title").value().toString()));

      sl = new KSqueezedTextLabel(this);
      sl->setText(i18n("Comment: %1\n").tqarg(info.item("Comment").value().toString()));

      desc.append(i18n("Biterate: 160 kbits/s", "Bitrate: %1 %2\n").tqarg( info.item("Bitrate").value().toString() ).tqarg( info.item("Bitrate").suffix() ));
    }
    desc.append(i18n("Sample rate: %1 %2\n").tqarg( info.item("Sample Rate").value().toString() ).tqarg( info.item("Sample Rate").suffix() ));
    desc.append(i18n("Length: "));

    /* Calculate length in mm:ss format */
    int length = info.item("Length").value().toInt();
    if (length/60 < 10)
      desc.append("0");
    desc.append(TQString("%1:").tqarg(length/60, 0, 10));
    if (length%60 < 10)
      desc.append("0");
    desc.append(TQString("%1\n").tqarg(length%60, 0, 10));
  }
 
  description = new TQLabel(this);
  description->setText( desc );
  description->adjustSize();
  m_player = KParts::ComponentFactory::createInstanceFromQuery<KMediaPlayer::Player>( "KMediaPlayer/Player", TQString(), TQT_TQOBJECT(this) );
  if ( m_player )
  {
    static_cast<KParts::ReadOnlyPart*>(m_player)->openURL( url );
    m_player->widget()->show();
  }
}

void AudioPreview::downloadFile( const TQString& url )
{
  if( KIO::NetAccess::download( KURL::fromPathOrURL( url ), m_localFile , tqtopLevelWidget()) )
  {
    m_isTempFile = true;
    initView( KMimeType::findByPath( m_localFile )->name() );
  }
}

#include <audiopreview.moc>
