/***************************************************************************
 *   Copyright (C) 1997-2000 by Dimitri van Heesch                         *
 *   dimitri@stack.nl                                                      *
 *   Copyright (C) 2001 by Bernd Gehrmann                                  *
 *   bernd@kdevelop.org                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <tqlabel.h>
#include <tqlayout.h>
#include <tqcombobox.h>
#include <klineedit.h>
#include <tqpushbutton.h>
#include <tqspinbox.h>
#include <tqtooltip.h>
#include <klocale.h>
#include <kfiledialog.h>
#include <kglobal.h>
#include <kiconloader.h>

#include "input.h"


static const char * const add_xpm_data[] =
{
  "16 16 5 1",
  ". c None",
  "* c #0328f9",
  "# c #354396",
  "a c #353740",
  "c c #999999",
  "................",
  "......###.......",
  "......#*ac......",
  "......#*ac......",
  "......#*ac......",
  "......#*ac......",
  ".######*a#####..",
  ".#***********ac.",
  ".#aaaaa*aaaaaac.",
  "..cccc#*acccccc.",
  "......#*ac......",
  "......#*ac......",
  "......#*ac......",
  "......#aac......",
  ".......ccc......",
  "................"
};
const char **add_xpm = (const char **)add_xpm_data;

static const char * const del_xpm_data[] =
{
  "16 16 5 1",
  ". c None",
  "* c #0328f9",
  "# c #354396",
  "a c #353740",
  "c c #999999",
  "................",
  "................",
  "................",
  "................",
  "................",
  "................",
  ".#############..",
  ".#***********ac.",
  ".aaaaaaaaaaaaac.",
  "..ccccccccccccc.",
  "................",
  "................",
  "................",
  "................",
  "................",
  "................"
};
const char **del_xpm = (const char **)del_xpm_data;

static const char* const update_xpm_data[] =
{
  "16 16 5 1",
  /* colors */
  ". c #0328f9",
  "# c #354396",
  "a c #353740",
  "b c None",
  "c c #999999",
  /* pixels */
  "bbbbbbbbbbbbbbbb",
  "bbbbbbbb#####acb",
  "bbbbbbbb#....abb",
  "bbc##cbb#...acbb",
  "bb#..abb#....abb",
  "bc#..abb#.a..acb",
  "b#..acbbaac#..ab",
  "b#..abbbcbb#..ab",
  "b#..abbbbbb#..ab",
  "b#..acbbbbc#..ab",
  "bc#..#cbbc#..acb",
  "bb#...####...acb",
  "bbca........acbb",
  "bbbbaa....aaccbb",
  "bbbbbcaaaaccbbbb",
  "bbbbbbbbbbbbbbbb"
};
const char **update_xpm = (const char **)update_xpm_data;


InputBool::InputBool(const TQCString &k, const TQString &text, TQWidget * parent, bool &flag)
    : TQWidget(parent), state(flag), key(k)
{
    TQHBoxLayout *tqlayout = new TQHBoxLayout(this);
    cb = new TQCheckBox(text,this);

    init();

    tqlayout->addWidget(cb);
    tqlayout->addStretch(1);
    
    connect( cb, TQT_SIGNAL(toggled(bool)), this, TQT_SLOT(valueChanged(bool)));
}


InputBool::~InputBool()
{}


void InputBool::init()
{
    cb->setChecked(state);
}


void InputBool::valueChanged(bool s)
{
    if (s != state) {
        emit changed();
        emit toggle(key, s);
    }
    state = s;
}


void InputBool::setEnabled(bool b)
{
    cb->setEnabled(b);
}


InputInt::InputInt(const TQString &label, TQWidget *parent, int &val, int minVal, int maxVal)
    : TQWidget(parent), m_val(val), m_minVal(minVal), m_maxVal(maxVal)
{
    TQHBoxLayout *tqlayout = new TQHBoxLayout(this, 5);
    
    sp = new TQSpinBox(minVal, maxVal, 1, this);
    lab = new TQLabel(sp, label+":", this);
    
    init();
  
    tqlayout->addWidget(lab);
    tqlayout->addWidget(sp);
    tqlayout->addStretch(1);

    connect(sp, TQT_SIGNAL(valueChanged(int)), this, TQT_SLOT(valueChanged(int)));
}


InputInt::~InputInt()
{}


void InputInt::init()
{
    m_val = TQMAX(m_minVal, m_val);
    m_val = TQMIN(m_maxVal, m_val);
    sp->setValue(m_val);
}


void InputInt::valueChanged(int val)
{
    if (val != m_val)
        emit changed(); 
    m_val = val;
}


void InputInt::setEnabled(bool state)
{
    lab->setEnabled(state);
    sp->setEnabled(state);
}


InputString::InputString(const TQString & label, 
                         TQWidget *parent, TQCString &s, StringMode m)
    : TQWidget(parent), str(s), sm(m), m_values(0), m_index(0)
{
    le = 0; br = 0; com = 0;
    
    if (m == StringFixed) {
        TQHBoxLayout *tqlayout = new TQHBoxLayout(this, 5);
        com = new TQComboBox(this); 
	lab = new TQLabel(com,label+":", this);
	tqlayout->addWidget(lab);
        tqlayout->addWidget(com);
        tqlayout->addStretch(1);
    } else {
        TQGridLayout *tqlayout = new TQGridLayout(this, 1, m==StringFree? 1 : 3, 5);
        le = new KLineEdit(this);
	lab = new TQLabel(le,label+":", this);
	tqlayout->addWidget(lab, 0, 0);
        le->setText(s);
        tqlayout->addWidget(le, 0, 1);
        
        if (m == StringFile || m == StringDir) {
            br = new TQPushButton(this);
            br->setPixmap(SmallIcon(m==StringFile? "document" : "folder"));
            TQToolTip::add(br, m==StringFile? i18n("Browse to a file") : i18n("Browse to a folder"));
            tqlayout->addWidget(br, 0, 2);
        }
    }
    
    if (le)
        connect( le,   TQT_SIGNAL(textChanged(const TQString&)), 
                 this, TQT_SLOT(textChanged(const TQString&)) );
    if (br)
        connect( br,   TQT_SIGNAL(clicked()), this, TQT_SLOT(browse()) );
    if (com)
        connect( com,  TQT_SIGNAL(activated(const TQString &)), 
                 this, TQT_SLOT(textChanged(const TQString &)) );
}

InputString::~InputString()
{
    if (m_values)
        delete m_values;
}


void InputString::init()
{
    if (sm == StringFixed) {
        int *itemIndex = m_values->find(str);
        if (itemIndex) 
            com->setCurrentItem(*itemIndex);
        else
            com->setCurrentItem(0);
    } else
        le->setText(str);
}


void InputString::addValue(const char *s)
{
    if (sm == StringFixed) {
        if (!m_values)
            m_values = new TQDict<int>;
        m_values->setAutoDelete(true);
        m_values->insert(s, new int(m_index++));
        com->insertItem(s);
    }
}


void InputString::clear()
{
    le->setText("");
    if (!str.isEmpty()) {
        emit changed();
        str = "";
    }
}


void InputString::textChanged(const TQString &s)
{
    if (str!=s.latin1()) {
        str = s.latin1();
        emit changed();
    }
}

void InputString::setEnabled(bool state)
{
    lab->setEnabled(state);
    if (le)
        le->setEnabled(state);
    if (br)
        br->setEnabled(state);
    if (com)
        com->setEnabled(state);
}


void InputString::browse()
{
    if (sm == StringFile) {
        TQString fileName = KFileDialog::getOpenFileName();
    
        if (!fileName.isNull()) {
            le->setText(fileName);
            if (str != le->text().latin1()) {
                str = le->text().latin1(); 
                emit changed();
            }
        }
    } else { // sm==StringDir
        TQString dirName = KFileDialog::getExistingDirectory();

        if (!dirName.isNull()) {
            le->setText( dirName ); 	
            if (str != le->text().latin1()) {
                str = le->text().latin1();
                emit changed();
            }
        }	
    }
}


InputStrList::InputStrList(const TQString & label, 
                           TQWidget *parent, TQStrList &sl, ListMode lm)
    : TQWidget(parent), strList(sl)
{
    TQGridLayout *tqlayout = new TQGridLayout(this, 2, 2, 5);
    
    TQWidget *dw = new TQWidget(this); /* dummy widget used for layouting */
    TQHBoxLayout *boxtqlayout = new TQHBoxLayout(dw, 0, 5);
    le  = new KLineEdit(dw);
    lab = new TQLabel(le,label+":", this );
    tqlayout->addWidget(lab, 0, 0);
    boxtqlayout->addWidget(le, 1);

    add = new TQPushButton(dw);
    add->setPixmap(TQPixmap( add_xpm ));
    TQToolTip::add(add, i18n("Add item"));
    boxtqlayout->addWidget(add);
    
    del = new TQPushButton(dw);
    del->setPixmap(TQPixmap( del_xpm ));
    TQToolTip::add(del, i18n("Delete selected item"));
    boxtqlayout->addWidget(del);

    upd = new TQPushButton(dw); 
    upd->setPixmap(TQPixmap( update_xpm ));
    TQToolTip::add(upd, i18n("Update selected item"));
    boxtqlayout->addWidget(upd);
    
    lb  = new TQListBox(this);
    lb->setMinimumSize(400, 100);
    init();
    lb->setVScrollBarMode(TQScrollView::Auto);
    lb->setHScrollBarMode(TQScrollView::Auto);
    
    brFile = 0;
    brDir = 0;
    if (lm != ListString) {
        if (lm & ListFile) {
            brFile = new TQPushButton(dw);
            brFile->setPixmap(SmallIcon("document"));
            TQToolTip::add(brFile, i18n("Browse to a file"));
            boxtqlayout->addWidget(brFile);
        } 
        if (lm & ListDir) {
            brDir = new TQPushButton(dw);
            brDir->setPixmap(SmallIcon("folder"));
            TQToolTip::add(brDir, i18n("Browse to a folder"));
            boxtqlayout->addWidget(brDir);
        }
    }
    tqlayout->addWidget(dw, 0, 1);
    tqlayout->addWidget(lb, 1, 1);
    
    connect( le,   TQT_SIGNAL(returnPressed()), 
             this, TQT_SLOT(addString()) );
    connect( add,  TQT_SIGNAL(clicked()), 
             this, TQT_SLOT(addString()) );
    connect( del,  TQT_SIGNAL(clicked()), 
             this, TQT_SLOT(delString()) );
    connect( upd,  TQT_SIGNAL(clicked()), 
             this, TQT_SLOT(updateString()) );
    if (brFile) 
        connect( brFile, TQT_SIGNAL(clicked()),
                 this, TQT_SLOT(browseFiles()) );
    if (brDir)
        connect( brDir, TQT_SIGNAL(clicked()),
                 this, TQT_SLOT(browseDir()) );
    connect( lb,   TQT_SIGNAL(selected(const TQString &)), 
             this, TQT_SLOT(selectText(const TQString &)) );
    
    strList = sl;
}


InputStrList::~InputStrList()
{}


void InputStrList::init()
{
    le->clear();
    lb->clear();
    char *s = strList.first();
    while (s) {
        lb->insertItem(s);
        s = strList.next();
    }
}


void InputStrList::addString()
{
    if (!le->text().isEmpty()) {
        lb->insertItem(le->text());
        strList.append(le->text().latin1());
        emit changed();
        le->clear();
    }
}


void InputStrList::delString()
{
    if (lb->currentItem() != -1) {
        int itemIndex = lb->currentItem();
        lb->removeItem(itemIndex);
        strList.remove(itemIndex);
        emit changed();
    }
}


void InputStrList::updateString()
{
    if (lb->currentItem() != -1 && !le->text().isEmpty()) {
        lb->changeItem(le->text(),lb->currentItem());
        strList.insert(lb->currentItem(),le->text().latin1());
        strList.remove(lb->currentItem()+1);
        emit changed();
    }
}


void InputStrList::selectText(const TQString &s)
{
    le->setText(s);
}


void InputStrList::setEnabled(bool state)
{
    lab->setEnabled(state);
    le->setEnabled(state);
    add->setEnabled(state);
    del->setEnabled(state);
    upd->setEnabled(state);
    lb->setEnabled(state);
    if (brFile)
        brFile->setEnabled(state);
    if (brDir)
        brDir->setEnabled(state);
}


void InputStrList::browseFiles()
{
    TQStringList fileNames = KFileDialog::getOpenFileNames();	
    
    if (!fileNames.isEmpty()) {
        TQStringList::Iterator it;
        for (it = fileNames.begin(); it != fileNames.end(); ++it) {
            lb->insertItem(*it);
            strList.append(( *it ).latin1());
            emit changed();
        }
        le->setText(*fileNames.begin());
    }
}


void InputStrList::browseDir()
{	
    TQString dirName = KFileDialog::getExistingDirectory();	
    
    if (!dirName.isNull()) {
        lb->insertItem(dirName);
        strList.append(dirName.latin1());
        emit changed();
        le->setText(dirName);
    }
}


#include "input.moc"
