/***************************************************************************
 *   Copyright (C) 2002-2004 by Alexander Dymo  <cloudtemple@mskat.net>    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Library General Public License as       *
 *   published by the Free Software Foundation; either version 2 of the    *
 *   License, or (at your option) any later version.                       *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU Library General Public     *
 *   License along with this program; if not, write to the                 *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.             *
 ***************************************************************************/
#ifndef PROPERTY_H
#define PROPERTY_H

#include <tqvariant.h>

#include <tqmap.h>

class TQWidget;
class TQString;

/**Namespace which contain property editing classes.*/
namespace PropertyLib{

/** @file property.h
@short Contains @ref PropertyLib::Property class and @ref PropertyLib::Property::PropertyType enum.
*/

/**
@short Property.

It includes support for TQStringList properties, an i18n'ed label and stores an old value to allow undo.

Contains name, type and value.

Type can be one of predefined types (including standard @ref TQVariant types) by @ref PropertyLib::Property::PropertyType 
enum or custom user type. User defined types should have values more than 3000.

Value is a @ref TQVariant.

Property can optionally have a list of possible values.
In that case use @ref ValueFromList type and valueList member.
Use @ref description for i18n'ed label.

Examples:
creating property:
\code 
Property *property = new Property(String, name, description, value)
\endcode
using convenience constructor to create property of ValueFromList type:
\code
Property *property = new Property(name, possibleValuesList, description, value);
\endcode
*/
class Property {
public:
    /** PropertyType.
        Integers that represent the type of the property. */
    enum PropertyType {
        //standard supported TQVariant types
        Invalid = TQVariant::Invalid        /**<invalid property type*/,
        Map = TQVariant::Map                /**<TQMap<TQString, TQVariant>*/,
        List = TQVariant::List              /**<TQValueList<TQVariant>*/,       
        String = TQVariant::String          /**<string*/,
        StringList = TQVariant::StringList  /**<string list*/,
        Font = TQVariant::Font              /**<font*/,
        Pixmap = TQVariant::Pixmap          /**<pixmap*/,
        //@todo implement TQVariant::Brush
        Rect = TQVariant::Rect              /**<rectangle (x,y, width, height)*/,
        Size = TQVariant::Size              /**<size (width, height)*/,
        Color = TQVariant::Color            /**<color*/,
        //@todo implement TQVariant::Palette
        //@todo implement TQVariant::ColorGroup
        //@todo implement TQVariant::IconSet
        Point = TQVariant::Point            /**<point (x,y)*/,
        //@todo implement TQVariant::Image
        Integer = TQVariant::Int            /**<integer*/,
        //@todo implement TQVariant::UInt
        Boolean = TQVariant::Bool           /**<boolean*/,
        Double = TQVariant::Double          /**<double*/,
        //@todo implement TQVariant::CString
        //@todo implement TQVariant::PointArray
        //@todo implement TQVariant::Region
        //@todo implement TQVariant::Bitmap
        Cursor = TQVariant::Cursor          /**<cursor*/,
        SizePolicy = TQVariant::SizePolicy  /**<size policy (horizontal, vertical)*/,
        Date = TQVariant::Date              /**<date*/,
        //@todo implement TQVariant::Time
        DateTime = TQVariant::DateTime      /**<date and time*/,
        //@todo implement TQVariant::ByteArray
        //@todo implement TQVariant::BitArray
        //@todo implement TQVariant::KeySequence
        //@todo implement TQVariant::Pen
        //@todo implement TQVariant::Long
        //@todo implement TQVariant::LongLong
        //@todo implement TQVariant::ULongLong
        

        //predefined custom types
        ValueFromList = 2000               /**<string value from a list*/,
        Symbol = 2001                      /**<tqunicode symbol code*/,
        FontName = 2002                    /**<font name, e.g. "times new roman"*/,
        FileURL = 2003                     /**<url of a file*/,
        DirectoryURL = 2004                /**<url of a directory*/,
        LineStyle = 2005                   /**<line style*/,

        UserDefined = 3000                 /**<plugin defined properties should start here*/
    };

    /**Constructs empty property.*/
    Property() {}
    /**Constructs property.*/
    Property(int type, const TQString &name, const TQString &description,
        const TQVariant &value = TQVariant(), bool save = true, bool readOnly = false);
    /**Constructs property with @ref ValueFromList type.*/
    Property(const TQString &name, const TQMap<TQString, TQVariant> &v_valueList,
        const TQString &description, const TQVariant &value = TQVariant(), bool save = true, bool readOnly = false);
    virtual ~Property();

    virtual bool operator<(const Property &prop) const;

    /**@return the name of the property.*/
    virtual TQString name() const;
    /**Sets the name of the property.*/
    virtual void setName(const TQString &name);
    /**@return the type of the property.*/
    virtual int type() const;
    /**Sets the type of the property.*/
    virtual void setType(int type);
    /**@return the value of the property.*/
    virtual TQVariant value() const;
    /**Sets the value of the property.*/
    virtual void setValue(const TQVariant &value, bool rememberOldValue = true);
    /**@return the description of the property.*/
    virtual TQString description() const;
    /**Sets the description of the property.*/
    virtual void setDescription(const TQString &description);
    /**Sets the string-to-value correspondence list of the property.
    This is used to create comboboxes-like property editors.*/
    virtual void setValueList(const TQMap<TQString, TQVariant> &list);
    /**The string-to-value correspondence list of the property.*/
    TQMap<TQString, TQVariant> valueList;

    /**Tells if the property can be saved to a stream, xml, etc.
    There is a possibility to use "GUI" properties that aren't
    stored but used only in a GUI.*/
    virtual bool allowSaving() const;
    /**Tells if the property is read only.*/
    virtual bool readOnly() const;
    /**Tells if the property is visible.*/
    virtual bool visible() const;
    /**Set the visibility.*/
    virtual void setVisible(const bool visible);
    
    /**Gets the previous property value.*/
    virtual TQVariant oldValue() const;
    
private:
//    Property(Property &property) {};
//    void operator=(Property &property) {};

    int m_type;
    TQString m_name;
    TQString m_description;
    TQVariant m_value;
    TQVariant m_oldValue;
    bool m_save;
    bool m_readOnly;
    bool m_visible;
};

}

#endif
