/***************************************************************************
*   Copyright (C) 2001-2002 by Bernd Gehrmann                             *
*   bernd@kdevelop.org                                                    *
*                                                                         *
*   Copyright (C) 2002 by Victor Rder                                    *
*   victor_roeder@gmx.de                                                  *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
***************************************************************************/

#include <tqpainter.h>
#include <tqinputdialog.h>
#include <tqregexp.h>

#include <kiconloader.h>
#include "misc.h"
#include "autolistviewitems.h"
#include "autoprojectpart.h"
#include "autoprojectwidget.h"
#include "autodetailsview.h"

/**
* Class ProjectItem
*/

ProjectItem::ProjectItem( Type type, TQListView *parent, const TQString &text )
		: TQListViewItem( parent, text ), typ( type )
{
	bld = false;
}


ProjectItem::ProjectItem( Type type, ProjectItem *parent, const TQString &text )
		: TQListViewItem( parent, text ), typ( type )
{
	bld = false;
}


void ProjectItem::paintCell( TQPainter *p, const TQColorGroup &cg,
                             int column, int width, int tqalignment )
{
	if ( isBold() )
	{
		TQFont font( p->font() );
		font.setBold( true );
		p->setFont( font );
	}
	TQListViewItem::paintCell( p, cg, column, width, tqalignment );
}


/**
* Class SubprojectItem
*/

SubprojectItem::SubprojectItem( TQListView *parent, const TQString &text )
		: ProjectItem( Subproject, parent, text )
{
	init();
}


SubprojectItem::SubprojectItem( SubprojectItem *parent, const TQString &text )
		: ProjectItem( Subproject, parent, text )
{
	init();
}


void SubprojectItem::init()
{
	targets.setAutoDelete( true );
	setPixmap( 0, SmallIcon( "folder" ) );
}


TQString SubprojectItem::relativePath()
{
    TQString relpath = subdir;

    SubprojectItem *it = this;
    while ( (it= dynamic_cast<SubprojectItem*>(it->parent())) )
    {
        relpath.prepend(it->subdir + "/");
    }
    relpath.remove(0, 2);

    return relpath;
}


/**
* Class TargetItem
*/

TargetItem::TargetItem( TQListView *lv, bool group, const TQString &text )
		: ProjectItem( Target, lv, text )
{
	sources.setAutoDelete( true );
	setPixmap( 0, group ? SmallIcon( "tar" ) : SmallIcon( "binary" ) );
}


/**
* Class FileItem
*/

FileItem::FileItem( TQListView *lv, const TQString &text, bool set_is_subst )
		: ProjectItem( File, lv, text ) , is_subst(set_is_subst)
{
	if(!is_subst)
	{
		setPixmap( 0, SmallIcon( "document" ) );
	}
	else
	{
		setPixmap( 0, SmallIcon( "variablenew" ) );
	}
}


void FileItem::changeSubstitution()
{
if(!is_subst)
return;

	bool ok;
	TQString text = TQInputDialog::getText(
	                   i18n("Edit Substitution"), i18n("Substitution:"), TQLineEdit::Normal,
	                   name, &ok );
	if ( ok && !text.isEmpty() )
	{
		// user entered something and pressed OK
		TQString new_name = text;
		if(new_name == name)
			return;
		setText(0,new_name);
		changeMakefileEntry(new_name);
		name = new_name;
	}
	else
	{
		// user entered nothing or pressed Cancel

	}
}

void FileItem::changeMakefileEntry(const TQString& new_name)
{
	TargetItem* target = dynamic_cast<TargetItem*>(parent());

	TQMap<TQString,TQString> replaceMap;

	TQString canontargetname = AutoProjectTool::canonicalize(target->name);
	TQString varname;
	if( target->primary == "PROGRAMS" || target->primary == "LIBRARIES" || target->primary == "LTLIBRARIES" )
		varname = canontargetname + "_SOURCES";
	else
		varname = target->prefix + "_" + target->primary;
	if( AutoDetailsView* lv = dynamic_cast<AutoDetailsView*>(listView()) )
	{
 		if ( SubprojectItem* subProject = lv->m_part->m_widget->selectedSubproject() )
		{
			TQStringList sources = TQStringList::split(TQRegExp("[ \t\n]"), subProject->variables[varname]);
			TQStringList::iterator it = sources.find(name);
			(*it) = new_name;
			subProject->variables[varname] = sources.join(" ");
			replaceMap.insert(varname, subProject->variables[varname]);
		
			AutoProjectTool::addToMakefileam(subProject->path + "/Makefile.am", replaceMap);
		
			if(new_name == "")
				target->sources.remove(this);
		}
	}
}
