/* This file is part of ksirc
   Copyright (c) 2001 Malte Starostik <malte@kde.org>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License version 2 as published by the Free Software Foundation.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/

// $Id: ksparser.cpp 1246075 2011-08-10 06:08:18Z tpearson $

#include <tqstring.h>
#include <tqcolor.h>
#include <tqregexp.h>

#include <kdebug.h>

#include "ksopts.h"
#include "ksparser.h"

TQString KSParser::parse( const TQString &message )
{
    TQString res;
    m_tags.clear();
    m_attributes.clear();
    m_beeped = false;

    for (unsigned int i = 0; i < message.length();)
    {
        TQChar ch = message[i++];
        if (ch.latin1() == 0x03 || ch == '~' && i < message.length())
        {
            TQChar next = message[ i++ ];
            if (next.latin1() >= 0x30 && next.latin1() <= 0x39)
            {
                int fg = -1, len;
                int bg = -1;
                TQRegExp colorRex("^[0-9]+");
                if (colorRex.search(message.mid(--i)) >= 0)
                {
                    len = colorRex.matchedLength();
                    fg = message.mid(i, len).toInt();
                    i += len;
                }
                if (message[i] == ',')
                {
                    if (colorRex.search(message.mid(++i)) >= 0)
                    {
                        len = colorRex.matchedLength();
                        bg = message.mid(i, len).toInt();
                        i += len;
                    }
		}
                TQString at;
                TQColor c = ircColor(fg);
		if ( c.isValid() ) {
		    at += TQString( "color=\"%1\" " ).tqarg( c.name() );
		}

                c = ircColor(bg);
		if ( c.isValid() ) {
		    at += TQString( "bgcolor=\"%1\" " ).tqarg( c.name() );
		}
		if(at.length() > 0){
                    res += pushTag( "font", at );
		}

            }
	    else if (ch.latin1() == 0x03) {
		res += popTag( "font" );
		/*
		 * We moved it forward looking for the number, back
		 * up or we miss a character.  Don't do
		 * res += next since we need to actaully parse that one
                 */
                --i;
	    }
            else if (ch == '~')
            {
                switch (next)
                {
		case 'c':
                    res += popTag( "font" );
                    break;
		case 'C':
                    res += popAll();
                    break;
		case 'r':
		    res += toggleTag( "r" );
                    break;
                case 's': break;
                case 'b':
                    res += toggleTag( "b" );
                    break;
                case 'g':
                    m_beeped = true;
                    break;
                case 'u':
                    res += toggleTag( "u" );
                    break;
                case 'n':
                    res += toggleTag( "font", TQString( "color=\"%1\"" ).tqarg( ksopts->nickForeground.name() ) );
                    break;
                case 'o':
                    res += pushTag( "font", TQString( "color=\"%1\"" ).tqarg( ksopts->msgContainNick.name() ) );
		    break;
		case '#':
		    res += pushTag( "font", TQString( "color=\"#%1\"" ).tqarg( message.mid(i, 6) ));
		    i+=6;
                    break;
                case '~':
		    res += ch;
                    break;
		default:
		    res += ch;
		    res += next;
                    break;
                }
            }
        }
        else
            res += ch;

    }

    res.append( popAll() );

    return res;
}

TQString KSParser::pushTag(const TQString &tag, const TQString &attributes)
{
    TQString res;
    m_tags.push(tag);
    if (!m_attributes.contains(tag))
        m_attributes.insert(tag, attributes);
    else if (!attributes.isEmpty())
        m_attributes.replace(tag, attributes);
    res.append("<" + tag);
    if (!m_attributes[tag].isEmpty())
        res.append(" " + m_attributes[tag]);
    return res + ">";
}

TQString KSParser::popTag(const TQString &tag)
{
    if (!m_tags.contains(tag))
        return TQString();

    TQString res;
    TQValueStack<TQString> savedTags;
    while (m_tags.top() != tag)
    {
        savedTags.push(m_tags.pop());
        res.append("</" + savedTags.top() + ">");
    }
    res.append("</" + m_tags.pop() + ">");
    m_attributes.remove(tag);
    while (!savedTags.isEmpty())
        res.append(pushTag(savedTags.pop()));
    return res;
}

TQString KSParser::toggleTag(const TQString &tag, const TQString &attributes)
{
    return m_attributes.contains(tag) ? popTag(tag) : pushTag(tag, attributes);
}

TQString KSParser::popAll()
{
    TQString res;
    while (!m_tags.isEmpty())
        res.append("</" + m_tags.pop() + ">");
    m_attributes.clear();
    return res;
}

TQColor KSParser::ircColor(int code)
{
    if (code >= 0 && code < 16)
        return ksopts->ircColors[code];
    return TQColor();
}
