/*
  KPF - Public fileserver for KDE

  Copyright 2001 Rik Hemsley (rikkus) <rik@kde.org>

  Permission is hereby granted, free of charge, to any person obtaining a copy
  of this software and associated documentation files (the "Software"), to
  deal in the Software without restriction, including without limitation the
  rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
  sell copies of the Software, and to permit persons to whom the Software is
  furnished to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be included in
  all copies or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
  AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
  ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
  WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

#include <tqlayout.h>
#include <tqcheckbox.h>
#include <tqspinbox.h>
#include <tqlabel.h>
#include <tqframe.h>
#include <tqwhatsthis.h>
#include <tqpushbutton.h>
#include <tqwidgetstack.h>
#include <tqtimer.h>
#include <tqdir.h>
#include <tqlineedit.h>

#include <kapplication.h>
#include <kglobal.h>
#include <dcopclient.h>
#include <kdialogbase.h>
#include <kmessagebox.h>
#include <kurl.h>
#include <kconfig.h>
#include <klocale.h>
#include <kseparator.h>
#include <kgenericfactory.h>

#include "Defines.h"
#include "Defaults.h"
#include "PropertiesDialogPlugin.h"
#include "StartingKPFDialog.h"
#include "WebServerManager_stub.h"
#include "WebServer_stub.h"
#include "Help.h"

#include <dnssd/servicebrowser.h>

namespace KPF
{
  class ServerState
  {
    public:

      ServerState()
        : shared            (false),
          listenPort        (Config::DefaultListenPort),
          bandwidthLimit    (Config::DefaultBandwidthLimit),
//          connectionLimit   (Config::DefaultConnectionLimit),
          followSymlinks    (Config::DefaultFollowSymlinks)
      {
      }

      bool operator == (const ServerState & other) const
      {
        return
          (
            other.shared          == shared
            &&
            other.listenPort      == listenPort
            &&
            other.bandwidthLimit  == bandwidthLimit
            &&
//            other.connectionLimit == connectionLimit
//            &&
            other.followSymlinks  == followSymlinks
          );
      }

      bool operator != (const ServerState & other) const
      {
        return
          (
            other.shared          != shared
            ||
            other.listenPort      != listenPort
            ||
            other.bandwidthLimit  != bandwidthLimit
            ||
//            other.connectionLimit != connectionLimit
//            ||
            other.followSymlinks  != followSymlinks
          );
      }


      bool  shared;
      uint  listenPort;
      uint  bandwidthLimit;
//      uint  connectionLimit;
      TQString serverName;
      bool  followSymlinks;
  };

  class PropertiesDialogPlugin::Private
  {
    public:

      Private()
        : l_listenPort              (0L),
          l_bandwidthLimit          (0L),
//          l_connectionLimit         (0L),
          sb_listenPort             (0L),
          sb_bandwidthLimit         (0L),
//          sb_connectionLimit        (0L),
	  le_serverName             (0L),
          cb_followSymlinks         (0L),
          cb_share                  (0L),
          stack                     (0L),
          initWidget                (0L),
          configWidget              (0L),
          webServerManagerInterface (0L),
          kpfRunning                (false)
      {
      }

      TQLabel                * l_listenPort;
      TQLabel                * l_bandwidthLimit;
//      TQLabel                * l_connectionLimit;
      TQLabel                * l_serverName;
      TQLabel                * l_kpftqStatus;

      TQSpinBox              * sb_listenPort;
      TQSpinBox              * sb_bandwidthLimit;
//      TQSpinBox              * sb_connectionLimit;
      TQLineEdit             * le_serverName;

      TQCheckBox             * cb_followSymlinks;
      TQCheckBox             * cb_share;

      TQPushButton           * pb_startKPF;

      TQWidgetStack          * stack;
      TQWidget               * initWidget;
      TQWidget               * configWidget;

      WebServerManager_stub * webServerManagerInterface;

      bool                    kpfRunning;
      DCOPRef                 webServerRef;
      KURL                    url;

      ServerState             currentState;
      ServerState             wantedState;
  };

  PropertiesDialogPlugin::PropertiesDialogPlugin(KPropertiesDialog * dialog,
                                                 const char *,
                                                 const TQStringList &)
    : KPropsDlgPlugin(dialog)
  {
    d = new Private;

    d->webServerManagerInterface =
      new WebServerManager_stub("kpf", "WebServerManager");

    d->url = dialog->kurl();

    if (
      d->url == TQDir::homeDirPath()
      ||  d->url == "file:" + TQDir::homeDirPath()
    )
    {
      // Don't even show ourselves if it's the home dir
      return;
    }

    TQWidget * widget = dialog->addPage(i18n("&Sharing"));

    d->stack = new TQWidgetStack(widget);

    TQVBoxLayout * stackLayout = new TQVBoxLayout(widget);
    stackLayout->addWidget(d->stack);

    d->initWidget   = createInitWidget(d->stack);
    d->configWidget = createConfigWidget(d->stack);

    d->stack->addWidget(d->initWidget,    0);
    d->stack->addWidget(d->configWidget,  1);

    kapp->dcopClient()->setNotifications(true);

    connect
      (
       kapp->dcopClient(),
       TQT_SIGNAL(applicationRegistered(const TQCString &)),
       TQT_SLOT(slotApplicationRegistered(const TQCString &))
      );

    connect
      (
       kapp->dcopClient(),
       TQT_SIGNAL(applicationRemoved(const TQCString &)),
       TQT_SLOT(slotApplicationUnregistered(const TQCString &))
      );

    d->kpfRunning = kapp->dcopClient()->isApplicationRegistered("kpf");

    if (!d->kpfRunning)
    {
      d->stack->raiseWidget(d->initWidget);
    }
    else
    {
      getServerRef();
      updateGUIFromCurrentState();
      d->stack->raiseWidget(d->configWidget);
    }
  }

  PropertiesDialogPlugin::~PropertiesDialogPlugin()
  {
    delete d->webServerManagerInterface;
    d->webServerManagerInterface = 0;

    delete d;
    d = 0;
  }

    void
  PropertiesDialogPlugin::slotSharingToggled(bool b)
  {
    if (b)
    {
      if (!userAcceptsWarning())
      {
        // Avoid loop.
        d->cb_share->blockSignals(true);
        d->cb_share->setChecked(false);
        d->cb_share->blockSignals(false);
        b = false;
      }
    }

    setControlsEnabled(b);
  }

    void
  PropertiesDialogPlugin::setControlsEnabled(bool b)
  {

    bool canPublish = b && DNSSD::ServiceBrowser::isAvailable() == DNSSD::ServiceBrowser::Working;
    
    d->l_serverName->setEnabled(canPublish);
    d->l_listenPort       ->setEnabled(b);
    d->l_bandwidthLimit   ->setEnabled(b);
//    d->l_connectionLimit  ->setEnabled(b);
    d->l_serverName       ->setEnabled(canPublish);
    
    d->sb_listenPort      ->setEnabled(b);
    d->sb_bandwidthLimit  ->setEnabled(b);
//    d->sb_connectionLimit ->setEnabled(b);
    d->le_serverName      ->setEnabled(canPublish);
    d->cb_followSymlinks  ->setEnabled(b);
  }

    TQWidget *
  PropertiesDialogPlugin::createInitWidget(TQWidget * parent)
  {
    TQWidget * w = new TQWidget(parent);

    TQLabel * about =
      new TQLabel
      (
       i18n
       (
        "<p>To share files via the web, you need to be"
        " running an 'applet' in your KDE panel. This"
        " 'applet' is a small program which provides"
        " file sharing capabilities."
        "</p>"
       ),
       w
      );

    d->pb_startKPF
      = new TQPushButton(i18n("Start Applet"), w);

    TQVBoxLayout * l = new TQVBoxLayout(w);

    l->addWidget(about);

    d->l_kpftqStatus =
      new TQLabel(i18n("Applet status: <strong>not running</strong>"), w);

    l->addWidget(d->l_kpftqStatus);

    TQHBoxLayout * l2 = new TQHBoxLayout(l);

    l2->addStretch(1);
    l2->addWidget(d->pb_startKPF);

    l->addStretch(1);

    connect(d->pb_startKPF, TQT_SIGNAL(clicked()), TQT_SLOT(slotStartKPF()));

    return w;
  }

    TQWidget *
  PropertiesDialogPlugin::createConfigWidget(TQWidget * parent)
  {
    TQWidget * w = new TQWidget(parent);

    d->cb_share =
      new TQCheckBox(i18n("Share this directory on the &Web"), w);

    d->l_listenPort       = new TQLabel(i18n("&Listen port:"),      w);
    d->l_bandwidthLimit   = new TQLabel(i18n("&Bandwidth limit:"),  w);
//    d->l_connectionLimit  = new TQLabel(i18n("Connection &limit"), w);
    d->l_serverName       = new TQLabel(i18n("&Server name:"), w);
    bool canPublish = DNSSD::ServiceBrowser::isAvailable() == DNSSD::ServiceBrowser::Working;
    d->l_serverName->setEnabled(canPublish);

    d->sb_listenPort      = new TQSpinBox(1000,  999999, 1, w);
    d->sb_bandwidthLimit  = new TQSpinBox(1,     999999, 1, w);
//    d->sb_connectionLimit = new TQSpinBox(1,     9999,   1, w);
    d->le_serverName      = new TQLineEdit( w);
    d->le_serverName->setEnabled(canPublish);
    
    d->cb_followSymlinks  =
      new TQCheckBox(i18n("&Follow symbolic links"), w);

    d->l_listenPort       ->setBuddy(d->sb_listenPort);
    d->l_serverName       ->setBuddy(d->le_serverName);
    d->l_bandwidthLimit   ->setBuddy(d->sb_bandwidthLimit);
//    d->l_connectionLimit  ->setBuddy(d->sb_connectionLimit);

    d->sb_listenPort      ->setValue(Config::DefaultListenPort);
    d->sb_bandwidthLimit  ->setValue(Config::DefaultBandwidthLimit);
    d->sb_bandwidthLimit  ->setSuffix(i18n("kB/s"));
//    d->sb_connectionLimit ->setValue(Config::DefaultConnectionLimit);
    d->cb_followSymlinks  ->setChecked(Config::DefaultFollowSymlinks);

    TQVBoxLayout * l0 =
      new TQVBoxLayout(w, KDialog::marginHint(), KDialog::spacingHint());

    l0->addWidget(d->cb_share);

    l0->addWidget(new KSeparator(TQFrame::HLine, w));

    TQGridLayout * l2 = new TQGridLayout(l0);

    l2->addWidget(d->l_listenPort,       0, 0);
    l2->addWidget(d->sb_listenPort,      0, 1);
    l2->addWidget(d->l_bandwidthLimit,   1, 0);
    l2->addWidget(d->sb_bandwidthLimit,  1, 1);
//    l2->addWidget(d->l_connectionLimit,  2, 0);
//    l2->addWidget(d->sb_connectionLimit, 2, 1);
    l2->addWidget(d->l_serverName,       2, 0);
    l2->addWidget(d->le_serverName,      2, 1);

    l0->addWidget(d->cb_followSymlinks);

    l0->addStretch(1);

    TQString shareHelp =
      i18n
      (
       "<p>"
       "Setting this option makes all files in this directory and"
       " any subdirectories available for reading to anyone"
       " who wishes to view them."
       "</p>"
       "<p>"
       "To view your files, a web browser or similar program"
       " may be used."
       "</p>"
       "<p>"
       "<strong>Warning!</strong> Before sharing a directory,"
       " you should be sure that it does not contain sensitive"
       " information, such as passwords, company secrets, your"
       " addressbook, etc."
       "</p>"
       "<p>"
       "Note that you cannot share your home directory"
       " (%1)"
       "</p>"
      )
      .tqarg(TQDir::homeDirPath());

    TQString listenPortHelp =
      i18n
      (
       "<p>"
       "Specify the network `port' on which the server should"
       " listen for connections."
       "</p>"
      );

    TQString bandwidthLimitHelp =
      i18n
      (
       "<p>"
       "Specify the maximum amount of data (in kilobytes) that will be"
       " sent out per second."
       "</p>"
       "<p>"
       "This allows you to keep some bandwidth for yourself instead"
       " of allowing connections with kpf to hog your connection."
       "</p>"
      );

    TQString connectionLimitHelp =
      i18n
      (
       "<p>"
       "Specify the maximum number of connections allowed at"
       " any one time."
       "</p>"
      );

    TQString followSymlinksHelp =
      i18n
      (
       "<p>"
       "Allow serving of files which have a symbolic link in"
       " the path from / to the file, or are a symbolic link"
       " themselves."
       "</p>"
       "<p>"
       "<strong>Warning!</strong> This could be a security"
       " risk. Use only if you understand the issues involved."
       "</p>"
      );
    TQString serverNameHelp = KPF::HelpText::getServerNameHelp();

    TQWhatsThis::add(d->cb_share,            shareHelp);
    TQWhatsThis::add(d->l_listenPort,        listenPortHelp);
    TQWhatsThis::add(d->sb_listenPort,       listenPortHelp);
    TQWhatsThis::add(d->l_bandwidthLimit,    bandwidthLimitHelp);
    TQWhatsThis::add(d->sb_bandwidthLimit,   bandwidthLimitHelp);
//    TQWhatsThis::add(d->l_connectionLimit,   connectionLimitHelp);
//    TQWhatsThis::add(d->sb_connectionLimit,  connectionLimitHelp);
    TQWhatsThis::add(d->l_serverName,        serverNameHelp);
    TQWhatsThis::add(d->le_serverName,       serverNameHelp);
    TQWhatsThis::add(d->cb_followSymlinks,   followSymlinksHelp);

    connect(d->cb_share, TQT_SIGNAL(toggled(bool)), TQT_SLOT(slotSharingToggled(bool)));

    slotSharingToggled(false);

    connect
      (
        d->cb_share,
        TQT_SIGNAL(toggled(bool)),
        TQT_SLOT(slotChanged())
      );

    connect
      (
        d->sb_listenPort,
        TQT_SIGNAL(valueChanged(int)),
        TQT_SLOT(slotChanged())
      );

    connect
      (
        d->sb_bandwidthLimit,
        TQT_SIGNAL(valueChanged(int)),
        TQT_SLOT(slotChanged())
      );

#if 0
    connect
      (
        d->sb_connectionLimit,
        TQT_SIGNAL(valueChanged(int)),
        TQT_SLOT(slotChanged())
      );
#endif
    connect
      (
        d->le_serverName,
	TQT_SIGNAL(textChanged(const TQString&)),
	TQT_SLOT(slotChanged())
      );

    connect
      (
        d->cb_followSymlinks,
        TQT_SIGNAL(toggled(bool)),
        TQT_SLOT(slotChanged())
      );

    return w;
  }

    void
  PropertiesDialogPlugin::slotStartKPF()
  {
    d->l_kpftqStatus
      ->setText(i18n("Applet status: <strong>starting...</strong>"));

    kapp->dcopClient()
      ->send("kicker", "default", "addApplet(TQString)", TQString("kpfapplet.desktop"));

    TQTimer::singleShot(4 * 1000, this, TQT_SLOT(slotStartKPFFailed()));
  }

    void
  PropertiesDialogPlugin::slotStartKPFFailed()
  {
    d->l_kpftqStatus
      ->setText(i18n("Applet status: <strong>failed to start</strong>"));

    d->pb_startKPF->setEnabled(true);
  }

    void
  PropertiesDialogPlugin::slotApplicationRegistered(const TQCString & s)
  {
    if ("kpf" == s)
    {
      d->kpfRunning = true;

      d->l_kpftqStatus
        ->setText(i18n("Applet status: <strong>running</strong>"));

      d->pb_startKPF->setEnabled(false);

      getServerRef();
      updateGUIFromCurrentState();
      d->stack->raiseWidget(d->configWidget);
    }
  }

    void
  PropertiesDialogPlugin::slotApplicationUnregistered(const TQCString & s)
  {
    if ("kpf" == s)
    {
      d->kpfRunning = false;

      d->webServerRef.clear();

      d->pb_startKPF->setEnabled(true);

      d->l_kpftqStatus
        ->setText(i18n("Applet status: <strong>not running</strong>"));

      d->stack->raiseWidget(d->initWidget);
    }
  }

    void
  PropertiesDialogPlugin::readSettings()
  {
    d->currentState = ServerState();

    if (!d->kpfRunning || d->webServerRef.isNull())
      return;

    d->currentState.shared = true;

    WebServer_stub webServer(d->webServerRef.app(), d->webServerRef.object());

    d->currentState.listenPort = webServer.listenPort();

    if (DCOPStub::CallFailed == webServer.status())
    {
      // TODO: warn user ?
      kpfDebug << "WebServer_stub call failed" << endl;
      d->currentState.listenPort = Config::DefaultListenPort;
      return;
    }

    d->currentState.bandwidthLimit = webServer.bandwidthLimit();

    if (DCOPStub::CallFailed == webServer.status())
    {
      // TODO: warn user ?
      kpfDebug << "WebServer_stub call failed" << endl;
      d->currentState.bandwidthLimit = Config::DefaultBandwidthLimit;
      return;
    }

#if 0
    d->currentState.connectionLimit = webServer.connectionLimit();

    if (DCOPStub::CallFailed == webServer.status())
    {
      // TODO: warn user ?
      kpfDebug << "WebServer_stub call failed" << endl;
      d->currentState.connectionLimit = Config::DefaultConnectionLimit;
      return;
    }
#endif

    d->currentState.serverName = webServer.serverName();
    if (DCOPStub::CallFailed == webServer.status())
    {
      // TODO: warn user ?
      kpfDebug << "WebServer_stub call failed" << endl;
      d->currentState.serverName = "";
      return;
    }
  

    d->currentState.followSymlinks = webServer.followSymlinks();

    if (DCOPStub::CallFailed == webServer.status())
    {
      // TODO: warn user ?
      kpfDebug << "WebServer_stub call failed" << endl;
      d->currentState.followSymlinks = Config::DefaultFollowSymlinks;
      return;
    }
  }

    void
  PropertiesDialogPlugin::getServerRef()
  {
    TQValueList<DCOPRef> serverRefList =
      d->webServerManagerInterface->serverList();

    if (DCOPStub::CallFailed == d->webServerManagerInterface->status())
    {
      // TODO: warn
      kpfDebug << "webServerManagerInterface.serverList call failed" << endl;
      return;
    }

    d->webServerRef.clear();

    TQValueList<DCOPRef>::ConstIterator it(serverRefList.begin());

    for (; it != serverRefList.end(); ++it)
    {
      DCOPRef serverRef(*it);

      WebServer_stub webServer(serverRef.app(), serverRef.object());

      if (webServer.root() == d->url.path())
      {
        d->webServerRef = serverRef;
        break;
      }
    }
  }

    bool
  PropertiesDialogPlugin::userAcceptsWarning() const
  {
    TQString noWarningKey("DoNotWarnAboutSharingDirectoriesViaHTTP");

    KConfig * config(KGlobal::config());

    if (config->readBoolEntry(noWarningKey, false))
      return true;

    return
      (
        KMessageBox::Continue
        ==
        KMessageBox::warningContinueCancel
        (
          d->configWidget,
          i18n(
            "<p>"
            "Before you share a directory, be <strong>absolutely"
            " certain</strong> that it does not contain sensitive"
            " information."
            "</p>"
            "<p>"
            "Sharing a directory makes all information"
            " in that directory <strong>and all subdirectories</strong>"
            " available to <strong>anyone</strong> who wishes to read it."
            "</p>"
            "<p>"
            "If you have a system administrator, please ask for permission"
            " before sharing a directory in this way."
            "</p>"
          ),
          i18n("Warning - Sharing Sensitive Information?"),
          i18n("&Share Directory"),
          noWarningKey,
          true
        )
        );
  }

    void
  PropertiesDialogPlugin::slotChanged()
  {
    kpfDebug << "PropertiesDialogPlugin::slotChanged" << endl;
    readSettings();
    updateWantedStateFromGUI();

    setDirty(d->currentState != d->wantedState);
    kpfDebug << "Dirty: " << isDirty() << endl;
    emit(changed());
  }

    void
  PropertiesDialogPlugin::applyChanges()
  {
    readSettings();
    updateWantedStateFromGUI();

    enum Action
    {
      None,
      Enable,
      Disable,
      Reconfigure
    };

    bool needRestart = false;

    Action action = None;

    if (!d->currentState.shared && d->wantedState.shared)
    {
//    kpfDebug << "Not shared, but want to be. Action is Enable" << endl;
      action = Enable;
    }
    else if (d->currentState.shared && !d->wantedState.shared)
    {
//    kpfDebug << "Shared, but don't want to be. Action is Disable" << endl;
      action = Disable;
    }
    else if
      (
       d->currentState.listenPort       != d->wantedState.listenPort
       ||
       d->currentState.bandwidthLimit   != d->wantedState.bandwidthLimit
       ||
//       d->currentState.connectionLimit  != d->wantedState.connectionLimit
//       ||
       d->currentState.serverName       != d->wantedState.serverName
       ||
       d->currentState.followSymlinks   != d->wantedState.followSymlinks
      )
    {
//    kpfDebug << "Config changed. Action is Reconfigure" << endl;
      action = Reconfigure;

      if (d->currentState.listenPort != d->wantedState.listenPort)
        needRestart = true;
    }

    if (None == action)
    {
//    kpfDebug << "Nothing changed. Action = None" << endl;
      return;
    }

    switch (action)
    {
      case Enable:
        {
          DCOPRef ref =
            d->webServerManagerInterface->createServer
            (
             d->url.path(),
             d->wantedState.listenPort,
             d->wantedState.bandwidthLimit,
             Config::DefaultConnectionLimit,//d->wantedState.connectionLimit,
             d->wantedState.followSymlinks,
	     d->wantedState.serverName
            );

          if (ref.isNull())
          {
            // TODO: Warn user.
            kpfDebug
              << "kpf refused to create server - warn user here !" << endl;
            break;
          }
          else
          {
            d->webServerRef = ref;
          }
        }
        break;

      case Disable:
        if (d->webServerRef.isNull())
        {
          // TODO: Warn user.
          kpfDebug << "Disable, but d->webServerRef is null" << endl;
        }
        else
        {
          d->webServerManagerInterface->disableServer(d->webServerRef);
        }
        break;

      case Reconfigure:

        if (d->webServerRef.isNull())
        {
          kpfDebug << "Need restart, but d->webServerRef is null" << endl;
        }
        else
        {
          WebServer_stub webServer
            (d->webServerRef.app(), d->webServerRef.object());

          webServer.set
            (
             d->wantedState.listenPort,
             d->wantedState.bandwidthLimit,
             Config::DefaultConnectionLimit,//d->wantedState.connectionLimit,
             d->wantedState.followSymlinks,
	     d->wantedState.serverName
            );

          if (DCOPStub::CallFailed == webServer.status())
          {
            // TODO: Warn user.
            kpfDebug << "Reconfigure failed" << endl;
          }

          if (needRestart)
          {
            webServer.restart();

            if (DCOPStub::CallFailed == webServer.status())
            {
              // TODO: Warn user.
              kpfDebug << "Restart failed" << endl;
            }
          }
        }
        break;

      default:
        kpfDebug << "Code error in KPF::PropertiesDialogPlugin." << endl;
        break;
    }
  }

    void
  PropertiesDialogPlugin::updateGUIFromCurrentState()
  {
    readSettings();

    // We don't want slotSharingToggled to be called.
    d->cb_share->blockSignals(true);
    d->cb_share->setChecked(d->currentState.shared);
    d->cb_share->blockSignals(false);

    d->sb_listenPort      ->setValue    (d->currentState.listenPort);
    d->sb_bandwidthLimit  ->setValue    (d->currentState.bandwidthLimit);
//    d->sb_connectionLimit ->setValue    (d->currentState.connectionLimit);
    d->le_serverName      ->setText     (d->currentState.serverName);
    d->cb_followSymlinks  ->setChecked  (d->currentState.followSymlinks);

    setControlsEnabled(d->currentState.shared);
  }

    void
  PropertiesDialogPlugin::updateWantedStateFromGUI()
  {
    d->wantedState.shared           = d->cb_share->isChecked();
    d->wantedState.listenPort       = d->sb_listenPort->value();
    d->wantedState.bandwidthLimit   = d->sb_bandwidthLimit->value();
//    d->wantedState.connectionLimit  = d->sb_connectionLimit->value();
    d->wantedState.serverName       = d->le_serverName->text();
    d->wantedState.followSymlinks   = d->cb_followSymlinks->isChecked();
  }

  typedef KGenericFactory<PropertiesDialogPlugin, KPropertiesDialog> PropertiesDialogPluginFactory;
}

K_EXPORT_COMPONENT_FACTORY( kpfpropertiesdialog, 
                            KPF::PropertiesDialogPluginFactory( "kpf" ) )

#include "PropertiesDialogPlugin.moc"
// vim:ts=2:sw=2:tw=78:et
