/*
 * ibb.h - Inband bytestream
 * Copyright (C) 2001, 2002  Justin Karneges
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#ifndef JABBER_IBB_H
#define JABBER_IBB_H

#include<tqobject.h>
#include<tqdom.h>
#include<tqstring.h>
#include<tqptrlist.h>
#include"bytestream.h"
#include"im.h"

namespace XMPP
{
	class Client;
	class IBBManager;

	// this is an IBB connection.  use it much like a qsocket
	class IBBConnection : public ByteStream
	{
		Q_OBJECT
  TQ_OBJECT
	public:
		enum { ErrRequest, ErrData };
		enum { Idle, Requesting, WaitingForAccept, Active };
		IBBConnection(IBBManager *);
		~IBBConnection();

		void connectToJid(const Jid &peer, const TQDomElement &comment);
		void accept();
		void close();

		int state() const;
		Jid peer() const;
		TQString streamid() const;
		TQDomElement comment() const;

		bool isOpen() const;
		void write(const TQByteArray &);
		TQByteArray read(int bytes=0);
		int bytesAvailable() const;
		int bytesToWrite() const;

	signals:
		void connected();

	private slots:
		void ibb_finished();
		void trySend();

	private:
		class Private;
		Private *d;

		void reset(bool clear=false);

		friend class IBBManager;
		void waitForAccept(const Jid &peer, const TQString &sid, const TQDomElement &comment, const TQString &iq_id);
		void takeIncomingData(const TQByteArray &, bool close);
	};

	typedef TQPtrList<IBBConnection> IBBConnectionList;
	typedef TQPtrListIterator<IBBConnection> IBBConnectionListIt;
	class IBBManager : public TQObject
	{
		Q_OBJECT
  TQ_OBJECT
	public:
		IBBManager(Client *);
		~IBBManager();

		Client *client() const;

		IBBConnection *takeIncoming();

	signals:
		void incomingReady();

	private slots:
		void ibb_incomingRequest(const Jid &from, const TQString &id, const TQDomElement &);
		void ibb_incomingData(const Jid &from, const TQString &streamid, const TQString &id, const TQByteArray &data, bool close);

	private:
		class Private;
		Private *d;

		TQString genKey() const;

		friend class IBBConnection;
		IBBConnection *findConnection(const TQString &sid, const Jid &peer="") const;
		TQString genUniqueKey() const;
		void link(IBBConnection *);
		void unlink(IBBConnection *);
		void doAccept(IBBConnection *c, const TQString &id);
		void doReject(IBBConnection *c, const TQString &id, int, const TQString &);
	};

	class JT_IBB : public Task
	{
		Q_OBJECT
  TQ_OBJECT
	public:
		enum { ModeRequest, ModeSendData };
		JT_IBB(Task *, bool serve=false);
		~JT_IBB();

		void request(const Jid &, const TQDomElement &comment);
		void sendData(const Jid &, const TQString &streamid, const TQByteArray &data, bool close);
		void respondSuccess(const Jid &, const TQString &id, const TQString &streamid);
		void respondError(const Jid &, const TQString &id, int code, const TQString &str);
		void respondAck(const Jid &to, const TQString &id);

		void onGo();
		bool take(const TQDomElement &);

		TQString streamid() const;
		Jid jid() const;
		int mode() const;

	signals:
		void incomingRequest(const Jid &from, const TQString &id, const TQDomElement &);
		void incomingData(const Jid &from, const TQString &streamid, const TQString &id, const TQByteArray &data, bool close);

	private:
		class Private;
		Private *d;
	};
}

#endif
