#include <kapplication.h>
#include <kdebug.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kstaticdeleter.h>

#include "clientiface_stub.h"
#include "networkstatuscommon.h"

#include "connectionmanager.h"

// ConnectionManager's private parts
class ConnectionManagerPrivate
{
	public:
		// this holds the currently active state
		ConnectionManager::State m_state;
		ClientIface_stub * m_stub;
		bool m_userInitiatedOnly;
};

// Connection manager itself
ConnectionManager::ConnectionManager( TQObject * parent, const char * name ) : DCOPObject( "ConnectionManager" ),TQObject( parent, name )
{
	d = new ConnectionManagerPrivate;
	
	d->m_stub = new ClientIface_stub( kapp->dcopClient(), "kded", "networkstatus" );
	
	connectDCOPSignal( "kded", "networkstatus", "statusChange(TQString,int)", "slotStatusChanged(TQString,int)", false );
	d->m_userInitiatedOnly = false;
	initialise();
}

ConnectionManager *ConnectionManager::s_self = 0L;

ConnectionManager *ConnectionManager::self()
{
	static KStaticDeleter<ConnectionManager> deleter;
	if(!s_self)
		deleter.setObject( s_self, new ConnectionManager( 0, "connection_manager" ) );
	return s_self;	
}

void ConnectionManager::initialise()
{
	// determine initial state and set the state object accordingly.
	d->m_state = Inactive;
	updatetqStatus();
}

void ConnectionManager::updatetqStatus()
{
	NetworktqStatus::EnumtqStatus daemontqStatus = (NetworktqStatus::EnumtqStatus)d->m_stub->status( TQString() );
	kdDebug() << k_funcinfo << endl;
	switch ( daemontqStatus )
	{
		case NetworktqStatus::Offline:
		case NetworktqStatus::OfflineFailed:
		case NetworktqStatus::OfflineDisconnected:
		case NetworktqStatus::ShuttingDown:
			if ( d->m_state == Online )
			{
				kdDebug() << "STATE IS PENDING" << endl;
				d->m_state = Pending;
			}
			else
			{
				kdDebug() << "STATE IS OFFLINE" << endl;
				d->m_state = Offline;
			}
			break;
		case NetworktqStatus::Establishing:
		case NetworktqStatus::Online:
			kdDebug() << "STATE IS ONLINE" << endl;
			d->m_state = Online;
			break;
		case NetworktqStatus::NoNetworks:
		case NetworktqStatus::Unreachable:
			kdDebug() << "STATE IS INACTIVE" << endl;
			d->m_state = Inactive;
			break;
	}
}

ConnectionManager::~ConnectionManager()
{
	delete d;
}

NetworktqStatus::EnumtqStatus ConnectionManager::status( const TQString & host )
{
	// need also to check that the daemon hasn't died
	updatetqStatus();
	if ( d->m_state == Pending )
		return NetworktqStatus::Offline;
	if ( d->m_state == Online )
		return NetworktqStatus::Online;
	if ( d->m_state == Offline )
		return NetworktqStatus::Offline;
	return NetworktqStatus::NoNetworks;
}

NetworktqStatus::EnumRequestResult ConnectionManager::requestConnection( TQWidget * mainWidget, const TQString & host, bool userInitiated )
{
	kdDebug() << k_funcinfo << endl;
	NetworktqStatus::EnumRequestResult result;
	// if offline and the user has previously indicated they didn't want any new connections, suppress it
	if ( d->m_state == Offline && !userInitiated && d->m_userInitiatedOnly )
		result = NetworktqStatus::UserRefused;
	// if offline, ask the user whether this connection should be allowed
	if ( d->m_state == Offline )
	{
		if ( askToConnect( mainWidget ) )
			//result = NetworktqStatus::Connected;
			result = (NetworktqStatus::EnumRequestResult)d->m_stub->request( host, userInitiated );
		else
			result = NetworktqStatus::UserRefused;
	}
	// otherwise, just ask for the connection
	else
		result = (NetworktqStatus::EnumRequestResult)d->m_stub->request( host, userInitiated );
	
	return result;
}

void ConnectionManager::relinquishConnection( const TQString & host )
{
	d->m_stub->relinquish( host );
}

void ConnectionManager::slotStatusChanged( TQString host, int status )
{
	kdDebug() << k_funcinfo << endl;
	updatetqStatus();
	// reset user initiated only flag if we are now online
	if ( d->m_state == Online )
		d->m_userInitiatedOnly = false;

	emit statusChanged( host, (NetworktqStatus::EnumtqStatus)status );
}

bool ConnectionManager::askToConnect( TQWidget * mainWidget )
{
	i18n( "A network connection was disconnected.  The application is now in offline mode.  Do you want the application to resume network operations when the network is available again?" );
	i18n( "This application is currently in offline mode.  Do you want to connect?" );
	return ( KMessageBox::questionYesNo( mainWidget,
			i18n("This application is currently in offline mode.  Do you want to connect in order to carry out this operation?"),
			i18n("Leave Offline Mode?"),
			i18n("Connect"), i18n("Stay Offline"),
			TQString::tqfromLatin1("OfflineModeAlwaysGoOnline") ) == KMessageBox::Yes );
}

#include "connectionmanager.moc"
