#include <ktextedit.h>
#include <kaction.h>
#include <kcolordialog.h>
#include <kglobalsettings.h>
#include <kfontdialog.h>
#include <kdebug.h>
#include <kconfig.h>
#include <kdeversion.h>
#include <tqapplication.h>
#include <tqclipboard.h>
#include <tqevent.h>
#include <kparts/genericfactory.h>
#include <tqrichtext_p.h>

#include "krichtexteditpart.h"
#include "krichtexteditpart.moc"
#include "kopeteprotocol.h"
#include "kopeteprefs.h"

typedef KParts::GenericFactory<KopeteRichTextEditPart> KopeteRichTextEditPartFactory;
K_EXPORT_COMPONENT_FACTORY( libkopeterichtexteditpart, KopeteRichTextEditPartFactory )

class KopeteTextEdit : public KTextEdit
{
public:
	KopeteTextEdit( TQWidget *parent ) : KTextEdit( parent ) {}
	const TQTextCursor * cursor() { return textCursor(); }
	bool event(TQEvent *event)
	{
		// don't allow TQTextEdit to override accels
		if ( event->type() == TQEvent::AccelOverride )
			return TQWidget::event(event);
		else
			return KTextEdit::event(event);
	}
};

KopeteRichTextEditPart::KopeteRichTextEditPart( TQWidget *wparent, const char *wname, TQObject*, const char*, const TQStringList& )
	: KParts::ReadOnlyPart( TQT_TQOBJECT(wparent), wname ? wname : "rich_text_part" )
{
	KopeteRichTextEditPart( wparent, wname, false );
}

KopeteRichTextEditPart::KopeteRichTextEditPart( TQWidget *parent, const char *name, int capabilities )
  : KParts::ReadOnlyPart( TQT_TQOBJECT(parent), name ? name : "rich_text_part" ),
	m_capabilities( capabilities ),
	m_richTextEnabled( true )
{
	// we need an instance
	setInstance( KopeteRichTextEditPartFactory::instance() );

	editor = new KopeteTextEdit( parent );
	editor->setReadOnly( false );

	setWidget( editor );

	m_richTextAvailable = (
		m_capabilities & Kopete::Protocol::RichFormatting ||
		m_capabilities & Kopete::Protocol::Alignment ||
		m_capabilities & Kopete::Protocol::RichFont ||
		m_capabilities & Kopete::Protocol::RichColor
	);

	createActions();

	setXMLFile( "kopeterichtexteditpartfull.rc" );
	enableRichText->setEnabled( m_richTextAvailable );
	enableRichText->setChecked( m_richTextAvailable );
	slotSetRichTextEnabled( m_richTextAvailable );

	//Set colors, font
	readConfig();
}

void KopeteRichTextEditPart::slotSetRichTextEnabled( bool enable )
{
	m_richTextEnabled = enable && m_richTextAvailable;

	if( m_richTextEnabled )
	{
		editor->setTextFormat( TQt::RichText );
	}
	else
	{
		editor->setTextFormat( TQt::PlainText );
	}

	emit toggleToolbar( buttonsEnabled() );

	// Spellchecking disabled when using rich text because the
	// text we were getting from widget was coloured HTML!
	editor->setCheckSpellingEnabled( !m_richTextEnabled );
	checkSpelling->setEnabled( !m_richTextEnabled );

	//Enable / disable buttons
	updateActions();
	enableRichText->setChecked( m_richTextEnabled );
}

void KopeteRichTextEditPart::checkToolbarEnabled()
{
	emit toggleToolbar( buttonsEnabled() );
}

void KopeteRichTextEditPart::reloadConfig()
{
	readConfig();
}

void KopeteRichTextEditPart::createActions()
{
	createActions( actionCollection() );
}

KAboutData *KopeteRichTextEditPart::createAboutData()
{
	KAboutData *aboutData = new KAboutData("kopeterichtexteditpart", I18N_NOOP("KopeteRichTextEditPart"), "0.1",
						I18N_NOOP("A simple rich text editor part for Kopete"),
						KAboutData::License_LGPL );
	aboutData->addAuthor("Richard J. Moore", 0, "rich@kde.org", "http://xmelegance.org/" );
	aboutData->addAuthor("Jason Keirstead", 0, "jason@keirstead.org", "http://www.keirstead.org/" );
	return aboutData;
}

void KopeteRichTextEditPart::createActions( KActionCollection *ac )
{
	enableRichText = new KToggleAction(i18n("Enable &Rich Text"), "pencil", 0,
				ac, "enableRichText" );
	enableRichText->setCheckedState(i18n("Disable &Rich Text"));
	connect( enableRichText, TQT_SIGNAL( toggled(bool) ),
			this, TQT_SLOT( slotSetRichTextEnabled(bool) ) );

	checkSpelling = new KAction( i18n("Check &Spelling"), "spellcheck", 0,
				TQT_TQOBJECT(editor), TQT_SLOT( checkSpelling() ), ac, "check_spelling" );

	//Fg Color
	actionFgColor = new KAction( i18n("Text &Color..."), "color_line", 0,
		this, TQT_SLOT( setFgColor() ),
		ac, "format_color" );

	//BG Color
	actionBgColor = new KAction( i18n("Background Co&lor..."), "color_fill", 0,
		this, TQT_SLOT( setBgColor() ),
		ac, "format_bgcolor" );

	//Font Family
	action_font = new KFontAction( i18n("&Font"), 0,
			ac, "format_font" );
	connect( action_font, TQT_SIGNAL( activated( const TQString & ) ),
		this, TQT_SLOT( setFont( const TQString & ) ) );

	//Font Size
	action_font_size = new KFontSizeAction( i18n("Font &Size"), 0,
			ac, "format_font_size" );
	connect( action_font_size, TQT_SIGNAL( fontSizeChanged(int) ),
		this, TQT_SLOT( setFontSize(int) ) );

	//Formatting
	action_bold = new KToggleAction( i18n("&Bold"), "text_bold", CTRL+Key_B,
			ac, "format_bold" );
	connect( action_bold, TQT_SIGNAL( toggled(bool) ),
		this, TQT_SLOT( setBold(bool) ) );

	action_italic = new KToggleAction( i18n("&Italic"), "text_italic", CTRL+Key_I,
			ac, "format_italic" );
	connect( action_italic, TQT_SIGNAL( toggled(bool) ),
		this, TQT_SLOT( setItalic(bool) ) );

	action_underline = new KToggleAction( i18n("&Underline"), "text_under", CTRL+Key_U,
				ac, "format_underline" );
	connect( action_underline, TQT_SIGNAL( toggled(bool) ),
		this, TQT_SLOT( setUnderline(bool) ) );

	connect( editor, TQT_SIGNAL( currentFontChanged( const TQFont & ) ),
		this, TQT_SLOT( updateCharFmt() ) );
	updateCharFmt();

	connect( editor, TQT_SIGNAL( currentFontChanged( const TQFont & ) ),
		this, TQT_SLOT( updateFont() ) );
	updateFont();

	//Alignment
	action_align_left = new KToggleAction( i18n("Align &Left"), "text_left", 0,
			ac, "format_align_left" );
	connect( action_align_left, TQT_SIGNAL( toggled(bool) ),
		this, TQT_SLOT( setAlignLeft(bool) ) );

	action_align_center = new KToggleAction( i18n("Align &Center"), "text_center", 0,
			ac, "format_align_center" );
	connect( action_align_center, TQT_SIGNAL( toggled(bool) ),
		this, TQT_SLOT( setAlignCenter(bool) ) );

	action_align_right = new KToggleAction( i18n("Align &Right"), "text_right", 0,
			ac, "format_align_right" );
	connect( action_align_right, TQT_SIGNAL( toggled(bool) ),
		this, TQT_SLOT( setAlignRight(bool) ) );

	action_align_justify = new KToggleAction( i18n("&Justify"), "text_block", 0,
			ac, "format_align_justify" );
	connect( action_align_justify, TQT_SIGNAL( toggled(bool) ),
		this, TQT_SLOT( setAlignJustify(bool) ) );

	action_align_left->setExclusiveGroup( "tqalignment" );
	action_align_center->setExclusiveGroup( "tqalignment" );
	action_align_right->setExclusiveGroup( "tqalignment" );
	action_align_justify->setExclusiveGroup( "tqalignment" );

	connect( editor, TQT_SIGNAL( cursorPositionChanged( int,int ) ),
		this, TQT_SLOT( updateAligment() ) );

	updateAligment();
}

void KopeteRichTextEditPart::updateActions()
{
	bool buttonsEnabled = this->buttonsEnabled();
	bool enableFgColor = m_capabilities & Kopete::Protocol::BaseFgColor || m_capabilities & Kopete::Protocol::RichFgColor;
	bool enableBGColor = m_capabilities & Kopete::Protocol::BaseBgColor || m_capabilities & Kopete::Protocol::RichBgColor;
	bool activateAlignment = buttonsEnabled && ( m_capabilities & Kopete::Protocol::Alignment );
	bool activateFont = m_capabilities & Kopete::Protocol::BaseFont || m_capabilities & Kopete::Protocol::RichFont;

	bool activateBFormat = m_capabilities & Kopete::Protocol::BaseBFormatting || m_capabilities & Kopete::Protocol::RichBFormatting;
	bool activateUFormat = m_capabilities & Kopete::Protocol::BaseUFormatting || m_capabilities & Kopete::Protocol::RichUFormatting;
	bool activateIFormat = m_capabilities & Kopete::Protocol::BaseIFormatting || m_capabilities & Kopete::Protocol::RichIFormatting;

	actionFgColor->setEnabled( buttonsEnabled && enableFgColor );
	actionBgColor->setEnabled( buttonsEnabled && enableBGColor );

	action_font->setEnabled( buttonsEnabled && activateFont );
	action_font_size->setEnabled( buttonsEnabled && activateFont );

	action_bold->setEnabled( buttonsEnabled && activateBFormat );
	action_italic->setEnabled( buttonsEnabled && activateIFormat );
	action_underline->setEnabled( buttonsEnabled && activateUFormat );

	action_align_left->setEnabled( activateAlignment );
	action_align_center->setEnabled( activateAlignment );
	action_align_right->setEnabled( activateAlignment );
	action_align_justify->setEnabled( activateAlignment );
}

void KopeteRichTextEditPart::updateCharFmt()
{
	action_bold->setChecked( editor->bold() );
	action_italic->setChecked( editor->italic() );
	action_underline->setChecked( editor->underline() );
}

void KopeteRichTextEditPart::clear()
{
	editor->setText( TQString() );
	setFont( mFont );
	setFgColor( mFgColor );

	if( m_capabilities & Kopete::Protocol::BaseBFormatting || m_capabilities & Kopete::Protocol::RichBFormatting )
	{
		editor->setBold( action_bold->isChecked() );
	}
	if( m_capabilities & Kopete::Protocol::BaseIFormatting || m_capabilities & Kopete::Protocol::RichIFormatting )
	{
		editor->setItalic( action_italic->isChecked() );
	}
	if( m_capabilities & Kopete::Protocol::BaseUFormatting || m_capabilities & Kopete::Protocol::RichUFormatting )
	{
		editor->setUnderline( action_underline->isChecked() );
	}
}

void KopeteRichTextEditPart::updateAligment()
{
	int align = editor->tqalignment();

	switch ( align )
	{
		case AlignRight:
			action_align_right->setChecked( true );
			break;
		case AlignCenter:
			action_align_center->setChecked( true );
			break;
		case AlignLeft:
			action_align_left->setChecked( true );
			break;
		case AlignJustify:
			action_align_justify->setChecked( true );
			break;
		default:
			break;
	}
}

void KopeteRichTextEditPart::updateFont()
{
	if ( editor->pointSize() > 0 )
		action_font_size->setFontSize( editor->pointSize() );
	action_font->setFont( editor->family() );
}

void KopeteRichTextEditPart::readConfig()
{
	// Don't update config untill we read whole config first
	m_configWriteLock = true;
	KConfig *config = KGlobal::config();
	config->setGroup("RichTextEditor");

	TQColor tmpColor = KGlobalSettings::textColor();
	setFgColor( config->readColorEntry("FgColor", &tmpColor ) );

	tmpColor = KGlobalSettings::baseColor();
	setBgColor( config->readColorEntry("BgColor", &tmpColor ) );

	TQFont tmpFont = KopetePrefs::prefs()->fontFace();
	setFont( config->readFontEntry("Font", &tmpFont ) );

	int tmp = KGlobalSettings::generalFont().pixelSize();
	setFontSize( config->readNumEntry( "FontSize", tmp ) );

	action_bold->setChecked( config->readBoolEntry( "FontBold" ) );
	action_italic->setChecked( config->readBoolEntry( "FontItalic" ) );
	action_underline->setChecked( config->readBoolEntry( "FontUnderline" ) );

	switch( config->readNumEntry( "EditAlignment", AlignLeft ) )
	{
		case AlignLeft:
			action_align_left->activate();
		break;
		case AlignCenter:
			action_align_center->activate();
		break;
		case AlignRight:
			action_align_right->activate();
		break;
		case AlignJustify:
			action_align_justify->activate();
		break;
	}
	m_configWriteLock = false;
}

void KopeteRichTextEditPart::writeConfig()
{
	// If true we're still reading the conf write now, so don't write.
	if( m_configWriteLock ) return;

	KConfig *config = KGlobal::config();
	config->setGroup("RichTextEditor");
	config->writeEntry("Font", mFont );
	config->writeEntry("FontSize", mFont.pointSize() );
	config->writeEntry("FontBold", mFont.bold() );
	config->writeEntry("FontItalic", mFont.italic() );
	config->writeEntry("FontUnderline", mFont.underline() );
	config->writeEntry("BgColor", mBgColor );
	config->writeEntry("FgColor", mFgColor );
	config->writeEntry("EditAlignment", editor->tqalignment() );
	config->sync();
}

void KopeteRichTextEditPart::setFgColor()
{
	TQColor col=editor->color();

	int s = KColorDialog::getColor( col, KGlobalSettings::textColor() , editor );
	if(!col.isValid())
		col= KGlobalSettings::textColor() ;
	if ( s != TQDialog::Accepted  )
		return;

	setFgColor( col );

	writeConfig();
}

void KopeteRichTextEditPart::setFgColor( const TQColor &newColor )
{
	mFgColor = newColor;

	if( !(m_capabilities & Kopete::Protocol::RichColor) )
	{
		TQPalette pal = editor->palette();
		pal.setColor(TQPalette::Active, TQColorGroup::Text, mFgColor );
		pal.setColor(TQPalette::Inactive, TQColorGroup::Text, mFgColor );

		if ( pal == TQApplication::palette( editor ) )
			editor->unsetPalette();
		else
			editor->setPalette(pal);
	}

	editor->setColor( mFgColor );
}

TQColor KopeteRichTextEditPart::fgColor()
{
	if( mFgColor == KGlobalSettings::textColor())
		return TQColor();
	return mFgColor;
}

void KopeteRichTextEditPart::setBgColor()
{
	TQColor col=mBgColor;

	int s = KColorDialog::getColor( col, KGlobalSettings::baseColor(), editor );
	if(!col.isValid())
	{
		col=KGlobalSettings::baseColor();
	}

	if ( s != TQDialog::Accepted  )
		return;

	setBgColor( col );

	writeConfig();
}

void KopeteRichTextEditPart::setBgColor( const TQColor &newColor )
{
	mBgColor = newColor;

	TQPalette pal = editor->palette();
	pal.setColor(TQPalette::Active, TQColorGroup::Base, mBgColor );
	pal.setColor(TQPalette::Inactive, TQColorGroup::Base, mBgColor );
	pal.setColor(TQPalette::Disabled, TQColorGroup::Base, mBgColor );

	if ( pal == TQApplication::palette( editor ) )
		editor->unsetPalette();
	else
		editor->setPalette(pal);
}

TQColor KopeteRichTextEditPart::bgColor()
{
	if( mBgColor == KGlobalSettings::baseColor())
		return TQColor();
	return mBgColor;
}

void KopeteRichTextEditPart::setFontSize( int size )
{
	mFont.setPointSize( size );
	if( m_capabilities & Kopete::Protocol::RichFont )
		editor->setPointSize( size );
	else if( m_capabilities & Kopete::Protocol::BaseFont)
		editor->setFont( mFont );
	writeConfig();
}

void KopeteRichTextEditPart::setFont()
{
	KFontDialog::getFont(mFont, false, editor);
	setFont(mFont);
	writeConfig();
}

void KopeteRichTextEditPart::setFont( const TQFont &newFont )
{
	mFont = newFont;
	editor->setFont(mFont);
	updateFont();
}

void KopeteRichTextEditPart::setFont( const TQString &newFont )
{
	mFont.setFamily( newFont );
	if( m_capabilities & Kopete::Protocol::RichFont)
		editor->setFamily( newFont );
	else if( m_capabilities & Kopete::Protocol::BaseFont)
		editor->setFont( mFont );
	updateFont();
	writeConfig();
}


void KopeteRichTextEditPart::setBold( bool b )
{
	mFont.setBold(b);
	if( m_capabilities & Kopete::Protocol::RichBFormatting || m_capabilities & Kopete::Protocol::BaseBFormatting )
	{
		if( m_richTextEnabled )
			editor->setBold(b);
		else
			editor->setFont(mFont);
	}
	writeConfig();
}

void KopeteRichTextEditPart::setItalic( bool b )
{
	mFont.setItalic( b );
	if( m_capabilities & Kopete::Protocol::RichIFormatting ||  m_capabilities & Kopete::Protocol::BaseIFormatting )
	{
		if(m_richTextEnabled)
			editor->setItalic(b);
		else
			editor->setFont(mFont);
	}
	writeConfig();
}

void KopeteRichTextEditPart::setUnderline( bool b )
{
	mFont.setUnderline( b );
	if( m_capabilities & Kopete::Protocol::RichUFormatting ||  m_capabilities & Kopete::Protocol::BaseUFormatting  )
	{
		if(m_richTextEnabled)
			editor->setUnderline(b);
		else
			editor->setFont(mFont);
	}
	writeConfig();
}


void KopeteRichTextEditPart::setAlignLeft( bool yes )
{
	if ( yes )
		editor->tqsetAlignment( AlignLeft );
	writeConfig();
}

void KopeteRichTextEditPart::setAlignRight( bool yes )
{
	if ( yes )
		editor->tqsetAlignment( AlignRight );
	writeConfig();
}

void KopeteRichTextEditPart::setAlignCenter( bool yes )
{
	if ( yes )
		editor->tqsetAlignment( AlignCenter );
	writeConfig();
}

void KopeteRichTextEditPart::setAlignJustify( bool yes )
{
	if ( yes )
		editor->tqsetAlignment( AlignJustify );
	writeConfig();
}

TQString KopeteRichTextEditPart::text( TQt::TextFormat fmt ) const
{
	if( fmt == editor->textFormat() || fmt != TQt::PlainText )
		return editor->text();
	else
		return editor->cursor()->document()->plainText();
}

