 /*
    kopetechatwindowstyle.cpp - A Chat Window Style.

    Copyright (c) 2005      by Michaël Larouche     <michael.larouche@kdemail.net>

    Kopete    (c) 2002-2005 by the Kopete developers <kopete-devel@kde.org>

    *************************************************************************
    *                                                                       *
    * This program is free software; you can redistribute it and/or modify  *
    * it under the terms of the GNU General Public License as published by  *
    * the Free Software Foundation; either version 2 of the License, or     *
    * (at your option) any later version.                                   *
    *                                                                       *
    *************************************************************************
*/

#include "kopetechatwindowstyle.h"

// TQt includes
#include <tqfile.h>
#include <tqdir.h>
#include <tqstringlist.h>
#include <tqtextstream.h>

// KDE includes
#include <kdebug.h>


class ChatWindowStyle::Private
{
public:
	TQString stylePath;
	StyleVariants variantsList;
	TQString baseHref;
	TQString currentVariantPath;

	TQString headerHtml;
	TQString footerHtml;
	TQString incomingHtml;
	TQString nextIncomingHtml;
	TQString outgoingHtml;
	TQString nextOutgoingHtml;
	TQString statusHtml;
	TQString actionIncomingHtml;
	TQString actionOutgoingHtml;
};

ChatWindowStyle::ChatWindowStyle(const TQString &stylePath, int styleBuildMode)
	: d(new Private)
{
	init(stylePath, styleBuildMode);
}

ChatWindowStyle::ChatWindowStyle(const TQString &stylePath, const TQString &variantPath, int styleBuildMode)
	: d(new Private)
{
	d->currentVariantPath = variantPath;
	init(stylePath, styleBuildMode);
}

void ChatWindowStyle::init(const TQString &stylePath, int styleBuildMode)
{
	d->stylePath = stylePath;
	d->baseHref = stylePath + TQString::fromUtf8("/Contents/Resources/");
	readStyleFiles();
	if(styleBuildMode & StyleBuildNormal)
	{
		listVariants();
	}
}

ChatWindowStyle::~ChatWindowStyle()
{
	kdDebug(14000) << k_funcinfo << endl;
	delete d;
}

ChatWindowStyle::StyleVariants ChatWindowStyle::getVariants()
{
	// If the variantList is empty, list available variants.
	if( d->variantsList.isEmpty() )
	{
		listVariants();
	}
	return d->variantsList;
}

TQString ChatWindowStyle::getStylePath() const
{
	return d->stylePath;
}

TQString ChatWindowStyle::getStyleBaseHref() const
{
	return d->baseHref;
}

TQString ChatWindowStyle::getHeaderHtml() const
{
	return d->headerHtml;
}

TQString ChatWindowStyle::getFooterHtml() const
{
	return d->footerHtml;
}

TQString ChatWindowStyle::getIncomingHtml() const
{
	return d->incomingHtml;
}

TQString ChatWindowStyle::getNextIncomingHtml() const
{
	return d->nextIncomingHtml;
}

TQString ChatWindowStyle::getOutgoingHtml() const
{
	return d->outgoingHtml;
}

TQString ChatWindowStyle::getNextOutgoingHtml() const
{
	return d->nextOutgoingHtml;
}

TQString ChatWindowStyle::getStatusHtml() const
{
	return d->statusHtml;
}

TQString ChatWindowStyle::getActionIncomingHtml() const
{
	return d->actionIncomingHtml;	
}

TQString ChatWindowStyle::getActionOutgoingHtml() const
{
	return d->actionOutgoingHtml;
}

bool ChatWindowStyle::hasActionTemplate() const
{
	return ( !d->actionIncomingHtml.isEmpty() && !d->actionOutgoingHtml.isEmpty() );
}

void ChatWindowStyle::listVariants()
{
	TQString variantDirPath = d->baseHref + TQString::fromUtf8("Variants/");
	TQDir variantDir(variantDirPath);

	TQStringList variantList = variantDir.entryList("*.css");
	TQStringList::ConstIterator it, itEnd = variantList.constEnd();
	for(it = variantList.constBegin(); it != itEnd; ++it)
	{
		TQString variantName = *it, variantPath;
		// Retrieve only the file name.
		variantName = variantName.left(variantName.findRev("."));
		// variantPath is relative to baseHref.
		variantPath = TQString("Variants/%1").tqarg(*it);
		d->variantsList.insert(variantName, variantPath);
	}
}

void ChatWindowStyle::readStyleFiles()
{
	TQString headerFile = d->baseHref + TQString("Header.html");
	TQString footerFile = d->baseHref + TQString("Footer.html");
	TQString incomingFile = d->baseHref + TQString("Incoming/Content.html");
	TQString nextIncomingFile = d->baseHref + TQString("Incoming/NextContent.html");
	TQString outgoingFile = d->baseHref + TQString("Outgoing/Content.html");
	TQString nextOutgoingFile = d->baseHref + TQString("Outgoing/NextContent.html");
	TQString statusFile = d->baseHref + TQString("Status.html");
	TQString actionIncomingFile = d->baseHref + TQString("Incoming/Action.html");
	TQString actionOutgoingFile = d->baseHref + TQString("Outgoing/Action.html");

	TQFile fileAccess;
	// First load header file.
	if( TQFile::exists(headerFile) )
	{
		fileAccess.setName(headerFile);
		fileAccess.open(IO_ReadOnly);
		TQTextStream headerStream(&fileAccess);
		headerStream.setEncoding(TQTextStream::UnicodeUTF8);
		d->headerHtml = headerStream.read();
		kdDebug(14000) << k_funcinfo << "Header HTML: " << d->headerHtml << endl;
		fileAccess.close();
	}
	// Load Footer file
	if( TQFile::exists(footerFile) )
	{
		fileAccess.setName(footerFile);
		fileAccess.open(IO_ReadOnly);
		TQTextStream headerStream(&fileAccess);
		headerStream.setEncoding(TQTextStream::UnicodeUTF8);
		d->footerHtml = headerStream.read();
		kdDebug(14000) << k_funcinfo << "Footer HTML: " << d->footerHtml << endl;
		fileAccess.close();
	}
	// Load incoming file
	if( TQFile::exists(incomingFile) )
	{
		fileAccess.setName(incomingFile);
		fileAccess.open(IO_ReadOnly);
		TQTextStream headerStream(&fileAccess);
		headerStream.setEncoding(TQTextStream::UnicodeUTF8);
		d->incomingHtml = headerStream.read();
		kdDebug(14000) << k_funcinfo << "Incoming HTML: " << d->incomingHtml << endl;
		fileAccess.close();
	}
	// Load next Incoming file
	if( TQFile::exists(nextIncomingFile) )
	{
		fileAccess.setName(nextIncomingFile);
		fileAccess.open(IO_ReadOnly);
		TQTextStream headerStream(&fileAccess);
		headerStream.setEncoding(TQTextStream::UnicodeUTF8);
		d->nextIncomingHtml = headerStream.read();
		kdDebug(14000) << k_funcinfo << "NextIncoming HTML: " << d->nextIncomingHtml << endl;
		fileAccess.close();
	}
	// Load outgoing file
	if( TQFile::exists(outgoingFile) )
	{
		fileAccess.setName(outgoingFile);
		fileAccess.open(IO_ReadOnly);
		TQTextStream headerStream(&fileAccess);
		headerStream.setEncoding(TQTextStream::UnicodeUTF8);
		d->outgoingHtml = headerStream.read();
		kdDebug(14000) << k_funcinfo << "Outgoing HTML: " << d->outgoingHtml << endl;
		fileAccess.close();
	}
	// Load next outgoing file
	if( TQFile::exists(nextOutgoingFile) )
	{
		fileAccess.setName(nextOutgoingFile);
		fileAccess.open(IO_ReadOnly);
		TQTextStream headerStream(&fileAccess);
		headerStream.setEncoding(TQTextStream::UnicodeUTF8);
		d->nextOutgoingHtml = headerStream.read();
		kdDebug(14000) << k_funcinfo << "NextOutgoing HTML: " << d->nextOutgoingHtml << endl;
		fileAccess.close();
	}
	// Load status file
	if( TQFile::exists(statusFile) )
	{
		fileAccess.setName(statusFile);
		fileAccess.open(IO_ReadOnly);
		TQTextStream headerStream(&fileAccess);
		headerStream.setEncoding(TQTextStream::UnicodeUTF8);
		d->statusHtml = headerStream.read();
		kdDebug(14000) << k_funcinfo << "Status HTML: " << d->statusHtml << endl;
		fileAccess.close();
	}
	
	// Load Action Incoming file
	if( TQFile::exists(actionIncomingFile) )
	{
		fileAccess.setName(actionIncomingFile);
		fileAccess.open(IO_ReadOnly);
		TQTextStream headerStream(&fileAccess);
		headerStream.setEncoding(TQTextStream::UnicodeUTF8);
		d->actionIncomingHtml = headerStream.read();
		kdDebug(14000) << k_funcinfo << "ActionIncoming HTML: " << d->actionIncomingHtml << endl;
		fileAccess.close();
	}
	// Load Action Outgoing file
	if( TQFile::exists(actionOutgoingFile) )
	{
		fileAccess.setName(actionOutgoingFile);
		fileAccess.open(IO_ReadOnly);
		TQTextStream headerStream(&fileAccess);
		headerStream.setEncoding(TQTextStream::UnicodeUTF8);
		d->actionOutgoingHtml = headerStream.read();
		kdDebug(14000) << k_funcinfo << "ActionOutgoing HTML: " << d->actionOutgoingHtml << endl;
		fileAccess.close();
	}
}

void ChatWindowStyle::reload()
{
	d->variantsList.clear();
	readStyleFiles();
	listVariants();
}
