/***************************************************************************
                          smbpasswdfile.cpp  -  description
                             -------------------
    begin                : Tue June 6 2002
    copyright            : (C) 2002 by Jan Schfer
    email                : janschaefer@users.sourceforge.net
 ***************************************************************************/

/******************************************************************************
 *                                                                            *
 *  This file is part of KSambaPlugin.                                        *
 *                                                                            *
 *  KSambaPlugin is free software; you can redistribute it and/or modify      *
 *  it under the terms of the GNU General Public License as published by      *
 *  the Free Software Foundation; either version 2 of the License, or         *
 *  (at your option) any later version.                                       *
 *                                                                            *
 *  KSambaPlugin is distributed in the hope that it will be useful,           *
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *
 *  GNU General Public License for more details.                              *
 *                                                                            *
 *  You should have received a copy of the GNU General Public License         *
 *  along with KSambaPlugin; if not, write to the Free Software               *
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301 USA  *
 *                                                                            *
 ******************************************************************************/

#include <tqstring.h>
#include <tqfile.h>
#include <tqtextstream.h>
#include <tqstringlist.h>

#include <kurl.h>
#include <kdebug.h>
#include <kpassdlg.h>
#include <klocale.h>
#include <kprocess.h>

#include "sambafile.h"
#include "smbpasswdfile.h"
#include "passwd.h"


TQStringList SambaUserList::getUserNames()
{
  TQStringList list;

  SambaUser *user;
  for ( user = first(); user; user = next() )
  {
     list.append(user->name);
  }

  return list;
}


SmbPasswdFile::SmbPasswdFile() {
}


SmbPasswdFile::SmbPasswdFile(const KURL & url)
{
  setUrl(url);
}

SmbPasswdFile::~SmbPasswdFile()
{
}

void SmbPasswdFile::setUrl(const KURL & url) {
  _url = url;
}

/**
 * Returns a list of all users stored in
 * the smbpasswd file
 **/
SambaUserList SmbPasswdFile::getSambaUserList()
{
  TQFile f(_url.path());

  SambaUserList list;

  if ( f.open(IO_ReadOnly) )
  {
    TQTextStream t( &f );
    TQString s;
    while ( !t.eof() )
    {
      s = t.readLine().stripWhiteSpace();

      // Ignore comments
      if (s.left(1)=="#")
         continue;

      TQStringList l = TQStringList::split(":",s);

      SambaUser* user = new SambaUser(l[0],l[1].toInt());
      user->gid = getUserGID(l[0]);
      user->isUserAccount = l[4].contains('U');
      user->hasNoPassword = l[4].contains('N');;
      user->isDisabled = l[4].contains('D');;
      user->isWorkstationTrustAccount = l[4].contains('W');;
      list.append(user);
    }
    f.close();
  }

  return list;
}

bool SmbPasswdFile::executeSmbpasswd(const TQStringList & args) {
  KProcess p;
  p << "smbpasswd" << args;

  connect( &p, TQT_SIGNAL(receivedStdout(KProcess*,char*,int)),
           this, TQT_SLOT(smbpasswdStdOutReceived(KProcess*,char*,int)));

  _smbpasswdOutput = "";

  bool result = p.start(KProcess::Block,KProcess::Stdout);

  if (result)
  {
    kdDebug(5009) << _smbpasswdOutput << endl;
  }

  return result;
}

/**
 * Tries to add the passed user to the smbpasswd file
 * returns true if successful otherwise false
 **/
bool SmbPasswdFile::addUser(const SambaUser & user,const TQString & password)
{
  KProcess p;
  p << "smbpasswd" << "-a" << user.name;

  p << password;

  connect( &p, TQT_SIGNAL(receivedStdout(KProcess*,char*,int)),
           this, TQT_SLOT(smbpasswdStdOutReceived(KProcess*,char*,int)));

  _smbpasswdOutput = "";

  bool result = p.start(KProcess::Block,KProcess::Stdout);

  if (result)
  {
    kdDebug(5009) << _smbpasswdOutput << endl;
  }

  return result;
}

/**
 * Tries to remove the passed user from the smbpasswd file
 * returns true if successful otherwise false
 **/
bool SmbPasswdFile::removeUser(const SambaUser & user)
{
  TQStringList l;
  l << "-x" << user.name;
  return executeSmbpasswd(l);
}

bool SmbPasswdFile::changePassword(const SambaUser & user, const TQString & newPassword)
{
  return addUser(user,newPassword);
}


void SmbPasswdFile::smbpasswdStdOutReceived(KProcess *, char *buffer, int buflen)
{
  _smbpasswdOutput+=TQString::tqfromLatin1(buffer,buflen);
}


/**
 * Returns the Url of the smbpasswd file
 * specified in the [global] section of
 * the smb.conf file.
 * If there is no entry in the [global] section
 * it tries to guess the location.
 **/
KURL SmbPasswdFile::getUrlFromSambaFile(const SambaFile * /*file*/)
{
  kdWarning() << "SmbPasswdFile::getUrlFromSambaFile unimplemeneted!" << endl;
  return KURL("");
}

bool SmbPasswdFile::enableUser(const SambaUser & user) {
  TQStringList l;
  l << "-e" << user.name;
  return executeSmbpasswd(l);
}

bool SmbPasswdFile::disableUser(const SambaUser & user) {
  TQStringList l;
  l << "-d" << user.name;
  return executeSmbpasswd(l);
}

bool SmbPasswdFile::setNoPassword(const SambaUser & user) {
  TQStringList l;
  l << "-n" << user.name;
  return executeSmbpasswd(l);
}

bool SmbPasswdFile::setMachineTrustAccount(const SambaUser & user) {
  TQStringList l;
  l << "-m" << user.name;
  return executeSmbpasswd(l);
}

bool SmbPasswdFile::joinADomain(const TQString & domain, const TQString & server,
      const TQString & user, const TQString & password) {
  TQStringList l;
  l << "-j" << domain;
  l << "-r" << server;
  l << "-U" << user << "%" << password;
  return executeSmbpasswd(l);
}


#include "smbpasswdfile.moc"
