/* $Id: service.h 1237912 2011-06-22 00:30:31Z tpearson $ */
#ifndef _RSS_SERVICE
#define _RSS_SERVICE

/***************************************************************************
                          service.h  -  A DCOP Service to provide RSS data
                             -------------------
    begin                : Saturday 15 February 2003
    copyright            : (C) 2003 by Ian Reinhart Geiser
    email                : geiseri@kde.org
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <dcopobject.h>
#include <dcopref.h>
#include <tqdict.h>
#include <tqptrlist.h>
#include <tqstringlist.h>
#include <tqstring.h>
#include <tqdatetime.h>
#include <tqpixmap.h>
#include <librss/global.h>
#include <librss/loader.h>
#include <librss/document.h>
#include <librss/article.h>
#include <librss/image.h>
/**
* This is a DCOP Service do not include this header in anything
*
**/
using namespace RSS;

class RSSDocument;
class RSSArticle;

class RSSService : public DCOPObject
{
	K_DCOP

	private:

		TQDict<RSSDocument> m_list;

	public:
		RSSService();
		~RSSService();
		void saveLinks();
		void loadLinks();


	k_dcop_signals:
		/**
		* Emmitted when a new document has been added.  You can then
		* use document(TQString) to get the dcop ref for the object.
		* Note: this document may or may not be valid at this
		* point so you should connect your dcop signals and then
		* do a documentValid() on the dcop ref to make sure of its
		* state.
		**/

		void added(TQString);
		/**
		* Emmitted when the document has been removed.
		* note at this point the DCOPRef for this object is
		* invalid and you will cannot access it any longer.
		* When in doubt call a refresh on it, since if its in the
		* process of loading the document call will be safely ignored
		* and you will be notified of the updates.
		**/
		void removed(TQString);
	k_dcop:
		/**
		* Add a new rdf file resource.  This will return a dcop reference to the resource.  If its a new
		* one it will be added otherwise an existing resource reference will be returned.
		* once this reference has been returned you may connect dcop signals and then call
		* refresh on the RSSDocument.  The document will not be updated until refresh is called.
		**/
		DCOPRef add(TQString url);
		/**
		* Return a list of current rss documents
		**/
		TQStringList list();
		/**
		* Remove an rss document resource.  NOTE: Be aware that others may be using this
		* resource and if you remove it they may break.  Likewise be aware that someone may
		* decide to remove your resource on you so you should always check to see if the resource
		* is valid before you access it.
		**/
		void remove(TQString url);
		/**
		* Return the reference to a requested resource.  If this resource is not present a null dcopref is
		* returned.
		**/
		DCOPRef document(TQString url);
		/**
		* Exit the RSSService.  This will clean everything up and exit.
		**/
		void exit();
};

class RSSDocument :  public TQObject, public DCOPObject
{
	Q_OBJECT
//	TQ_OBJECT
	K_DCOP

	private:
		bool m_isLoading;
		TQString m_Url;
		Document *m_Doc;
		TQPixmap m_pix;
		TQPtrList<RSSArticle> m_list;
		TQMap<TQString,int> m_state;
		TQDateTime m_Timeout;
		int m_maxAge;

	private slots:
		void pixmapLoaded(const TQPixmap&);
		void loadingComplete(Loader *, Document, tqStatus);

	public:
		RSSDocument(const TQString& url);
		~RSSDocument();

	k_dcop_signals:
		/**
		* The pixmap is currently loading
		**/
		void pixmapUpdating(DCOPRef);
		/**
		*  The pixmap is ready for viewing
		*  you can then use dcopref->call("pixmap()"); to return it.
		*
		**/
		void pixmapUpdated(DCOPRef);
		/**
		* The document is currently updating
		**/
		void documentUpdating(DCOPRef);
		/**
		* The document is ready for viewing
		* you can then use dcopref->call() to access its data
		**/
		void documentUpdated(DCOPRef);
		/**
		* The document failed to update, with and error...
		* 1 - RSS Parse Error
		* 2 - Could not access file
		* 3 - Unknown error.
		**/
		void documentUpdateError(DCOPRef, int);

	k_dcop:
		/**
		* Return the webmaster information from the RSS::Document
		**/
		TQString webMaster();
		/**
		* Return the manageing editor from the RSS::Document
		**/
		TQString managingEditor();
		/**
		* Returns the rating of the RSS::Document
		**/
		TQString rating();
		/**
		* Returns the last build date from the RSS::Document
		**/
		TQDateTime lastBuildDate();
		/**
		* Returns the publication date from the RSS::Document
		**/
		TQDateTime pubDate();
		/**
		* Returns the copyright information from the RSS::Document
		**/
		TQString copyright();
		/**
		* Returns a list of article titles
		**/
		TQStringList articles();
		/**
		* Returns the number of articles
		**/
		int count();
		/**
		* Returns a dcop reference to the article from the index
		**/
		DCOPRef article(int idx);
		/**
		* Returns the link from the  RSS::Document
		**/
		TQString link();
		/**
		* Returns the description from the RSS::Document
		**/
		TQString description();
		/**
		* Returns the title from the RSS::Document
		**/
		TQString title();
		/**
		*  Returns the text version from the RSS::Document
		**/
		TQString verbVersion();
		/**
		*  Returns the url for the pixmap from the RSS::Document
		**/
		TQString pixmapURL();
		/**
		* Returns the actual pixmap from the RSS::Document's RSS::Image
		**/
		TQPixmap pixmap();
		/**
		* Returns if the RSSDocument contains a valid RSS::Document yet.
		**/
		bool documentValid();
		/**
		* Returns if the RSSDocument contains a valid RSS::Image
		**/
		bool pixmapValid();
		/**
		* Refresh the current RSS::Document.
		* This must be called before the document is valid.
		**/
		void refresh();

		/**
		* Return the maximum age of the RSS document (Default is 60 minutes)
		**/
		int maxAge();

		/**
		* Set the maximum age of the RSS document.
		**/
		void setMaxAge(int minutes);
		
		/**
		 * Returns the state of the article
		 * 0 - not present (deleted from the rss service)
		 * 1 - new
		 * 2 - unread
		 * 3 - read
		 */
		int state( const TQString &title) const;
		
		/**
		 * Set the article state
		 */
		void setState( const TQString &title, int s );
		
		/**
		 * Convience method that will set a title to read.
		 */
		void read( const TQString &title);
};

class RSSArticle : public DCOPObject
{
	K_DCOP

	private:
		Article *m_Art;

	public:
		RSSArticle(Article *art);
		~RSSArticle();

	k_dcop:
		/**
		* Return the articles title
		**/
		TQString title();
		/**
		* Return the articles description
		**/
		TQString description();
		/**
		* Return the link to the article
		**/
		TQString link();
};
#endif
