/* This file is part of the KDE libraries
    Copyright (C) 2001 Carsten Pfeiffer <pfeiffer@kde.org>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/

#ifndef KSORTABLEVALUELIST_H
#define KSORTABLEVALUELIST_H

#include <tqpair.h>
#include <tqvaluelist.h>
#include "kdelibs_export.h"

/**
 * KSortableItem is a TQPair that provides several operators
 * for sorting.
 * @see KSortableValueList
 */
template<class T, class Key = int> class KSortableItem : public TQPair<Key,T>
{
public:
    /**
     * Creates a new KSortableItem with the given values.
     * @param i the first value
     * @param t the second value
     */
    KSortableItem( Key i, const T& t ) : TQPair<Key, T>( i, t ) {}
    /**
     * Creates a new KSortableItem that copies another one.
     * @param rhs the other item to copy
     */
    KSortableItem( const KSortableItem<T, Key> &rhs )
        : TQPair<Key,T>( rhs.first, rhs.second ) {}

    /**
     * Creates a new KSortableItem with uninitialized values.
     */
    KSortableItem() {}

    /**
     * Assignment operator, just copies the item.
     */
    KSortableItem<T, Key> &operator=( const KSortableItem<T, Key>& i ) {
        this->first  = i.first;
        this->second = i.second;
        return *this;
    }

    // operators for sorting
    /**
     * Compares the two items. This implementation only compares
     * the first value.
     */
    bool operator> ( const KSortableItem<T, Key>& i2 ) const {
        return (i2.first < this->first);
    }
    /**
     * Compares the two items. This implementation only compares
     * the first value.
     */
    bool operator< ( const KSortableItem<T, Key>& i2 ) const {
        return (this->first < i2.first);
    }
    /**
     * Compares the two items. This implementation only compares
     * the first value.
     */
    bool operator>= ( const KSortableItem<T, Key>& i2 ) const {
        return (this->first >= i2.first);
    }
    /**
     * Compares the two items. This implementation only compares
     * the first value.
     */
    bool operator<= ( const KSortableItem<T, Key>& i2 ) const {
        return !(i2.first < this->first);
    }
    /**
     * Compares the two items. This implementation only compares
     * the first value.
     */
    bool operator== ( const KSortableItem<T, Key>& i2 ) const {
        return (this->first == i2.first);
    }
    /**
     * Compares the two items. This implementation only compares
     * the first value.
     */
    bool operator!= ( const KSortableItem<T, Key>& i2 ) const {
        return (this->first != i2.first);
    }

    /**
     * @return the second value
     */
    T& value() { return this->second; }

    /**
     * Returns the second value.
     */
    const T& value() const { return this->second; }

    /**
     * @return the first value.
     */
    Key index() const { return this->first; }
};


/**
 * KSortableValueList is a special TQValueList for
 * KSortableItem. It includes convenience operators
 * to get the first value of the KSortableItem and a method
 * to sort all items.
 */
template <class T, class Key = int>
class KSortableValueList : public TQValueList<KSortableItem<T, Key> >
{
public:
    /**
     * Insert a KSortableItem with the given values.
     * @param i the first value
     * @param t the second value
     */
    void insert( Key i, const T& t ) {
        TQValueList<KSortableItem<T, Key> >::append( KSortableItem<T, Key>( i, t ) );
    }
    // add more as you please...

    /**
     * Returns the first value of the KSortableItem at the given position.
     * @return the first value of the KSortableItem
     */
    T& operator[]( Key i ) {
        return TQValueList<KSortableItem<T, Key> >::operator[]( i ).value();
    }

    /**
     * Returns the first value of the KSortableItem at the given position.
     * @return the first value of the KSortableItem
     */
    const T& operator[]( Key i ) const {
        return TQValueList<KSortableItem<T, Key> >::operator[]( i ).value();
    }

    /**
     * Sorts the KSortableItems.
     */
    void sort() {
        qHeapSort( *this );
    }
};

// template <class T> class KSortableValueListIterator : public TQValueListIterator<KSortableItem<T>  >
// {
// };

#endif // KSORTABLEVALUELIST_H
