/**********************************************************************
** Copyright (C) 2000 Trolltech AS.  All rights reserved.
**
** This file is part of TQt Designer.
**
** This file may be distributed and/or modified under the terms of the
** GNU General Public License version 2 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
** See http://www.trolltech.com/gpl/ for GPL licensing information.
**
** Contact info@trolltech.com if any conditions of this licensing are
** not clear to you.
**
**********************************************************************/

#if !defined(UIC)
#include "pixmapchooser.h"
#endif
#ifndef KOMMANDER
#include "widgetinterface.h"
#endif
#include "widgetdatabase.h"

#include <tqapplication.h>
//#define NO_STATIC_COLORS
#include "globaldefs.h"
#include <tqstrlist.h>
#include <tqdict.h>
#include <tqfile.h>
#include <tqtextstream.h>
#include <tqcleanuphandler.h>

#include <tqfeatures.h>

#include <stdlib.h>

#include <kommanderfactory.h>
#include <klocale.h>

const int dbsize = 300;
const int dbcustom = 200;
const int dbdictsize = 211;
static WidgetDatabaseRecord* widget_db[ dbsize ];
static TQDict<int> *className2Id = 0;
static int dbcount  = 0;
static int dbcustomcount = 200;
static TQStrList *wGroups;
static TQStrList *invisibleGroups;
static bool whatsThisLoaded = false;
#ifndef KOMMANDER
static TQPluginManager<WidgetInterface> *widgetPluginManager = 0;
#endif
static bool plugins_set_up = false;
static bool was_in_setup = false;

#ifndef KOMMANDER
TQCleanupHandler<TQPluginManager<WidgetInterface> > cleanup_manager;
#endif

WidgetDatabaseRecord::WidgetDatabaseRecord()
{
    isForm = false;
    isContainer = false;
    iconSet = 0;
    nameCounter = 0;
}

WidgetDatabaseRecord::~WidgetDatabaseRecord()
{
    delete iconSet;
}


/*!
  \class WidgetDatabase widgetdatabase.h
  \brief The WidgetDatabase class holds information about widgets

  The WidgetDatabase holds information about widgets like toolTip(),
  iconName(), ... It works Id-based, so all access functions take the
  widget id as parameter. To get the id for a widget (classname), use
  idFromClassName().

  All access functions are static.  Having multiple widgetdatabases in
  one application doesn't make sense anyway and so you don't need more
  than an instance of the widgetdatabase.

  For creating widgets, layouts, etc. see WidgetFactory.
*/

/*!
  Creates widget database. Does nothing.
*/

WidgetDatabase::WidgetDatabase()
{
}

/*!  Sets up the widget database. If the static widgetdatabase already
  exists, the functions returns immediately.
*/

void WidgetDatabase::setupDataBase( int id )
{
  was_in_setup = true;
#ifndef UIC
  Q_UNUSED(id) if (dbcount)
    return;
#else
  if (dbcount && id != -2)
    return;
  if (dbcount && !plugins_set_up)
  {
    setupPlugins();
    return;
  }
  if (dbcount && plugins_set_up)
    return;
#endif

  wGroups = new TQStrList;
  invisibleGroups = new TQStrList;
  invisibleGroups->append("Forms");
  invisibleGroups->append("Temp");
  className2Id = new TQDict < int >(dbdictsize);
  className2Id->setAutoDelete(true);

  WidgetDatabaseRecord *r = 0;

  /* Designer widgets */
  r = new WidgetDatabaseRecord;
  r->iconName = "pushbutton.xpm";
  r->name = TQPUSHBUTTON_OBJECT_NAME_STRING;
  r->group = widgetGroup("Buttons");
  r->toolTip = i18n("Push Button");
  append(r);

  r = new WidgetDatabaseRecord;
  r->iconName = "toolbutton.xpm";
  r->name = TQTOOLBUTTON_OBJECT_NAME_STRING;
  r->group = widgetGroup("Buttons");
  r->toolTip = i18n("Tool Button");
  append(r);

  r = new WidgetDatabaseRecord;
  r->iconName = "radiobutton.xpm";
  r->name = TQRADIOBUTTON_OBJECT_NAME_STRING;
  r->group = widgetGroup("Buttons");
  r->toolTip = i18n("Radio Button");
  append(r);

  r = new WidgetDatabaseRecord;
  r->iconName = "checkbox.xpm";
  r->name = TQCHECKBOX_OBJECT_NAME_STRING;
  r->group = widgetGroup("Buttons");
  r->toolTip = i18n("Check Box");
  append(r);

  r = new WidgetDatabaseRecord;
  r->iconName = "groupbox.xpm";
  r->name = TQGROUPBOX_OBJECT_NAME_STRING;
  r->group = widgetGroup("Containers");
  r->toolTip = i18n("Group Box");
  r->isContainer = true;
  append(r);

  r = new WidgetDatabaseRecord;
  r->iconName = "buttongroup.xpm";
  r->name = TQBUTTONGROUP_OBJECT_NAME_STRING;
  r->group = widgetGroup("Containers");
  r->toolTip = i18n("Button Group");
  r->isContainer = true;
  append(r);

  r = new WidgetDatabaseRecord;
  r->iconName = "frame.xpm";
  r->name = TQFRAME_OBJECT_NAME_STRING;
  r->group = widgetGroup("Containers");
  r->toolTip = i18n("Frame");
  r->isContainer = true;
  append(r);

  r = new WidgetDatabaseRecord;
  r->iconName = "tabwidget.xpm";
  r->name = TQTABWIDGET_OBJECT_NAME_STRING;
  r->group = widgetGroup("Containers");
  r->toolTip = i18n("Tabwidget");
  r->isContainer = true;
  append(r);

  r = new WidgetDatabaseRecord;
  r->iconName = "listbox.xpm";
  r->name = TQLISTBOX_OBJECT_NAME_STRING;
  r->group = widgetGroup("Views");
  r->toolTip = i18n("List Box");
  append(r);

  r = new WidgetDatabaseRecord;
  r->iconName = "listview.xpm";
  r->name = TQLISTVIEW_OBJECT_NAME_STRING;
  r->group = widgetGroup("Views");
  r->toolTip = i18n("List View");
  append(r);

#if !defined(TQT_NO_ICONVIEW) || defined(UIC)
  r = new WidgetDatabaseRecord;
  r->iconName = "iconview.xpm";
  r->name = TQICONVIEW_OBJECT_NAME_STRING;
  r->group = widgetGroup("Views");
  r->toolTip = i18n("Icon View");
  append(r);
#endif

#if !defined(TQT_NO_TABLE)
  r = new WidgetDatabaseRecord;
  r->iconName = "table.xpm";
  r->name = TQTABLE_OBJECT_NAME_STRING;
  r->group = widgetGroup("Views");
  r->toolTip = i18n("Table");
  append(r);
#endif

#if !defined(TQT_NO_SQL)
  r = new WidgetDatabaseRecord;
  r->iconName = "datatable.xpm";
  r->includeFile = "tqdatatable.h";
  r->name = TQDATATABLE_OBJECT_NAME_STRING;
  r->group = widgetGroup("Database");
  r->toolTip = i18n("Data Table");
  append(r);
#endif

  r = new WidgetDatabaseRecord;
  r->iconName = "lineedit.xpm";
  r->name = TQLINEEDIT_OBJECT_NAME_STRING;
  r->group = widgetGroup("Input");
  r->toolTip = i18n("Line Edit");
  append(r);

  r = new WidgetDatabaseRecord;
  r->iconName = "spinbox.xpm";
  r->name = TQSPINBOX_OBJECT_NAME_STRING;
  r->group = widgetGroup("Input");
  r->toolTip = i18n("Spin Box");
  append(r);

  r = new WidgetDatabaseRecord;
  r->iconName = "dateedit.xpm";
  r->name = TQDATEEDIT_OBJECT_NAME_STRING;
  r->group = widgetGroup("Input");
  r->toolTip = i18n("Date Edit");
  r->includeFile = "tqdatetimeedit.h";
  append(r);

  r = new WidgetDatabaseRecord;
  r->iconName = "timeedit.xpm";
  r->name = TQTIMEEDIT_OBJECT_NAME_STRING;
  r->group = widgetGroup("Input");
  r->toolTip = i18n("Time Edit");
  r->includeFile = "tqdatetimeedit.h";
  append(r);

  r = new WidgetDatabaseRecord;
  r->iconName = "datetimeedit.xpm";
  r->name = TQDATETIMEEDIT_OBJECT_NAME_STRING;
  r->group = widgetGroup("Input");
  r->toolTip = i18n("Date-Time Edit");
  r->includeFile = "tqdatetimeedit.h";
  append(r);

  r = new WidgetDatabaseRecord;
  r->iconName = "multilineedit.xpm";
  r->name = TQMULTILINEEDIT_OBJECT_NAME_STRING;
  r->group = widgetGroup("Temp");
  r->toolTip = i18n("Multi Line Edit");
  append(r);

  r = new WidgetDatabaseRecord;
  r->iconName = "richtextedit.xpm";
  r->name = TQTEXTEDIT_OBJECT_NAME_STRING;
  r->group = widgetGroup("Input");
  r->toolTip = i18n("Rich Text Edit");
  append(r);

  r = new WidgetDatabaseRecord;
  r->iconName = "combobox.xpm";
  r->name = TQCOMBOBOX_OBJECT_NAME_STRING;
  r->group = widgetGroup("Input");
  r->toolTip = i18n("Combo Box");
  append(r);

  r = new WidgetDatabaseRecord;
  r->iconName = "slider.xpm";
  r->name = TQSLIDER_OBJECT_NAME_STRING;
  r->group = widgetGroup("Input");
  r->toolTip = i18n("Slider");
  append(r);

  r = new WidgetDatabaseRecord;
  r->iconName = "scrollbar.xpm";
  r->name = TQSCROLLBAR_OBJECT_NAME_STRING;
  r->group = widgetGroup("Input");
  r->toolTip = i18n("Scrollbar");
  append(r);

  r = new WidgetDatabaseRecord;
  r->iconName = "dial.xpm";
  r->name = TQDIAL_OBJECT_NAME_STRING;
  r->group = widgetGroup("Input");
  r->toolTip = i18n("Dial");
  append(r);

  r = new WidgetDatabaseRecord;
  r->iconName = "label.xpm";
  r->name = TQLABEL_OBJECT_NAME_STRING;
  r->group = widgetGroup("Temp");
  r->toolTip = i18n("Label");
  append(r);

  r = new WidgetDatabaseRecord;
  r->iconName = "lcdnumber.xpm";
  r->name = TQLCDNUMBER_OBJECT_NAME_STRING;
  r->group = widgetGroup("Display");
  r->toolTip = i18n("LCD Number");
  append(r);

  r = new WidgetDatabaseRecord;
  r->iconName = "progress.xpm";
  r->name = TQPROGRESSBAR_OBJECT_NAME_STRING;
  r->group = widgetGroup("Display");
  r->toolTip = i18n("Progress Bar");
  append(r);

  r = new WidgetDatabaseRecord;
  r->iconName = "textview.xpm";
  r->name = TQTEXTVIEW_OBJECT_NAME_STRING;
  r->group = widgetGroup("Temp");
  r->toolTip = i18n("Text View");
  append(r);

  r = new WidgetDatabaseRecord;
  r->iconName = "textbrowser.xpm";
  r->name = TQTEXTBROWSER_OBJECT_NAME_STRING;
  r->group = widgetGroup("Display");
  r->toolTip = i18n("Text Browser");
  append(r);

  r = new WidgetDatabaseRecord;
  r->iconName = "spacer.xpm";
  r->name = "Spacer";
  r->group = widgetGroup("Temp");
  r->toolTip = i18n("Spacer");
  r->whatsThis =
      i18n("The Spacer provides horizontal and vertical spacing to be able to manipulate the behaviour of layouts.");
  append(r);

  r = new WidgetDatabaseRecord;
  r->name = TQWIDGET_OBJECT_NAME_STRING;
  r->isForm = true;
  r->group = widgetGroup("Forms");
  append(r);

  r = new WidgetDatabaseRecord;
  r->name = TQDIALOG_OBJECT_NAME_STRING;
  r->group = widgetGroup("Forms");
  r->isForm = true;
  append(r);

  r = new WidgetDatabaseRecord;
  r->name = TQWIZARD_OBJECT_NAME_STRING;
  r->group = widgetGroup("Forms");
  r->isContainer = true;
  append(r);

  r = new WidgetDatabaseRecord;
  r->name = "QDesignerWizard";
  r->group = widgetGroup("Forms");
  r->isContainer = true;
  append(r);

  r = new WidgetDatabaseRecord;
  r->name = TQLAYOUTWIDGET_OBJECT_NAME_STRING;
  r->group = widgetGroup("Temp");
  r->includeFile = "";
  r->isContainer = true;
  append(r);

  r = new WidgetDatabaseRecord;
  r->name = TQSPLITTER_OBJECT_NAME_STRING;
  r->group = widgetGroup("Temp");
  r->includeFile = "tqsplitter.h";
  r->isContainer = true;
  append(r);

  r = new WidgetDatabaseRecord;
  r->iconName = "tabwidget.xpm";
  r->name = "QDesignerTabWidget";
  r->group = widgetGroup("Temp");
  r->isContainer = true;
  append(r);

  r = new WidgetDatabaseRecord;
  r->iconName = "tabwidget.xpm";
  r->name = "QDesignerWidget";
  r->group = widgetGroup("Temp");
  r->isContainer = true;
  append(r);

  r = new WidgetDatabaseRecord;
  r->iconName = "tabwidget.xpm";
  r->name = "QDesignerDialog";
  r->group = widgetGroup("Temp");
  r->isContainer = true;
  append(r);

  r = new WidgetDatabaseRecord;
  r->iconName = "";
  r->name = TQMAINWINDOW_OBJECT_NAME_STRING;
  r->includeFile = "tqmainwindow.h";
  r->group = widgetGroup("Temp");
  r->isContainer = true;
  append(r);

#ifndef TQT_NO_SQL
  r = new WidgetDatabaseRecord;
  r->name = TQDATABROWSER_OBJECT_NAME_STRING;
  r->includeFile = "tqdatabrowser.h";
  r->group = widgetGroup("Database");
  r->toolTip = "Data Browser";
  r->iconName = "databrowser.xpm";
  r->isContainer = true;
  append(r);

  r = new WidgetDatabaseRecord;
  r->name = TQDATAVIEW_OBJECT_NAME_STRING;
  r->includeFile = "tqdataview.h";
  r->group = widgetGroup("Database");
  r->toolTip = "Data View";
  r->iconName = "dataview.xpm";
  r->isContainer = true;
  append(r);
#endif

  r = new WidgetDatabaseRecord;
  r->name = "Dialog";
  r->group = widgetGroup("Forms");
  r->isContainer = true;
  r->isForm = true;
  append(r);

  r = new WidgetDatabaseRecord;
  r->name = "Wizard";
  r->group = widgetGroup("Forms");
  r->isContainer = true;
  r->isForm = true;
  append(r);
  
  r = new WidgetDatabaseRecord;
  r->name = "EditorDialog";
  r->group = widgetGroup("Temp");
  r->isContainer = true;
  r->isForm = true;
  append(r);

  r = new WidgetDatabaseRecord;
  r->name = "EditorWizard";
  r->group = widgetGroup("Temp");
  r->isContainer = true;
  r->isForm = true;
  append(r);
  
  r = new WidgetDatabaseRecord;
  r->name = "EditorTabWidget";
  r->group = widgetGroup("Temp");
  r->isContainer = true;
  append(r);

  r = new WidgetDatabaseRecord;
  r->name = "EditorToolBox";
  r->group = widgetGroup("Temp");
  r->isContainer = true;
  append(r);


  /* Kommandr widgets */
  r = new WidgetDatabaseRecord;
  r->iconName = "label.xpm";
  r->name = "Label";
  r->group = widgetGroup("Kommander");
  r->toolTip = i18n("Text Label");
  r->whatsThis = i18n("The Text Label provides a widget to display static text.");
  append(r);

  r = new WidgetDatabaseRecord;
  r->iconName = "pixlabel.xpm";
  r->name = "PixmapLabel";
  r->group = widgetGroup("Kommander");
  r->toolTip = i18n("Pixmap Label");
  r->whatsThis = i18n("The Pixmap Label provides a widget to display pixmaps.");
  append(r);

  r = new WidgetDatabaseRecord;
  r->iconName = "lineedit.xpm";
  r->name = "LineEdit";
  r->group = widgetGroup("Kommander");
  r->toolTip = i18n("A line edit");
  append(r);

  r = new WidgetDatabaseRecord;
  r->name = "TextEdit";
  r->iconName = "textedit.png";
  r->group = widgetGroup("Kommander");
  r->toolTip = i18n("A rich text edit");
  append(r);

  r = new WidgetDatabaseRecord;
  r->name = "TextBrowser";
  r->iconName = "textbrowser.png";
  r->group = widgetGroup("Kommander");
  r->toolTip = i18n("Text Browser");
  append(r);

  r = new WidgetDatabaseRecord;
  r->iconName = "listbox.xpm";
  r->name = "ListBox";
  r->group = widgetGroup("Kommander");
  r->toolTip = i18n("List Box");
  append(r);

  r = new WidgetDatabaseRecord;
  r->iconName = "combobox.xpm";
  r->name = "ComboBox";
  r->group = widgetGroup("Kommander");
  r->toolTip = i18n("A combo box");
  append(r);

  r = new WidgetDatabaseRecord;
  r->name = "TreeWidget";
  r->iconName = "listview.png";
  r->group = widgetGroup("Kommander");
  r->toolTip = i18n("A tree widget");
  append(r);

  r = new WidgetDatabaseRecord;
  r->name = "Table";
  r->iconName = "table.xmp";
  r->group = widgetGroup("Kommander");
  r->toolTip = i18n("A table widget");
  append(r);

  r = new WidgetDatabaseRecord;
  r->iconName = "pushbutton.xpm";
  r->name = "ExecButton";
  r->group = widgetGroup("Kommander");
  r->toolTip = i18n("A button that when clicked, execute a command");
  append(r);

  r = new WidgetDatabaseRecord;
  r->iconName = "closebutton.png";
  r->name = "CloseButton";
  r->group = widgetGroup("Kommander");
  r->toolTip = i18n("A button that closes the dialog it is in");
  append(r);

  r = new WidgetDatabaseRecord;
  r->name = "Konsole";
  r->iconName = "konsole.png";
  r->group = widgetGroup("Kommander");
  r->toolTip = i18n("A listbox showing output of a script");
  append(r);
  
  r = new WidgetDatabaseRecord;
  r->name = "FileSelector";
  r->iconName = "lineedit.xpm";
  r->group = widgetGroup("Kommander");
  r->toolTip = i18n("A widget made up of a line edit and push button, for selecting files and folders");
  append(r);

  r = new WidgetDatabaseRecord;
  r->iconName = "checkbox.xpm";
  r->name = "CheckBox";
  r->group = widgetGroup("Kommander");
  r->toolTip = i18n("A check box");
  append(r);

  r = new WidgetDatabaseRecord;
  r->iconName = "radiobutton.xpm";
  r->name = "RadioButton";
  r->group = widgetGroup("Kommander");
  r->toolTip = i18n("A radio button");
  append(r);

  r = new WidgetDatabaseRecord;
  r->iconName = "buttongroup.xpm";
  r->name = "ButtonGroup";
  r->group = widgetGroup("Kommander");
  r->toolTip = i18n("A widget for grouping buttons together");
  r->isContainer = true;
  append(r);

  r = new WidgetDatabaseRecord;
  r->iconName = "groupbox.xpm";
  r->name = "GroupBox";
  r->group = widgetGroup("Kommander");
  r->toolTip = i18n("A widget for grouping other widgets together");
  r->isContainer = true;
  append(r);

  r = new WidgetDatabaseRecord;
  r->iconName = "tabwidget.xpm";
  r->name = "TabWidget";
  r->group = widgetGroup("Kommander");
  r->toolTip = i18n("A widget with tabs");
  r->isContainer = true;
  append(r);


//   r = new WidgetDatabaseRecord;
//   r->iconName = "";
//   r->name = "ToolBox";
//   r->group = widgetGroup("Kommander");
//   r->toolTip = "";
//   r->isContainer = true;
//   append(r);

  r = new WidgetDatabaseRecord;
  r->iconName = "spinbox.xpm";
  r->name = "SpinBoxInt";
  r->group = widgetGroup("Kommander");
  r->toolTip = i18n("A spin box");
  append(r);

  r = new WidgetDatabaseRecord;
  r->name = "Slider";
  r->iconName = "slider.png";
  r->group = widgetGroup("Kommander");
  r->toolTip = i18n("Slider");
  append(r);

  r = new WidgetDatabaseRecord;
  r->name = "RichTextEditor";
  r->iconName = "richtextedit.xpm";
  r->group = widgetGroup("Kommander");
  r->toolTip = i18n("A small rich text editor");
  append(r);

  r = new WidgetDatabaseRecord;
  r->name = "StatusBar";
  r->iconName = "statusbar.png";
  r->group = widgetGroup("Kommander");
  r->toolTip = i18n("A status bar");
  append(r);

  r = new WidgetDatabaseRecord;
  r->name = "ProgressBar";
  r->iconName = "progress.png";
  r->group = widgetGroup("Kommander");
  r->toolTip = i18n("A progress bar");
  append(r);
  
  r = new WidgetDatabaseRecord;
  r->name = "ScriptObject";
  r->iconName = "shellscript.png";
  r->group = widgetGroup( "Kommander" );
  r->toolTip = i18n("A hidden script container");
  append(r);

  r = new WidgetDatabaseRecord;
  r->name = "Timer";
  r->iconName = "timer.png";
  r->group = widgetGroup( "Kommander" );
  r->toolTip = i18n("A timer for running scripts periodically");
  append(r);
 
 r = new WidgetDatabaseRecord;
  r->iconName = "datepicker.xpm";
  r->name = "DatePicker";
  r->group = widgetGroup("Kommander");
  r->toolTip = i18n("A date selection widget");
  append(r);

#ifndef UIC
  setupPlugins();
#endif
}

void WidgetDatabase::setupPlugins()
{
  if (plugins_set_up)
    return;
  //qDebug("WidgetDatabase::setupPlugins");
  plugins_set_up = true;
  FeatureList widgets = KommanderFactory::featureList();
  //qDebug("num features = %d", widgets.count());
  for (FeatureList::Iterator it = widgets.begin(); it != widgets.end(); ++it)
  {
    if (hasWidget(it.key()))
      continue;
    WidgetDatabaseRecord *r = new WidgetDatabaseRecord;

    TQString grp = (*it).group;
    if (grp.isEmpty())
      grp = "Kommander";
    r->group = widgetGroup(grp);
    r->toolTip = (*it).toolTip;
    r->iconSet = (*it).iconSet;
    r->whatsThis = (*it).whatsThis;
    r->isContainer = (*it).isContainer;
    r->name = it.key();
    append(r);
  }
}

/*!
  Returns the number of elements in the widget database.
*/

int WidgetDatabase::count()
{
    setupDataBase( -1 );
    return dbcount;
}

/*!
  Returns the id at which the ids of custom widgets start.
*/

int WidgetDatabase::startCustom()
{
    setupDataBase( -1 );
    return dbcustom;
}

/*!
  Returns the iconset which represents the class registered as \a id.
*/

TQIconSet WidgetDatabase::iconSet( int id )
{
    setupDataBase( id );
    WidgetDatabaseRecord *r = at( id );
    if ( !r )
	   return TQIconSet();
#if !defined(UIC) && !defined(RESOURCE)
    if ( !r->iconSet )
	   r->iconSet = new TQIconSet( PixmapChooser::loadPixmap( r->iconName, PixmapChooser::Small ),
				PixmapChooser::loadPixmap( r->iconName, PixmapChooser::Large ) );
    return *r->iconSet;
#else
    return TQIconSet();
#endif
}

/*!
  Returns the classname of the widget which is registered as \a id.
*/

TQString WidgetDatabase::className( int id )
{
    setupDataBase( id );
    WidgetDatabaseRecord *r = at( id );
    if ( !r )
	return TQString();
    return r->name;
}

/*!
  Returns the group the widget registered as \a id belongs to.
*/

TQString WidgetDatabase::group( int id )
{
    setupDataBase( id );
    WidgetDatabaseRecord *r = at( id );
    if ( !r )
	return TQString();
    return r->group;
}

/*!
  Returns the tooltip text of the widget which is registered as \a id.
*/

TQString WidgetDatabase::toolTip( int id )
{
    setupDataBase( id );
    WidgetDatabaseRecord *r = at( id );
    if ( !r )
	return TQString();
    return r->toolTip;
}

/*!
  Returns the what's this? text of the widget which is registered as \a id.
*/

TQString WidgetDatabase::whatsThis( int id )
{
    setupDataBase( id );
    WidgetDatabaseRecord *r = at( id );
    if ( !r )
	return TQString();
    return r->whatsThis;
}

/*!
  Returns the include file if the widget which is registered as \a id.
*/

TQString WidgetDatabase::includeFile( int id )
{
    setupDataBase( id );
    WidgetDatabaseRecord *r = at( id );
    if ( !r )
	return TQString();
    if ( r->includeFile.isNull() )
	return r->name.lower() + ".h";
    return r->includeFile;
}

/*!  Returns whether the widget registered as \a id is a form.
*/
bool WidgetDatabase::isForm( int id )
{
    setupDataBase( id );
    WidgetDatabaseRecord *r = at( id );
    if ( !r )
	return false;
    return r->isForm;
}

/*!  Returns whether the widget registered as \a id can have tqchildren.
*/

bool WidgetDatabase::isContainer( int id )
{
    setupDataBase( id );
    WidgetDatabaseRecord *r = at( id );
    if ( !r )
	return false;
    return r->isContainer || r->isForm;
}

TQString WidgetDatabase::createWidgetName( int id )
{
    setupDataBase( id );
    TQString n = className( id );
    if ( n == TQLAYOUTWIDGET_OBJECT_NAME_STRING )
	n = "Layout";
    if ( n[ 0 ] == 'Q' )
	n = n.mid( 1 );
    WidgetDatabaseRecord *r = at( id );
    if ( !r )
	return n;
    n += TQString::number( ++r->nameCounter );
    return n;
}

/*!  Returns the id for \a name or -1 if \a name is unknown.
 */
int WidgetDatabase::idFromClassName( const TQString &name )
{
    setupDataBase( -1 );
    if ( name.isEmpty() )
	return 0;
    int *i = className2Id->find( name );
    if ( i )
	return *i;
    if ( name == "FormWindow" )
	return idFromClassName( TQLAYOUTWIDGET_OBJECT_NAME_STRING );
#ifdef UIC
    setupDataBase( -2 );
    i = className2Id->find( name );
    if ( i )
	return *i;
#endif
    return -1;
}

bool WidgetDatabase::hasWidget( const TQString &name )
{
    return className2Id->find( name ) != 0;
}

WidgetDatabaseRecord *WidgetDatabase::at( int index )
{
    if ( index < 0 )
	return 0;
    if ( index >= dbcustom && index < dbcustomcount )
	return widget_db[ index ];
    if ( index < dbcount )
	return widget_db[ index ];
    return 0;
}

void WidgetDatabase::insert( int index, WidgetDatabaseRecord *r )
{
    if ( index < 0 || index >= dbsize )
	return;
    widget_db[ index ] = r;
    className2Id->insert( r->name, new int( index ) );
    if ( index < dbcustom )
	dbcount = TQMAX( dbcount, index );
}

void WidgetDatabase::append( WidgetDatabaseRecord *r )
{
    if ( !was_in_setup )
	setupDataBase( -1 );
    insert( dbcount++, r );
}

TQString WidgetDatabase::widgetGroup( const TQString &g )
{
    if ( wGroups->find( g ) == -1 )
	wGroups->append( g );
    return g;
}

bool WidgetDatabase::isGroupEmpty( const TQString &grp )
{
    WidgetDatabaseRecord *r = 0;
    for ( int i = 0; i < dbcount; ++i ) {
	if ( !( r = widget_db[ i ] ) )
	    continue;
	if ( r->group == grp )
	{
	// FIXME
	    if(r->group == "Kommander")
		    return false;

	    if(r->name[0] != 'Q')
  	        return false;
	}
    }
    return true;
}

TQString WidgetDatabase::widgetGroup( int i )
{
    setupDataBase( -1 );
    if ( i >= 0 && i < (int)wGroups->count() )
	return wGroups->at( i );
    return TQString();
}

int WidgetDatabase::numWidgetGroups()
{
    setupDataBase( -1 );
    return wGroups->count();
}

bool WidgetDatabase::isGroupVisible( const TQString &g )
{
    setupDataBase( -1 );
    return invisibleGroups->find( g ) == -1;
}

int WidgetDatabase::addCustomWidget( WidgetDatabaseRecord *r )
{
    insert( dbcustomcount++, r );
    return dbcustomcount - 1;
}

bool WidgetDatabase::isCustomWidget( int id )
{
    if ( id >= dbcustom && id < dbcustomcount )
	return true;
    return false;
}

bool WidgetDatabase::isWhatsThisLoaded()
{
    return whatsThisLoaded;
}

void WidgetDatabase::loadWhatsThis( const TQString &docPath )
{
    TQString whatsthisFile = docPath + "/whatsthis";
    TQFile f( whatsthisFile );
    if ( !f.open( IO_ReadOnly ) )
	return;
    TQTextStream ts( &f );
    while ( !ts.atEnd() ) {
	TQString s = ts.readLine();
	TQStringList l = TQStringList::split( " | ", s );
	int id = idFromClassName( l[ 1 ] );
	WidgetDatabaseRecord *r = at( id );
	if ( r )
	    r->whatsThis = l[ 0 ];
    }
    whatsThisLoaded = true;
}


// ### TQt 3.1: make these publically accessible via TQWidgetDatabase API
#ifndef KOMMANDER
#if defined(UIC)
bool dbnounload = false;
TQStringList *dbpaths = 0;
#endif

TQPluginManager<WidgetInterface> *widgetManager()
{
    if ( !widgetPluginManager ) {
#ifndef KOMMANDER
	widgetPluginManager = new TQPluginManager<WidgetInterface>( IID_Widget, TQApplication::libraryPaths(), "/designer" );
#else
	widgetPluginManager = new TQPluginManager<WidgetInterface>( IID_Widget, TQApplication::libraryPaths(), "" );
#endif

	cleanup_manager.add( &widgetPluginManager );
#if defined(UIC)
	if ( dbnounload )
	    widgetPluginManager->setAutoUnload( false );
	if ( dbpaths ) {
	    TQStringList::ConstIterator it = dbpaths->begin();
	    for ( ; it != dbpaths->end(); ++it )
		widgetPluginManager->addLibraryPath( *it );
	}
#endif
    }
    return widgetPluginManager;
}
#endif
