/*
    This file is part of the KDE games library
    Copyright (C) 2001,2002,2003 Nicolas Hadacek (hadacek@kde.org)

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License version 2 as published by the Free Software Foundation.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/

#include "kgamelcd.h"
#include "kgamelcd.moc"

#include <tqlayout.h>
#include <tqlabel.h>
#include <tqtimer.h>

#include <kglobal.h>


//-----------------------------------------------------------------------------
KGameLCD::KGameLCD(uint nbDigits, TQWidget *parent, const char *name)
    : TQLCDNumber(nbDigits, parent, name), _htime(800)
{
    const TQPalette &p = palette();
    _fgColor = p.color(TQPalette::Active, TQColorGroup::Foreground);
    _hlColor = p.color(TQPalette::Active, TQColorGroup::HighlightedText);

    _timer = new TQTimer(this);
    connect(_timer, TQT_SIGNAL(timeout()), TQT_SLOT(timeout()));

    setFrameStyle(Panel | Plain);
	setSegmentStyle(Flat);

    displayInt(0);
}

KGameLCD::~KGameLCD()
{}

void KGameLCD::setDefaultBackgroundColor(const TQColor &color)
{
    TQPalette p = palette();
    p.setColor(TQColorGroup::Background, color);
    setPalette(p);
}

void KGameLCD::setDefaultColor(const TQColor &color)
{
    _fgColor = color;
    TQPalette p = palette();
    p.setColor(TQColorGroup::Foreground, color);
    setPalette(p);
}

void KGameLCD::setHighlightColor(const TQColor &color)
{
    _hlColor = color;
}

void KGameLCD::setLeadingString(const TQString &s)
{
    _lead = s;
    displayInt(0);
}

void KGameLCD::setHighlightTime(uint time)
{
    _htime = time;
}

void KGameLCD::resetColor()
{
    setColor(TQColor());
}

void KGameLCD::setColor(const TQColor &color)
{
    const TQColor &c = (color.isValid() ? color : _fgColor);
    TQPalette p = palette();
    p.setColor(TQColorGroup::Foreground, c);
    setPalette(p);
}

void KGameLCD::displayInt(int v)
{
    int n = numDigits() - _lead.length();
    display(_lead + TQString::number(v).rightJustify(n));
}

void KGameLCD::highlight()
{
    highlight(true);
    _timer->start(_htime, true);
}

void KGameLCD::highlight(bool light)
{
    if (light) setColor(_hlColor);
    else resetColor();
}

//-----------------------------------------------------------------------------
KGameLCDClock::KGameLCDClock(TQWidget *parent, const char *name)
: KGameLCD(5, parent, name)
{
    _timerClock = new TQTimer(this);
    connect(_timerClock, TQT_SIGNAL(timeout()), TQT_SLOT(timeoutClock()));
}

KGameLCDClock::~KGameLCDClock()
{}

void KGameLCDClock::timeoutClock()
{
    // waiting an hour does not restart timer
    if ( _min==59 && _sec==59 ) return;
    _sec++;
    if (_sec==60) {
        _min++;
        _sec = 0;
    }
    showTime();
}

TQString KGameLCDClock::pretty() const
{
    TQString sec = TQString::number(_sec).rightJustify(2, '0', true);
    TQString min = TQString::number(_min).rightJustify(2, '0', true);
    return min + ':' + sec;
}

void KGameLCDClock::showTime()
{
    display(pretty());
}

void KGameLCDClock::reset()
{
    _timerClock->stop();
	_sec = 0;
    _min = 0;
	showTime();
}

void KGameLCDClock::start()
{
    _timerClock->start(1000); // 1 second
}

void KGameLCDClock::stop()
{
    _timerClock->stop();
}

uint KGameLCDClock::seconds() const
{
    return _min*60 + _sec;
}

void KGameLCDClock::setTime(uint sec)
{
    Q_ASSERT( sec<3600 );
    _sec = sec % 60;
    _min = sec / 60;
    showTime();
}

void KGameLCDClock::setTime(const TQString &s)
{
    Q_ASSERT( s.length()==5 && s[2]==':' );
    uint min = kMin(s.section(':', 0, 0).toUInt(), uint(59));
    uint sec = kMin(s.section(':', 1, 1).toUInt(), uint(59));
    setTime(sec + min*60);
}


//-----------------------------------------------------------------------------
class KGameLCDList::KGameLCDListPrivate
{
public:
  TQValueVector<TQLabel *> _leadings;
};

KGameLCDList::KGameLCDList(const TQString &title, TQWidget *parent,
                           const char *name)
    : TQWidget(parent, name)
{
    init(title);
}

KGameLCDList::KGameLCDList(TQWidget *parent, const char *name)
    : TQWidget(parent, name)
{
    init(TQString());
}

KGameLCDList::~KGameLCDList()
{
  delete d;
}

void KGameLCDList::init(const TQString &title)
{
    d = new KGameLCDListPrivate;

    TQGridLayout *top = new TQGridLayout(this, 1, 2, 5);
    top->setColStretch(1, 1);

    _title = new TQLabel(title, this);
    _title->tqsetAlignment(AlignCenter);
    top->addMultiCellWidget(_title, 0, 0, 0, 1, AlignCenter);
}

void KGameLCDList::append(TQLCDNumber *lcd)
{
    append(TQString(), lcd);
}

void KGameLCDList::append(const TQString &leading, TQLCDNumber *lcd)
{
    uint i = size();
    TQLabel *label = 0;
    if ( !leading.isEmpty() ) {
      label = new TQLabel(leading, this);
      static_cast<TQGridLayout *>(tqlayout())->addWidget(label, i+1, 0);
    }
    d->_leadings.push_back(label);
    _lcds.push_back(lcd);
    static_cast<TQGridLayout *>(tqlayout())->addWidget(lcd, i+1, 1);
}

void KGameLCDList::clear()
{
    for (uint i=0; i<_lcds.size(); i++) {
        delete d->_leadings[i];
        delete _lcds[i];
    }
    d->_leadings.clear();
    _lcds.clear();
}
