/*
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */


// QT includes
#include <tqlabel.h>
#include <tqlineedit.h>
#include <tqcheckbox.h>
#include <tqcombobox.h>
#include <tqlayout.h>

// KDE includes
#include <klocale.h>
#include <knuminput.h>
#include <kapplication.h>
#include <kconfig.h>
#include <knumvalidator.h>

// own includes
#include "defines.h"
#include "newgamedlg.h"


NewGameDlg::NewGameDlg(TQWidget* parent)
  : KDialogBase(Plain, i18n("New Game"), 
		Ok|Cancel, Ok, parent, 0, true, true)
{
  TQVBoxLayout  *topLayout = new TQVBoxLayout(plainPage(), spacingHint());
  TQHBoxLayout  *l         = new TQHBoxLayout(topLayout);

  KConfig* conf = kapp->config();
  conf->setGroup("NewGameDlg");
  bool showNewGameDlg = conf->readBoolEntry("showNewGameDlgOnStartup", 
					    SHOWNEWGAME_DEFAULT);
  bool readConfig     = conf->readBoolEntry("readFromConfig", 
					    LOADGAME_DEFAULT);
  int playerNr        = conf->readNumEntry("players", DEFAULT_PLAYERS);
  int money           = conf->readNumEntry("startMoney", START_MONEY);
 
  readFromConfig = new TQCheckBox(i18n("Try loading a game"), plainPage());
  readFromConfig->adjustSize();
  readFromConfig->setChecked(readConfig);
  l->addWidget(readFromConfig);

  readFromConfigLabel = new TQLabel(i18n("The following values are used if loading from config fails"), plainPage());
  if (!readFromConfig->isChecked())
    readFromConfigLabel->hide();
  readFromConfigLabel->adjustSize();
  l->addWidget(readFromConfigLabel);
  connect(readFromConfig, TQT_SIGNAL(toggled(bool)), 
	  this,           TQT_SLOT(changeReadFromConfig(bool)));

  players = new KIntNumInput(playerNr, plainPage());
  players->setRange(1, MAX_PLAYERS);
  players->setLabel(i18n("How many players do you want?"));
  topLayout->addWidget(players);

  l = new TQHBoxLayout(topLayout);
  l->addWidget(new TQLabel(i18n("Your name:"), plainPage()));
  player1Name = new TQLineEdit(plainPage());
  l->addWidget(player1Name);

  l = new TQHBoxLayout(topLayout);
  l->addWidget(new TQLabel(i18n("Players' starting money:"), plainPage()));
  moneyOfPlayers = new TQLineEdit(TQString("%1").tqarg(money), plainPage());
  moneyOfPlayers->setValidator( new KIntValidator( 0,999999,moneyOfPlayers ) );

  l->addWidget(moneyOfPlayers);

  l = new TQHBoxLayout(topLayout);
  l->addWidget(new TQLabel(i18n("The names of your opponents:"), plainPage()));
  computerNames = new TQComboBox(true, plainPage());
  computerNames->setInsertionPolicy(TQComboBox::AtCurrent);
  l->addWidget(computerNames);

  l = new TQHBoxLayout(topLayout);
  l->addWidget(new TQLabel(i18n("Show this dialog every time on startup"), 
			  plainPage()));
  showDialogOnStartup = new TQCheckBox(plainPage());
  showDialogOnStartup->setChecked(showNewGameDlg);
  l->addWidget(showDialogOnStartup);

  setPlayerNames();
}


NewGameDlg::~NewGameDlg()
{
  if (result() == Accepted) {
    KConfig* conf = kapp->config(); 
    conf->setGroup("NewGameDlg"); // defaults for the newGameDlg only
    conf->writeEntry("showNewGameDlgOnStartup", showOnStartup());
    conf->writeEntry("readFromConfig", readFromConfigFile()); // just a default!
    conf->writeEntry("players", getPlayers());
    conf->writeEntry("startMoney", money());
  }

  //delete the visible elements:
  delete readFromConfigLabel;
  delete readFromConfig;
  delete players;
  delete moneyOfPlayers;
  delete showDialogOnStartup;
  delete player1Name;
  delete computerNames;

}


void NewGameDlg::setPlayerNames(int no, TQString playerName)
{
  if (no < 0) {
    kapp->config()->setGroup("Save");
    player1Name->setText(kapp->config()->readEntry("Name_0", i18n("You")));
    computerNames->clear();
    for (int i = 1; i < MAX_PLAYERS; i++) {
      computerNames->insertItem(kapp->config()->readEntry(TQString("Name_%1").tqarg(i), i18n("Computer %1").tqarg(i)));
    }
  } else if (no == 0) {
    player1Name->setText(playerName);
  } else {
    if (computerNames->count() > no)
      computerNames->insertItem(playerName, no-1);
    else
      computerNames->changeItem(playerName, no-1);
  }
}


void NewGameDlg::changeReadFromConfig(bool show)
{
  if (show)
    readFromConfigLabel->show();
  else
    readFromConfigLabel->hide();
}


bool NewGameDlg::showOnStartup()
{
  return showDialogOnStartup->isChecked();
}


int NewGameDlg::getPlayers()
{
  return players->value();
}


bool NewGameDlg::readFromConfigFile()
{
  return readFromConfig->isChecked(); 
}


int NewGameDlg::money()
{
  bool  ok    = true;
  int   money = moneyOfPlayers->text().toInt(&ok);
  if (ok) 
    return money;
  else
    return START_MONEY;
}


TQString NewGameDlg::name(int nr)
{
  if (computerNames->currentText() != computerNames->text(computerNames->currentItem()))
    computerNames->changeItem(computerNames->currentText(), computerNames->currentItem());
	
  if (nr == 0)
    return player1Name->text();

  if (nr <= computerNames->count())
    return computerNames->text(nr-1);

  return i18n("Player");
}


void NewGameDlg::hideReadingFromConfig()
{
  readFromConfig->hide();
  readFromConfigLabel->hide();
  readFromConfig->setChecked(false);
}


#include "newgamedlg.moc"
