// Copyright (c) 2002-2003 Rob Kaper <cap@capsi.com>
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// version 2 as published by the Free Software Foundation.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; see the file COPYING.  If not, write to
// the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
// Boston, MA 02110-1301, USA.

#include <tqvgroupbox.h>
#include <tqradiobutton.h>

#include <kdebug.h>
#include <kdialog.h>
#include <klocale.h>
#include <kiconloader.h>
#include <knotifyclient.h>

#include <atlantic_core.h>
#include <game.h>
#include <player.h>

#include "selectgame_widget.h"

SelectGame::SelectGame(AtlanticCore *atlanticCore, TQWidget *parent, const char *name) : TQWidget(parent, name)
{
	m_atlanticCore = atlanticCore;

	connect(m_atlanticCore, TQT_SIGNAL(createGUI(Game *)), this, TQT_SLOT(addGame(Game *)));
	connect(m_atlanticCore, TQT_SIGNAL(removeGUI(Game *)), this, TQT_SLOT(delGame(Game *)));

	m_mainLayout = new TQVBoxLayout(this, KDialog::marginHint());
	Q_CHECK_PTR(m_mainLayout);

	TQVGroupBox *groupBox;
	groupBox = new TQVGroupBox(i18n("Create or Select monopd Game"), this, "groupBox");
	m_mainLayout->addWidget(groupBox);

	// List of games
	m_gameList = new KListView(groupBox, "m_gameList");
	m_gameList->addColumn(i18n("Game"));
	m_gameList->addColumn(i18n("Description"));
	m_gameList->addColumn(i18n("Id"));
	m_gameList->addColumn(i18n("Players"));
	m_gameList->setAllColumnsShowFocus(true);
//	m_mainLayout->addWidget(m_gameList);

	connect(m_gameList, TQT_SIGNAL(clicked(TQListViewItem *)), this, TQT_SLOT(validateConnectButton()));
	connect(m_gameList, TQT_SIGNAL(doubleClicked(TQListViewItem *)), this, TQT_SLOT(connectClicked()));
	connect(m_gameList, TQT_SIGNAL(rightButtonClicked(TQListViewItem *, const TQPoint &, int)), this, TQT_SLOT(validateConnectButton()));
	connect(m_gameList, TQT_SIGNAL(selectionChanged(TQListViewItem *)), this, TQT_SLOT(validateConnectButton()));

	TQHBoxLayout *buttonBox = new TQHBoxLayout(m_mainLayout, KDialog::spacingHint());

	KPushButton *backButton = new KPushButton(SmallIcon("back"), i18n("Server List"), this);
	buttonBox->addWidget(backButton);

	connect(backButton, TQT_SIGNAL(clicked()), this, TQT_SIGNAL(leaveServer()));

	buttonBox->addItem(new TQSpacerItem(20, 20, TQSizePolicy::Expanding, TQSizePolicy::Minimum));

	m_connectButton = new KPushButton(SmallIconSet("forward"), i18n("Create Game"), this);
	m_connectButton->setEnabled(false);
	buttonBox->addWidget(m_connectButton);

	connect(m_connectButton, TQT_SIGNAL(clicked()), this, TQT_SLOT(connectClicked()));

}

void SelectGame::addGame(Game *game)
{
	connect(game, TQT_SIGNAL(changed(Game *)), this, TQT_SLOT(updateGame(Game *)));

	if (game->id() == -1)
	{
		TQListViewItem *item = new TQListViewItem( m_gameList, i18n("Create a new %1 Game").tqarg(game->name()), game->description(), TQString(), TQString(), game->type() );
		item->setPixmap(0, TQPixmap(SmallIcon("filenew")));
	}
	else
	{
		Player *master = game->master();
		TQListViewItem *item = new TQListViewItem( m_gameList, i18n("Join %1's %2 Game").tqarg( (master ? master->name() : TQString()), game->name() ), game->description(), TQString::number(game->id()), TQString::number(game->players()), game->type() );
		item->setPixmap( 0, TQPixmap(SmallIcon("atlantik")) );
		item->setEnabled(game->canBeJoined());

		KNotifyClient::event(winId(), "newgame");

		connect(master, TQT_SIGNAL(changed(Player *)), this, TQT_SLOT(playerChanged(Player *)));
	}

//	validateConnectButton();
}

void SelectGame::delGame(Game *game)
{
	TQListViewItem *item = findItem(game);
	if (!item)
		return;

	delete item;

	validateConnectButton();
}

void SelectGame::updateGame(Game *game)
{
	TQListViewItem *item = findItem(game);
	if (!item)
		return;

	item->setText( 1, game->description() );

	if (game->id() == -1)
		item->setText(0, i18n("Create a new %1 Game").tqarg(game->name()));
	else
	{
		Player *master = game->master();
		item->setText( 0, i18n("Join %1's %2 Game").tqarg( (master ? master->name() : TQString()), game->name() ) );
		item->setText( 3, TQString::number( game->players() ) );
		item->setEnabled( game->canBeJoined() );

		connect(master, TQT_SIGNAL(changed(Player *)), this, TQT_SLOT(playerChanged(Player *)));
	}
	m_gameList->triggerUpdate();

	validateConnectButton();
}

void SelectGame::playerChanged(Player *player)
{
	TQListViewItem *item = m_gameList->firstChild();
	Game *game = 0;

	while (item)
	{
		game = m_atlanticCore->findGame( item->text(2).toInt() );
		if ( game && game->master() == player )
		{
			item->setText( 0, i18n("Join %1's %2 Game").tqarg( player->name(), game->name() ) );
			return;
		}
		item = item->nextSibling();
	}
}

TQListViewItem *SelectGame::findItem(Game *game)
{
	TQListViewItem *item = m_gameList->firstChild();
	while (item)
	{
		if ( (game->id() == -1 || item->text(2) == TQString::number(game->id())) && item->text(4) == game->type() )
			return item;

		item = item->nextSibling();
	}
	return 0;
}

void SelectGame::validateConnectButton()
{
	if (TQListViewItem *item = m_gameList->selectedItem())
	{
		if (item->text(2).toInt() > 0)
			m_connectButton->setText(i18n("Join Game"));
		else
			m_connectButton->setText(i18n("Create Game"));

		m_connectButton->setEnabled(true);
	}
	else
		m_connectButton->setEnabled(false);
}

void SelectGame::connectClicked()
{
	if (TQListViewItem *item = m_gameList->selectedItem())
	{
		if (int gameId = item->text(2).toInt())
			emit joinGame(gameId);
		else
			emit newGame(item->text(4));
	}
}

#include "selectgame_widget.moc"
