/* ============================================================
 *
 * This file is a part of kipi-plugins project
 * http://www.kipi-plugins.org
 *
 * Date        : 2006-10-12
 * Description : EXIF caption settings page.
 *
 * Copyright (C) 2006-2007 by Gilles Caulier <caulier dot gilles at gmail dot com>
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * ============================================================ */

// QT includes.

#include <tqlayout.h>
#include <tqhgroupbox.h>
#include <tqgroupbox.h>
#include <tqlabel.h>
#include <tqwhatsthis.h>
#include <tqvalidator.h>
#include <tqcheckbox.h>

// KDE includes.

#include <klocale.h>
#include <kdialog.h>
#include <klineedit.h>
#include <ktextedit.h>
#include <kapplication.h>
#include <kaboutdata.h>
#include <kactivelabel.h>

// LibKExiv2 includes. 

#include <libkexiv2/kexiv2.h>

// Local includes.

#include "pluginsversion.h"
#include "exifcaption.h"
#include "exifcaption.moc"

namespace KIPIMetadataEditPlugin
{

class EXIFCaptionPriv
{
public:

    EXIFCaptionPriv()
    {
        documentNameEdit     = 0;
        imageDescEdit        = 0;
        artistEdit           = 0;
        copyrightEdit        = 0;
        userCommentEdit      = 0;
        userCommentCheck     = 0;
        documentNameCheck    = 0;
        imageDescCheck       = 0;
        artistCheck          = 0;
        copyrightCheck       = 0;
        syncJFIFCommentCheck = 0;
        syncHOSTCommentCheck = 0;
        syncIPTCCaptionCheck = 0;
    }

    TQCheckBox *documentNameCheck;
    TQCheckBox *imageDescCheck;
    TQCheckBox *artistCheck;
    TQCheckBox *copyrightCheck;
    TQCheckBox *userCommentCheck;
    TQCheckBox *syncJFIFCommentCheck;
    TQCheckBox *syncHOSTCommentCheck;
    TQCheckBox *syncIPTCCaptionCheck;

    KTextEdit *userCommentEdit;

    KLineEdit *documentNameEdit;
    KLineEdit *imageDescEdit;
    KLineEdit *artistEdit;
    KLineEdit *copyrightEdit;
};

EXIFCaption::EXIFCaption(TQWidget* parent)
           : TQWidget(parent)
{
    d = new EXIFCaptionPriv;
    TQVBoxLayout *vlay = new TQVBoxLayout( parent, 0, KDialog::spacingHint() );

    // EXIF only accept printable Ascii char.
    TQRegExp asciiRx("[\x20-\x7F]+$");
    TQValidator *asciiValidator = new TQRegExpValidator(asciiRx, TQT_TQOBJECT(this));
  
    // --------------------------------------------------------

    d->documentNameCheck = new TQCheckBox(i18n("Document name (*):"), parent);
    d->documentNameEdit  = new KLineEdit(parent);
    d->documentNameEdit->setValidator(asciiValidator);
    vlay->addWidget(d->documentNameCheck);
    vlay->addWidget(d->documentNameEdit);
    TQWhatsThis::add(d->documentNameEdit, i18n("<p>Enter the name of the document from which "
                                         "this image was been scanned. This field is limited "
                                         "to ASCII characters."));

    // --------------------------------------------------------

    d->imageDescCheck = new TQCheckBox(i18n("Image description (*):"), parent);
    d->imageDescEdit  = new KLineEdit(parent);
    d->imageDescEdit->setValidator(asciiValidator);
    vlay->addWidget(d->imageDescCheck);
    vlay->addWidget(d->imageDescEdit);
    TQWhatsThis::add(d->imageDescEdit, i18n("<p>Enter the image title. This field is limited "
                                      "to ASCII characters."));
        
    // --------------------------------------------------------

    d->artistCheck = new TQCheckBox(i18n("Artist (*):"), parent);
    d->artistEdit  = new KLineEdit(parent);
    d->artistEdit->setValidator(asciiValidator);
    vlay->addWidget(d->artistCheck);
    vlay->addWidget(d->artistEdit);
    TQWhatsThis::add(d->artistEdit, i18n("<p>Enter the image author's name. "
                                   "This field is limited to ASCII characters."));

    // --------------------------------------------------------

    d->copyrightCheck = new TQCheckBox(i18n("Copyright (*):"), parent);
    d->copyrightEdit  = new KLineEdit(parent);
    d->copyrightEdit->setValidator(asciiValidator);
    vlay->addWidget(d->copyrightCheck);
    vlay->addWidget(d->copyrightEdit);
    TQWhatsThis::add(d->copyrightEdit, i18n("<p>Enter the copyright owner of the image. "
                                      "This field is limited to ASCII characters."));

    // --------------------------------------------------------

    d->userCommentCheck = new TQCheckBox(i18n("Caption:"), parent);
    d->userCommentEdit  = new KTextEdit(parent);
    TQWhatsThis::add(d->userCommentEdit, i18n("<p>Enter the image's caption. "
                                             "This field is not limited. UTF8 encoding "
                                             "will be used to save the text."));

    d->syncHOSTCommentCheck = new TQCheckBox(i18n("Sync captions entered through %1")
                                            .arg(KApplication::kApplication()->aboutData()->appName()), 
                                            parent);
    d->syncJFIFCommentCheck = new TQCheckBox(i18n("Sync JFIF Comment section"), parent);
    d->syncIPTCCaptionCheck = new TQCheckBox(i18n("Sync IPTC caption (warning: limited to 2000 printable "
                                                 "Ascii characters set)"), parent);

    vlay->addWidget(d->userCommentCheck);
    vlay->addWidget(d->userCommentEdit);
    vlay->addWidget(d->syncHOSTCommentCheck);
    vlay->addWidget(d->syncJFIFCommentCheck);
    vlay->addWidget(d->syncIPTCCaptionCheck);

    // --------------------------------------------------------

    KActiveLabel *note = new KActiveLabel(i18n("<b>Note: "
                 "<b><a href='http://en.wikipedia.org/wiki/EXIF'>EXIF</a></b> "
                 "text tags marked by (*) only support printable "
                 "<b><a href='http://en.wikipedia.org/wiki/Ascii'>ASCII</a></b> "
                 "characters set.</b>"), parent);
    vlay->addWidget(note);
    vlay->addStretch();

    // --------------------------------------------------------

    connect(d->documentNameCheck, TQT_SIGNAL(toggled(bool)),
            d->documentNameEdit, TQT_SLOT(setEnabled(bool)));

    connect(d->imageDescCheck, TQT_SIGNAL(toggled(bool)),
            d->imageDescEdit, TQT_SLOT(setEnabled(bool)));

    connect(d->artistCheck, TQT_SIGNAL(toggled(bool)),
            d->artistEdit, TQT_SLOT(setEnabled(bool)));

    connect(d->copyrightCheck, TQT_SIGNAL(toggled(bool)),
            d->copyrightEdit, TQT_SLOT(setEnabled(bool)));

    connect(d->userCommentCheck, TQT_SIGNAL(toggled(bool)),
            d->userCommentEdit, TQT_SLOT(setEnabled(bool)));

    connect(d->userCommentCheck, TQT_SIGNAL(toggled(bool)),
            d->syncJFIFCommentCheck, TQT_SLOT(setEnabled(bool)));

    connect(d->userCommentCheck, TQT_SIGNAL(toggled(bool)),
            d->syncHOSTCommentCheck, TQT_SLOT(setEnabled(bool)));

    connect(d->userCommentCheck, TQT_SIGNAL(toggled(bool)),
            d->syncIPTCCaptionCheck, TQT_SLOT(setEnabled(bool)));

    // --------------------------------------------------------

    connect(d->documentNameCheck, TQT_SIGNAL(toggled(bool)),
            this, TQT_SIGNAL(signalModified()));
    
    connect(d->imageDescCheck, TQT_SIGNAL(toggled(bool)),
            this, TQT_SIGNAL(signalModified()));

    connect(d->artistCheck, TQT_SIGNAL(toggled(bool)),
            this, TQT_SIGNAL(signalModified()));

    connect(d->copyrightCheck, TQT_SIGNAL(toggled(bool)),
            this, TQT_SIGNAL(signalModified()));

    connect(d->userCommentCheck, TQT_SIGNAL(toggled(bool)),
            this, TQT_SIGNAL(signalModified()));

    // --------------------------------------------------------

    connect(d->userCommentEdit, TQT_SIGNAL(textChanged()),
            this, TQT_SIGNAL(signalModified()));
    
    connect(d->documentNameEdit, TQT_SIGNAL(textChanged(const TQString &)),
            this, TQT_SIGNAL(signalModified()));

    connect(d->imageDescEdit, TQT_SIGNAL(textChanged(const TQString &)),
            this, TQT_SIGNAL(signalModified()));

    connect(d->artistEdit, TQT_SIGNAL(textChanged(const TQString &)),
            this, TQT_SIGNAL(signalModified()));

    connect(d->copyrightEdit, TQT_SIGNAL(textChanged(const TQString &)),
            this, TQT_SIGNAL(signalModified()));
}

EXIFCaption::~EXIFCaption()
{
    delete d;
}

bool EXIFCaption::syncJFIFCommentIsChecked()
{
    return d->syncJFIFCommentCheck->isChecked();
}

bool EXIFCaption::syncHOSTCommentIsChecked()
{
    return d->syncHOSTCommentCheck->isChecked();
}

bool EXIFCaption::syncIPTCCaptionIsChecked()
{
    return d->syncIPTCCaptionCheck->isChecked();
}

TQString EXIFCaption::getEXIFUserComments()
{
    return d->userCommentEdit->text();
}

void EXIFCaption::setCheckedSyncJFIFComment(bool c)
{
    d->syncJFIFCommentCheck->setChecked(c);
}

void EXIFCaption::setCheckedSyncHOSTComment(bool c)
{
    d->syncHOSTCommentCheck->setChecked(c);
}

void EXIFCaption::setCheckedSyncIPTCCaption(bool c)
{
    d->syncIPTCCaptionCheck->setChecked(c);
}

void EXIFCaption::readMetadata(TQByteArray& exifData)
{
    blockSignals(true);
    KExiv2Iface::KExiv2 exiv2Iface;
    exiv2Iface.setExif(exifData);
    TQString data;

    d->documentNameEdit->clear();
    d->documentNameCheck->setChecked(false);
    data = exiv2Iface.getExifTagString("Exif.Image.DocumentName", false);    
    if (!data.isNull())
    {
        d->documentNameEdit->setText(data);
        d->documentNameCheck->setChecked(true);
    }
    d->documentNameEdit->setEnabled(d->documentNameCheck->isChecked());

    d->imageDescEdit->clear();
    d->imageDescCheck->setChecked(false);
    data = exiv2Iface.getExifTagString("Exif.Image.ImageDescription", false);     
    if (!data.isNull())
    {
        d->imageDescEdit->setText(data);
        d->imageDescCheck->setChecked(true);
    }
    d->imageDescEdit->setEnabled(d->imageDescCheck->isChecked());

    d->artistEdit->clear();
    d->artistCheck->setChecked(false);
    data = exiv2Iface.getExifTagString("Exif.Image.Artist", false);    
    if (!data.isNull())
    {
        d->artistEdit->setText(data);
        d->artistCheck->setChecked(true);
    }
    d->artistEdit->setEnabled(d->artistCheck->isChecked());

    d->copyrightEdit->clear();
    d->copyrightCheck->setChecked(false);
    data = exiv2Iface.getExifTagString("Exif.Image.Copyright", false);    
    if (!data.isNull())
    {
        d->copyrightEdit->setText(data);
        d->copyrightCheck->setChecked(true);
    }
    d->copyrightEdit->setEnabled(d->copyrightCheck->isChecked());

    d->userCommentEdit->clear();
    d->userCommentCheck->setChecked(false);
    data = exiv2Iface.getExifComment();    
    if (!data.isNull())
    {
        d->userCommentEdit->setText(data);
        d->userCommentCheck->setChecked(true);
    }
    d->userCommentEdit->setEnabled(d->userCommentCheck->isChecked());
    d->syncJFIFCommentCheck->setEnabled(d->userCommentCheck->isChecked());
    d->syncHOSTCommentCheck->setEnabled(d->userCommentCheck->isChecked());
    d->syncIPTCCaptionCheck->setEnabled(d->userCommentCheck->isChecked());

    blockSignals(false);
}

void EXIFCaption::applyMetadata(TQByteArray& exifData, TQByteArray& iptcData)
{
    KExiv2Iface::KExiv2 exiv2Iface;
    exiv2Iface.setExif(exifData);
    exiv2Iface.setIptc(iptcData);

    if (d->documentNameCheck->isChecked())
        exiv2Iface.setExifTagString("Exif.Image.DocumentName", d->documentNameEdit->text());
    else
        exiv2Iface.removeExifTag("Exif.Image.DocumentName");

    if (d->imageDescCheck->isChecked())
        exiv2Iface.setExifTagString("Exif.Image.ImageDescription", d->imageDescEdit->text());
    else
        exiv2Iface.removeExifTag("Exif.Image.ImageDescription");

    if (d->artistCheck->isChecked())
        exiv2Iface.setExifTagString("Exif.Image.Artist", d->artistEdit->text());
    else
        exiv2Iface.removeExifTag("Exif.Image.Artist");

    if (d->copyrightCheck->isChecked())
        exiv2Iface.setExifTagString("Exif.Image.Copyright", d->copyrightEdit->text());
    else
        exiv2Iface.removeExifTag("Exif.Image.Copyright");

    if (d->userCommentCheck->isChecked())
    {
        exiv2Iface.setExifComment(d->userCommentEdit->text());
        
        if (syncJFIFCommentIsChecked())
            exiv2Iface.setComments(d->userCommentEdit->text().utf8());
        
        if (syncIPTCCaptionIsChecked())
            exiv2Iface.setIptcTagString("Iptc.Application2.Caption", d->userCommentEdit->text());
    }
    else
        exiv2Iface.removeExifTag("Exif.Photo.UserComment");

    exiv2Iface.setImageProgramId(TQString("Kipi-plugins"), TQString(kipiplugins_version));

    exifData = exiv2Iface.getExif();
    iptcData = exiv2Iface.getIptc();
}

}  // namespace KIPIMetadataEditPlugin
