/**********************************************************************
** Copyright (C) 2000 Trolltech AS.  All rights reserved.
**
** This file is part of TQt Designer.
**
** This file may be distributed and/or modified under the terms of the
** GNU General Public License version 2 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
** See http://www.trolltech.com/gpl/ for GPL licensing information.
**
** Contact info@trolltech.com if any conditions of this licensing are
** not clear to you.
**
**********************************************************************/

#include <tqmenudata.h>
#include "resource.h"
#include "defs.h"
#include "metadatabase.h"
#include "formwindow.h"
#include "mainwindow.h"
#include "tqdom.h"
#include "widgetdatabase.h"
#include "widgetfactory.h"
#include "tqlayout.h"
#include "domtool.h"
#include "command.h"
#include "pixmapchooser.h"
#ifndef TQT_NO_SQL
#include "database.h"
#endif
#include "actiondnd.h"
#ifndef KOMMANDER
#include "project.h"
#include "pixmapcollection.h"
#endif
#include "formfile.h"

#include <tqfeatures.h>
#include <tqfile.h>
#include <tqtextstream.h>
#include <tqobject.h>
#include <tqwidget.h>
#include <tqobjectlist.h>
#include <tqmetaobject.h>
#include <tqworkspace.h>
#include <tqtabwidget.h>
#include <tqapplication.h>
#include <tqbuffer.h>
#include <tqlayout.h>
#include <tqtooltip.h>
#include <tqwhatsthis.h>
#include <tqlistbox.h>
#include <tqcombobox.h>
#include <tqwidgetstack.h>
#include <tqtabbar.h>
#include <tqheader.h>
#include <tqlistview.h>
#include <tqiconview.h>
#include <tqlabel.h>
#include <tqwizard.h>
#include <tqtextcodec.h>
#include <tqregexp.h>
#include <zlib.h>
#include <tqdatetime.h>
#ifndef TQT_NO_TABLE
#include <tqtable.h>
#endif

#include <klocale.h>

static TQString makeIndent( int indent )
{
    TQString s;
    s.fill( ' ', indent * 4 );
    return s;
}

static TQString entitize( const TQString &s, bool attribute = false )
{
    TQString s2 = s;
    s2 = s2.replace( TQRegExp( "&" ), "&amp;" );
    s2 = s2.replace( TQRegExp( ">" ), "&gt;" );
    s2 = s2.replace( TQRegExp( "<" ), "&lt;" );
    if ( attribute ) {
	s2 = s2.replace( TQRegExp( "\"" ), "&quot;" );
	s2 = s2.replace( TQRegExp( "'" ), "&apos;" );
    }
    return s2;
}

static TQString mkBool( bool b )
{
    return b? "true" : "false";
}

/*!
  \class Resource resource.h
  \brief Class for saving/loading, etc. forms

  This class is used for saving and loading forms, code generation,
  transferring data of widgets over the clipboard, etc..

*/


Resource::Resource()
{
    mainwindow = 0;
    formwindow = 0;
    toplevel = 0;
    copying = false;
    pasting = false;
    hadGeometry = false;
#ifndef KOMMANDER
    langIface = 0;
#endif
    hasFunctions = false;
}

Resource::Resource( MainWindow* mw )
    : mainwindow( mw )
{
    formwindow = 0;
    toplevel = 0;
    copying = false;
    pasting = false;
    hadGeometry = false;
#ifndef KOMMANDER
    langIface = 0;
#endif
    hasFunctions = false;
}

Resource::~Resource()
{
#ifndef KOMMANDER
    if ( langIface )
	langIface->release();
#endif
}

void Resource::setWidget( FormWindow *w )
{
    formwindow = w;
    toplevel = w;
}

TQWidget *Resource::widget() const
{
    return toplevel;
}

bool Resource::load( FormFile *ff )
{
    if ( !ff || ff->absFileName().isEmpty() )
	return false;
    currFileName = ff->absFileName();
    mainContainerSet = false;

    TQFile f( ff->absFileName() );
    f.open( IO_ReadOnly );

    bool b = load( ff, TQT_TQIODEVICE(&f) );
    f.close();

    return b;
}

bool Resource::load( FormFile *ff, TQIODevice* dev )
{
    TQDomDocument doc;
    TQString errMsg;
    int errLine;
    TQTextStream stream(dev);
    TQString content = stream.read();
    if (content.startsWith("#!"))
    {
      content = content.mid(content.find('\n'));
    }
    if ( !doc.setContent( content) ) {
	// qDebug( TQString("Parse error: ") + errMsg + TQString(" in line %d"), errLine );
	return false;
    }

    DomTool::fixDocument( doc );

    toplevel = formwindow = new FormWindow( ff, mainwindow->qWorkspace(), 0 );
#ifndef KOMMANDER
    formwindow->setProject( MainWindow::self->currProject() );
#endif
    formwindow->setMainWindow( mainwindow );
    MetaDataBase::addEntry( TQT_TQOBJECT(formwindow) );

#ifndef KOMMANDER
    if (!langIface) {
	langIface = MetaDataBase::languageInterface( mainwindow->currProject()->language() );
	if ( langIface )
	    langIface->addRef();
    }
#endif

    TQDomElement e = doc.firstChild().toElement().firstChild().toElement();

    TQDomElement forwards = e;
    while ( forwards.tagName() != "forwards" && !forwards.isNull() )
	forwards = forwards.nextSibling().toElement();

    TQDomElement includes = e;
    while ( includes.tagName() != "includes" && !includes.isNull() )
	includes = includes.nextSibling().toElement();

    TQDomElement variables = e;
    while ( variables.tagName() != "variables" && !variables.isNull() )
	variables = variables.nextSibling().toElement();

    TQDomElement eltSignals = e;
    while ( eltSignals.tagName() != "signals" && !eltSignals.isNull() )
	eltSignals = eltSignals.nextSibling().toElement();

    TQDomElement eltSlots = e;
    while ( eltSlots.tagName() != "slots" && !eltSlots.isNull() )
	eltSlots = eltSlots.nextSibling().toElement();

    TQDomElement connections = e;
    while ( connections.tagName() != "connections" && !connections.isNull() )
	connections = connections.nextSibling().toElement();

    TQDomElement imageCollection = e;
    images.clear();
    while ( imageCollection.tagName() != "images" && !imageCollection.isNull() )
	imageCollection = imageCollection.nextSibling().toElement();

    TQDomElement customWidgets = e;
    while ( customWidgets.tagName() != "customwidgets" && !customWidgets.isNull() )
	customWidgets = customWidgets.nextSibling().toElement();

    TQDomElement tabOrder = e;
    while ( tabOrder.tagName() != "tabstops" && !tabOrder.isNull() )
	tabOrder = tabOrder.nextSibling().toElement();

    TQDomElement actions = e;
    while ( actions.tagName() != "actions" && !actions.isNull() )
	actions = actions.nextSibling().toElement();

    TQDomElement toolbars = e;
    while ( toolbars.tagName() != "toolbars" && !toolbars.isNull() )
	toolbars = toolbars.nextSibling().toElement();

    TQDomElement menubar = e;
    while ( menubar.tagName() != "menubar" && !menubar.isNull() )
	menubar = menubar.nextSibling().toElement();

    TQDomElement functions = e;
    while ( functions.tagName() != "functions" && !functions.isNull() )
	functions = functions.nextSibling().toElement();


    TQDomElement widget;
    while ( !e.isNull() ) {
	if ( e.tagName() == "widget" ) {
	    widget = e;
	} else if ( e.tagName() == "include" ) { // compatibility with 2.x
	    MetaDataBase::Include inc;
	    inc.location = "global";
	    if ( e.attribute( "location" ) == "local" )
		inc.location = "local";
	    inc.implDecl = "in declaration";
	    if ( e.attribute( "impldecl" ) == "in implementation" )
		inc.implDecl = "in implementation";
	    inc.header = e.firstChild().toText().data();
	    if ( inc.header.right( 5 ) != ".ui.h" )
		metaIncludes.append( inc );
	} else if ( e.tagName() == "comment" ) {
	    metaInfo.comment = e.firstChild().toText().data();
	} else if ( e.tagName() == "forward" ) { // compatibility with old betas
	    metaForwards << e.firstChild().toText().data();
	} else if ( e.tagName() == "variable" ) { // compatibility with old betas
	    metaVariables << e.firstChild().toText().data();
	} else if ( e.tagName() == "author" ) {
	    metaInfo.author = e.firstChild().toText().data();
  } else if ( e.tagName() == "version" ) {
    metaInfo.version = e.firstChild().toText().data();
  } else if ( e.tagName() == "license" ) {
    metaInfo.license = e.firstChild().toText().data();
	} else if ( e.tagName() == "exportmacro" ) {
	    exportMacro = e.firstChild().toText().data();
	} else if ( e.tagName() == "layoutdefaults" ) {
	    formwindow->setLayoutDefaultSpacing( e.attribute( "spacing", TQString::number( formwindow->layoutDefaultSpacing() ) ).toInt() );
	    formwindow->setLayoutDefaultMargin( e.attribute( "margin", TQString::number( formwindow->layoutDefaultMargin() ) ).toInt() );
	}

	e = e.nextSibling().toElement();
    }

    if ( !imageCollection.isNull() )
	loadImageCollection( imageCollection );
    if ( !customWidgets.isNull() )
	loadCustomWidgets( customWidgets, this );

#if defined (TQT_NON_COMMERCIAL)
    bool previewMode = MainWindow::self->isPreviewing();
    TQWidget *w = (TQWidget*)createObject( widget, !previewMode ? (TQWidget*)formwindow : MainWindow::self);
    if ( !w )
	return false;
    if ( previewMode )
	w->reparent( MainWindow::self, TQt::WType_TopLevel,  w->pos(), true );
#else
    if ( !createObject( widget, formwindow) )
	return false;
#endif

    if ( !forwards.isNull() ) {
	for ( TQDomElement n = forwards.firstChild().toElement(); !n.isNull(); n = n.nextSibling().toElement() )
	    if ( n.tagName() == "forward" )
		metaForwards << n.firstChild().toText().data();
    }

    if ( !includes.isNull() ) {
	for ( TQDomElement n = includes.firstChild().toElement(); !n.isNull(); n = n.nextSibling().toElement() )
	    if ( n.tagName() == "include" ) {
		if ( n.tagName() == "include" ) {
		    MetaDataBase::Include inc;
		    inc.location = "global";
		    if ( n.attribute( "location" ) == "local" )
			inc.location = "local";
		    inc.implDecl = "in declaration";
		    if ( n.attribute( "impldecl" ) == "in implementation" )
			inc.implDecl = "in implementation";
		    inc.header = n.firstChild().toText().data();
		    if ( inc.header.right( 5 ) != ".ui.h" )
			metaIncludes.append( inc );
		}
	    }
    }

    if ( !variables.isNull() ) {
	for ( TQDomElement n = variables.firstChild().toElement(); !n.isNull(); n = n.nextSibling().toElement() )
	    if ( n.tagName() == "variable" )
		metaVariables << n.firstChild().toText().data();
    }
    if ( !eltSignals.isNull() ) {
	for ( TQDomElement n = eltSignals.firstChild().toElement(); !n.isNull(); n = n.nextSibling().toElement() )
	    if ( n.tagName() == "signal" )
		metaSignals << n.firstChild().toText().data();
    }
    if ( !eltSlots.isNull() ) {
	for ( TQDomElement n = eltSlots.firstChild().toElement(); !n.isNull(); n = n.nextSibling().toElement() )
	    if ( n.tagName() == "slot" ) {
		MetaDataBase::Slot slot;
		slot.specifier = n.attribute( "specifier", "virtual" );
		if ( slot.specifier.isEmpty() )
		    slot.specifier = "virtual";
		slot.access = n.attribute( "access", "public" );
		if ( slot.access.isEmpty() )
		    slot.access = "public";
		slot.language = n.attribute( "language", "C++" );
		slot.returnType = n.attribute( "returnType", "void" );
		if ( slot.returnType.isEmpty() )
		    slot.returnType = "void";
		slot.slot = n.firstChild().toText().data();
		if ( !MetaDataBase::hasSlot( TQT_TQOBJECT(formwindow), slot.slot, true ) )
		    MetaDataBase::addSlot( TQT_TQOBJECT(formwindow), slot.slot, slot.specifier,
					   slot.access, slot.language, slot.returnType );
		else
		    MetaDataBase::changeSlotAttributes( TQT_TQOBJECT(formwindow), slot.slot,
							slot.specifier, slot.access,
							slot.language, slot.returnType );
	    }
    }
    if ( !actions.isNull() )
	loadActions( actions );
    if ( !toolbars.isNull() )
	loadToolBars( toolbars );
    if ( !menubar.isNull() )
	loadMenuBar( menubar );

    if ( !connections.isNull() )
	loadConnections( connections );
    if ( !tabOrder.isNull() )
	loadTabOrder( tabOrder );

    if ( formwindow ) {
	MetaDataBase::setIncludes( TQT_TQOBJECT(formwindow), metaIncludes );
	MetaDataBase::setForwards( TQT_TQOBJECT(formwindow), metaForwards );
	MetaDataBase::setSignalList( TQT_TQOBJECT(formwindow), metaSignals );
	MetaDataBase::setMetaInfo( TQT_TQOBJECT(formwindow), metaInfo );
	MetaDataBase::setExportMacro( TQT_TQOBJECT(formwindow->mainContainer()), exportMacro );
    }


    if ( mainwindow && formwindow )
    {
	mainwindow->insertFormWindow( formwindow );
    }

    if ( formwindow ) {
	formwindow->killAccels( TQT_TQOBJECT(formwindow) );
	if ( formwindow->layout() )
	    formwindow->layout()->activate();
	if ( hadGeometry )
	    formwindow->resize( formwindow->size().expandedTo( formwindow->minimumSize().
							       expandedTo( formwindow->minimumSizeHint() ) ) );
	else
	    formwindow->resize( formwindow->size().expandedTo( formwindow->sizeHint() ) );
    }

    return true;
}

bool Resource::save( const TQString& filename, bool formCodeOnly )
{
    if ( !formwindow || filename.isEmpty() )
	return false;
    currFileName = filename;

    TQFile f( filename );
    if ( !f.open( IO_WriteOnly | IO_Translate ) )
	return false;
    bool b = save( TQT_TQIODEVICE(&f) );
    f.close();
    return b;
}

bool Resource::save( TQIODevice* dev )
{
    if ( !formwindow )
	return false;


    TQTextStream ts( dev );
    ts.setCodec( TQTextCodec::codecForName( "UTF-8" ) );
    if ( formwindow->mainContainer()->property("useShebang").toBool() )
      ts << formwindow->mainContainer()->property("shebang").toString() << endl;
    ts << "<!DOCTYPE UI><UI version=\"3.0\" stdsetdef=\"1\">" << endl;
    saveMetaInfoBefore( ts, 0 );
    saveObject( TQT_TQOBJECT(formwindow->mainContainer()), 0, ts, 0 );
    if ( formwindow->mainContainer()->inherits( TQMAINWINDOW_OBJECT_NAME_STRING ) ) {
	saveMenuBar( (TQMainWindow*)formwindow->mainContainer(), ts, 0 );
	saveToolBars( (TQMainWindow*)formwindow->mainContainer(), ts, 0 );
    }
    if ( !MetaDataBase::customWidgets()->isEmpty() && !usedCustomWidgets.isEmpty() )
	saveCustomWidgets( ts, 0 );
    if ( formwindow->mainContainer()->inherits( TQMAINWINDOW_OBJECT_NAME_STRING ) )
	saveActions( formwindow->actionList(), ts, 0 );
    if ( !images.isEmpty() )
	saveImageCollection( ts, 0 );
    if ( !MetaDataBase::connections( TQT_TQOBJECT(formwindow) ).isEmpty() || !MetaDataBase::slotList( TQT_TQOBJECT(formwindow) ).isEmpty() )
	saveConnections( ts, 0 );
    saveTabOrder( ts, 0 );
    saveMetaInfoAfter( ts, 0 );
    ts << "</UI>" << endl;
    images.clear();

    return true;
}

TQString Resource::copy()
{
    if ( !formwindow )
	return TQString();

    copying = true;
    TQString s;
    TQTextOStream ts( &s );

    ts << "<!DOCTYPE UI-SELECTION><UI-SELECTION>" << endl;
    TQWidgetList widgets = formwindow->selectedWidgets();
    TQWidgetList tmp( widgets );
    for ( TQWidget *w = widgets.first(); w; w = widgets.next() ) {
	TQWidget *p = w->parentWidget();
	bool save = true;
	while ( p ) {
	    if ( tmp.findRef( p ) != -1 ) {
		save = false;
		break;
	    }
	    p = p->parentWidget();
	}
	if ( save )
	    saveObject( TQT_TQOBJECT(w), 0, ts, 0 );
    }
    if ( !MetaDataBase::customWidgets()->isEmpty() && !usedCustomWidgets.isEmpty() )
	saveCustomWidgets( ts, 0 );
    if ( !images.isEmpty() )
	saveImageCollection( ts, 0 );
    ts << "</UI-SELECTION>" << endl;

    return s;
}


void Resource::paste( const TQString &cb, TQWidget *parent )
{
    if ( !formwindow )
	return;
    mainContainerSet = true;

    pasting = true;
    TQBuffer buf( TQCString( cb.utf8() ) );
    buf.open( IO_ReadOnly );
    TQDomDocument doc;
    TQString errMsg;
    int errLine;
    if ( !doc.setContent( &buf, &errMsg, &errLine ) ) {
	// qDebug( TQString("Parse error: ") + errMsg + TQString(" in line %d"), errLine );
    }

    TQDomElement firstWidget = doc.firstChild().toElement().firstChild().toElement();

    TQDomElement imageCollection = firstWidget;
    images.clear();
    while ( imageCollection.tagName() != "images" && !imageCollection.isNull() )
	imageCollection = imageCollection.nextSibling().toElement();

    TQDomElement customWidgets = firstWidget;
    while ( customWidgets.tagName() != "customwidgets" && !customWidgets.isNull() )
	customWidgets = customWidgets.nextSibling().toElement();

    if ( !imageCollection.isNull() )
	loadImageCollection( imageCollection );
    if ( !customWidgets.isNull() )
	loadCustomWidgets( customWidgets, this );

    TQWidgetList widgets;
    formwindow->clearSelection( false );
    formwindow->setPropertyShowingBlocked( true );
    formwindow->clearSelection( false );
    while ( !firstWidget.isNull() ) {
	if ( firstWidget.tagName() == "widget" ) {
	    TQWidget *w = (TQWidget*)createObject( firstWidget, parent, 0 );
	    if ( !w )
		continue;
	    widgets.append( w );
	    int x = w->x() + formwindow->grid().x();
	    int y = w->y() + formwindow->grid().y();
	    if ( w->x() + w->width() > parent->width() )
		x = TQMAX( 0, parent->width() - w->width() );
	    if ( w->y() + w->height() > parent->height() )
		y = TQMAX( 0, parent->height() - w->height() );
	    if ( x != w->x() || y != w->y() )
		w->move( x, y );
	    formwindow->selectWidget( TQT_TQOBJECT(w) );
	} else if ( firstWidget.tagName() == "spacer" ) {
	    TQWidget *w = createSpacer( firstWidget, parent, 0, firstWidget.tagName() == "vspacer" ? Qt::Vertical : Qt::Horizontal );
	    if ( !w )
		continue;
	    widgets.append( w );
	    int x = w->x() + formwindow->grid().x();
	    int y = w->y() + formwindow->grid().y();
	    if ( w->x() + w->width() > parent->width() )
		x = TQMAX( 0, parent->width() - w->width() );
	    if ( w->y() + w->height() > parent->height() )
		y = TQMAX( 0, parent->height() - w->height() );
	    if ( x != w->x() || y != w->y() )
		w->move( x, y );
	    formwindow->selectWidget( TQT_TQOBJECT(w) );
	}
	firstWidget = firstWidget.nextSibling().toElement();
    }
    formwindow->setPropertyShowingBlocked( false );
    formwindow->emitShowProperties();
    buf.close();

    PasteCommand *cmd = new PasteCommand( i18n("Paste" ), formwindow, widgets );
    formwindow->commandHistory()->addCommand( cmd );
}

void Resource::saveObject( TQObject *obj, QDesignerGridLayout* grid, TQTextStream &ts, int indent )
{
    if ( obj && obj->isWidgetType() && ( (TQWidget*)obj )->isHidden() && !(obj->isA("ScriptObject")) )
	return;
    TQString closeTag;
    if ( obj->isWidgetType() ) {
	const char* className = WidgetFactory::classNameOf( obj );
	if ( obj->isA( "CustomWidget" ) )
	    usedCustomWidgets << TQString( className );
	if ( TQT_BASE_OBJECT(obj) != TQT_BASE_OBJECT(formwindow) && !formwindow->widgets()->find( (TQWidget*)obj ) )
	    return; // we don't know anything about this thing

	TQString attributes;
	if ( grid ) {
	    QDesignerGridLayout::Item item = grid->items[ (TQWidget*)obj ];
	    attributes += TQString(" row=\"") + TQString::number(item.row) + "\"";
	    attributes += TQString(" column=\"") + TQString::number(item.column) + "\"";
	    if ( item.rowspan * item.colspan != 1 ) {
		attributes += TQString(" rowspan=\"") + TQString::number(item.rowspan) + "\"";
		attributes += TQString(" colspan=\"") + TQString::number(item.colspan) + "\"";
	    }
	}

	if ( qstrcmp( className, "Spacer" ) == 0 ) {
	    closeTag = makeIndent( indent ) + "</spacer>\n";
	    ts << makeIndent( indent ) << "<spacer" << attributes << ">" << endl;
	    ++indent;
	} else {
	    closeTag = makeIndent( indent ) + "</widget>\n";
        if ( qstrcmp( className, "KmdrMainWindow" ) == 0 ) {
          ts << makeIndent( indent ) << "<widget class=\"TQMainWindow\"" << attributes << ">" << endl;
        } else {
	       ts << makeIndent( indent ) << "<widget class=\"" << className << "\"" << attributes << ">" << endl;
        }
	    ++indent;
	}
	if ( WidgetFactory::hasItems( WidgetDatabase::idFromClassName( WidgetFactory::classNameOf( obj ) ) ) )
	    saveItems( obj, ts, indent );
	saveObjectProperties( obj, ts, indent );
    } else {
	// test for other objects we created. Nothing so far.
	return;
    }

    if ( obj->inherits( TQTABWIDGET_OBJECT_NAME_STRING ) ) {
	TQTabWidget* tw = (TQTabWidget*) obj;
	TQObjectList* tmpl = tw->queryList( TQWIDGETSTACK_OBJECT_NAME_STRING );
	TQWidgetStack *ws = (TQWidgetStack*)tmpl->first();
	TQTabBar *tb = ( (QDesignerTabWidget*)obj )->tabBar();
	for ( int i = 0; i < tb->count(); ++i ) {
	    TQTab *t = tb->tabAt( i );
	    if ( !t )
		continue;
	    TQWidget *w = ws->widget( t->identifier() );
	    if ( !w )
		continue;
	    if ( WidgetDatabase::idFromClassName( WidgetFactory::classNameOf( TQT_TQOBJECT(w) ) ) == -1 )
		continue; // we don't know this widget
	    ts << makeIndent( indent ) << "<widget class=\"TQWidget\">" << endl;
	    ++indent;
	    ts << makeIndent( indent ) << "<property name=\"name\">" << endl;
	    indent++;
	    ts << makeIndent( indent ) << "<cstring>" << entitize( w->name() ) << "</cstring>" << endl;
	    indent--;
	    ts << makeIndent( indent ) << "</property>" << endl;

	    ts << makeIndent( indent ) << "<attribute name=\"title\">" << endl;
	    indent++;
	    ts << makeIndent( indent ) << "<string>" << entitize( t->text() ) << "</string>" << endl;
	    indent--;
	    ts << makeIndent( indent ) << "</attribute>" << endl;
	    saveChildrenOf( TQT_TQOBJECT(w), ts, indent );
	    --indent;
	    ts << makeIndent( indent ) << "</widget>" << endl;
	}
	delete tmpl;
    } else
    if ( ::tqqt_cast<TQToolBox*>(obj) ) {
        TQToolBox* tb = (TQToolBox*)obj;
        for ( int i = 0; i < tb->count(); ++i ) {
            TQWidget *w = tb->item( i );
            if ( !w )
                continue;
            if ( WidgetDatabase::idFromClassName(WidgetFactory::classNameOf(TQT_TQOBJECT(w))) == -1 )
                continue; // we don't know this widget
            ts << makeIndent( indent ) << "<widget class=\"TQWidget\">" << endl;
            ++indent;
            ts << makeIndent( indent ) << "<property name=\"name\">" << endl;
            indent++;
            ts << makeIndent( indent ) << "<cstring>" << entitize( w->name() ) << "</cstring>" << endl;
            indent--;
            ts << makeIndent( indent ) << "</property>" << endl;
            ts << makeIndent( indent ) << "<property name=\"backgroundMode\">" << endl;
            indent++;
            saveEnumProperty( TQT_TQOBJECT(w), "backgroundMode", TQVariant::Invalid, ts, indent );
            indent--;
            ts << makeIndent( indent ) << "</property>" << endl;
            ts << makeIndent( indent ) << "<attribute name=\"label\">" << endl;
            indent++;
            ts << makeIndent( indent ) << "<string>" << entitize( tb->itemLabel( tb->indexOf(w) ) ) << "</string>" << endl;
            indent--;
            ts << makeIndent( indent ) << "</attribute>" << endl;
            saveChildrenOf( TQT_TQOBJECT(w), ts, indent );
            --indent;
            ts << makeIndent( indent ) << "</widget>" << endl;
        }
    }
    else if ( obj->inherits( TQWIZARD_OBJECT_NAME_STRING ) ) {
	TQWizard* wiz = (TQWizard*)obj; for ( int i = 0; i < wiz->pageCount(); ++i ) {
	    TQWidget *w = wiz->page( i );
	    if ( !w )
		continue;
	    if ( WidgetDatabase::idFromClassName( WidgetFactory::classNameOf( TQT_TQOBJECT(w) ) ) == -1 )
		continue; // we don't know this widget
	    ts << makeIndent( indent ) << "<widget class=\"TQWidget\">" << endl;
	    ++indent;
	    ts << makeIndent( indent ) << "<property name=\"name\">" << endl;
	    indent++;
	    ts << makeIndent( indent ) << "<cstring>" << entitize( w->name() ) << "</cstring>" << endl;
	    indent--;
	    ts << makeIndent( indent ) << "</property>" << endl;

	    ts << makeIndent( indent ) << "<attribute name=\"title\">" << endl;
	    indent++;
	    ts << makeIndent( indent ) << "<string>" << entitize( wiz->title( w ) ) << "</string>" << endl;
	    indent--;
	    ts << makeIndent( indent ) << "</attribute>" << endl;
	    saveChildrenOf( TQT_TQOBJECT(w), ts, indent );
	    --indent;
	    ts << makeIndent( indent ) << "</widget>" << endl;
	}
    } else if ( obj->inherits( TQMAINWINDOW_OBJECT_NAME_STRING ) ) {
	saveChildrenOf( TQT_TQOBJECT(( (TQMainWindow*)obj )->centralWidget()), ts, indent );
    } else {
	saveChildrenOf( obj, ts, indent );
    }

    indent--;
    ts << closeTag;
}

void Resource::saveItems( TQObject *obj, TQTextStream &ts, int indent )
{
    if ( obj->inherits( TQLISTBOX_OBJECT_NAME_STRING ) || obj->inherits( TQCOMBOBOX_OBJECT_NAME_STRING ) ) {
	TQListBox *lb = 0;
	if ( obj->inherits( TQLISTBOX_OBJECT_NAME_STRING ) )
	    lb = (TQListBox*)obj;
	else
	    lb = ( (TQComboBox*)obj )->listBox();

	TQListBoxItem *i = lb->firstItem();
	for ( ; i; i = i->next() ) {
	    ts << makeIndent( indent ) << "<item>" << endl;
	    indent++;
	    TQStringList text;
	    text << i->text();
	    TQPtrList<TQPixmap> pixmaps;
	    if ( i->pixmap() )
		pixmaps.append( i->pixmap() );
	    saveItem( text, pixmaps, ts, indent );
	    indent--;
	    ts << makeIndent( indent ) << "</item>" << endl;
	}
    } else if ( obj->inherits( TQICONVIEW_OBJECT_NAME_STRING ) ) {
	TQIconView *iv = (TQIconView*)obj;

	TQIconViewItem *i = iv->firstItem();
	for ( ; i; i = i->nextItem() ) {
	    ts << makeIndent( indent ) << "<item>" << endl;
	    indent++;
	    TQStringList text;
	    text << i->text();
	    TQPtrList<TQPixmap> pixmaps;
	    if ( i->pixmap() )
		pixmaps.append( i->pixmap() );
	    saveItem( text, pixmaps, ts, indent );
	    indent--;
	    ts << makeIndent( indent ) << "</item>" << endl;
	}
    } else if ( obj->inherits( TQLISTVIEW_OBJECT_NAME_STRING ) ) {
	TQListView *lv = (TQListView*)obj;
	int i;
	for ( i = 0; i < lv->header()->count(); ++i ) {
	    ts << makeIndent( indent ) << "<column>" << endl;
	    indent++;
	    TQStringList l;
	    l << lv->header()->label( i );
	    TQPtrList<TQPixmap> pix;
	    pix.setAutoDelete( true );
	    if ( lv->header()->iconSet( i ) )
		pix.append( new TQPixmap( lv->header()->iconSet( i )->pixmap() ) );
	    saveItem( l, pix, ts, indent );
	    ts << makeIndent( indent ) << "<property name=\"clickable\">" << endl;
	    indent++;
	    ts << makeIndent( indent ) << "<bool>" << mkBool( lv->header()->isClickEnabled( i ) )<< "</bool>" << endl;
	    indent--;
	    ts << makeIndent( indent ) << "</property>" << endl;
	    ts << makeIndent( indent ) << "<property name=\"resizeable\">" << endl;
	    indent++;
	    ts << makeIndent( indent ) << "<bool>" << mkBool( lv->header()->isResizeEnabled( i ) ) << "</bool>" << endl;
	    indent--;
	    ts << makeIndent( indent ) << "</property>" << endl;
	    indent--;
	    ts << makeIndent( indent ) << "</column>" << endl;
	}
	saveItem( lv->firstChild(), ts, indent - 1 );
    }
#ifndef TQT_NO_TABLE
    else if ( obj->inherits( TQTABLE_OBJECT_NAME_STRING ) ) {
	TQTable *table = (TQTable*)obj;
	int i;
	TQMap<TQString, TQString> columnFields = MetaDataBase::columnFields( TQT_TQOBJECT(table) );
	bool isDataTable = table->inherits( TQDATATABLE_OBJECT_NAME_STRING );
	for ( i = 0; i < table->horizontalHeader()->count(); ++i ) {
	    if ( !table->horizontalHeader()->label( i ).isNull() &&
		 table->horizontalHeader()->label( i ).toInt() != i + 1 ||
		 table->horizontalHeader()->iconSet( i ) ||
		  isDataTable ) {
		ts << makeIndent( indent ) << "<column>" << endl;
		indent++;
		TQStringList l;
		l << table->horizontalHeader()->label( i );
		TQPtrList<TQPixmap> pix;
		pix.setAutoDelete( true );
		if ( table->horizontalHeader()->iconSet( i ) )
		    pix.append( new TQPixmap( table->horizontalHeader()->iconSet( i )->pixmap() ) );
		saveItem( l, pix, ts, indent );
		if ( table->inherits( TQDATATABLE_OBJECT_NAME_STRING ) && !columnFields.isEmpty() ) {
		    ts << makeIndent( indent ) << "<property name=\"field\">" << endl;
		    indent++;
		    ts << makeIndent( indent ) << "<string>" << entitize( *columnFields.find( l[ 0 ] ) ) << "</string>" << endl;
		    indent--;
		    ts << makeIndent( indent ) << "</property>" << endl;
		}
		indent--;
		ts << makeIndent( indent ) << "</column>" << endl;
	    }
	}
	for ( i = 0; i < table->verticalHeader()->count(); ++i ) {
	    if ( !table->verticalHeader()->label( i ).isNull() &&
		 table->verticalHeader()->label( i ).toInt() != i + 1 ||
		 table->verticalHeader()->iconSet( i ) ) {
		ts << makeIndent( indent ) << "<row>" << endl;
		indent++;
		TQStringList l;
		l << table->verticalHeader()->label( i );
		TQPtrList<TQPixmap> pix;
		pix.setAutoDelete( true );
		if ( table->verticalHeader()->iconSet( i ) )
		    pix.append( new TQPixmap( table->verticalHeader()->iconSet( i )->pixmap() ) );
		saveItem( l, pix, ts, indent );
		indent--;
		ts << makeIndent( indent ) << "</row>" << endl;
	    }
	}
    }
#endif
}

void Resource::saveItem( TQListViewItem *i, TQTextStream &ts, int indent )
{
    TQListView *lv = i->listView();
    while ( i ) {
	ts << makeIndent( indent ) << "<item>" << endl;
	indent++;

	TQPtrList<TQPixmap> pixmaps;
	TQStringList textes;
	for ( int c = 0; c < lv->columns(); ++c ) {
	    pixmaps.append( i->pixmap( c ) );
	    textes << i->text( c );
	}
	saveItem( textes, pixmaps, ts, indent );

	if ( i->firstChild() )
	    saveItem( i->firstChild(), ts, indent );

	indent--;
	ts << makeIndent( indent ) << "</item>" << endl;
	i = i->nextSibling();
    }
}

void Resource::savePixmap( const TQPixmap &p, TQTextStream &ts, int indent, const TQString &tagname )
{
    if ( p.isNull() ) {
	ts << makeIndent( indent ) << "<" + tagname + "></"  + tagname + ">" << endl;
	return;
    }

    if ( formwindow && formwindow->savePixmapInline() )
	ts << makeIndent( indent ) << "<" + tagname + ">" << saveInCollection( p ) << "</" + tagname + ">" << endl;
#ifndef KOMMANDER
    else if ( formwindow && formwindow->savePixmapInProject() )
	ts << makeIndent( indent ) << "<" + tagname + ">" << MetaDataBase::pixmapKey( TQT_TQOBJECT(formwindow), p.serialNumber() )
	   << "</" + tagname + ">" << endl;
#endif
    else
	ts << makeIndent( indent ) << "<" + tagname + ">" << MetaDataBase::pixmapArgument( TQT_TQOBJECT(formwindow), p.serialNumber() )
	   << "</" + tagname + ">" << endl;
}

TQPixmap Resource::loadPixmap( const TQDomElement &e, const TQString &/*tagname*/ )
{
    TQString arg = e.firstChild().toText().data();

    if ( formwindow && formwindow->savePixmapInline() ) {
	TQImage img = loadFromCollection( arg );
	TQPixmap pix;
	pix.convertFromImage( img );
	MetaDataBase::setPixmapArgument( TQT_TQOBJECT(formwindow), pix.serialNumber(), arg );
	return pix;
    }
#ifndef KOMMANDER
    else if ( formwindow && formwindow->savePixmapInProject() ) {
	TQPixmap pix;
	if ( mainwindow && mainwindow->currProject() )
	    pix = mainwindow->currProject()->pixmapCollection()->pixmap( arg );
    }
#endif
    else
    {
	TQPixmap pix;
        pix = PixmapChooser::loadPixmap( "image.xpm" );
	MetaDataBase::setPixmapKey( TQT_TQOBJECT(formwindow), pix.serialNumber(), arg );
	return pix;
    }
    TQPixmap pix = PixmapChooser::loadPixmap( "image.xpm" );
    MetaDataBase::setPixmapArgument( TQT_TQOBJECT(formwindow), pix.serialNumber(), arg );
    return pix;
}

void Resource::saveItem( const TQStringList &text, const TQPtrList<TQPixmap> &pixmaps, TQTextStream &ts, int indent )
{
    TQStringList::ConstIterator it = text.begin();
    for ( ; it != text.end(); ++it ) {
	ts << makeIndent( indent ) << "<property name=\"text\">" << endl;
	indent++;
	ts << makeIndent( indent ) << "<string>" << entitize( *it ) << "</string>" << endl;
	indent--;
	ts << makeIndent( indent ) << "</property>" << endl;
    }

    for ( int i = 0; i < (int)pixmaps.count(); ++i ) {
	TQPixmap *p = ( (TQPtrList<TQPixmap>)pixmaps ).at( i );
	ts << makeIndent( indent ) << "<property name=\"pixmap\">" << endl;
	indent++;
	if ( p )
	    savePixmap( *p, ts, indent );
	else
	    savePixmap( TQPixmap(), ts, indent );
	indent--;
	ts << makeIndent( indent ) << "</property>" << endl;
    }
}

void Resource::saveChildrenOf( TQObject* obj, TQTextStream &ts, int indent )
{
    const TQObjectList l = obj->childrenListObject();
    if ( l.isEmpty() )
	return; // no children to save

    TQString closeTag;
    // if the widget has a layout we pretend that all widget's childs are childs of the layout - makes the structure nicer
    TQLayout *layout = 0;
    QDesignerGridLayout* grid = 0;
    if ( !obj->inherits( TQSPLITTER_OBJECT_NAME_STRING ) &&
	 WidgetDatabase::isContainer( WidgetDatabase::idFromClassName( WidgetFactory::classNameOf( obj ) ) ) &&
	 obj->isWidgetType() &&
	 WidgetFactory::layoutType( (TQWidget*)obj, layout ) != WidgetFactory::NoLayout ) {
	WidgetFactory::LayoutType lay = WidgetFactory::layoutType( (TQWidget*)obj, layout );
	switch ( lay ) {
	case WidgetFactory::HBox:
	    closeTag = makeIndent( indent ) + "</hbox>";
	    ts << makeIndent( indent ) << "<hbox>" << endl;
	    ++indent;
	    break;
	case WidgetFactory::VBox:
	    closeTag = makeIndent( indent ) + "</vbox>";
	    ts << makeIndent( indent ) << "<vbox>" << endl;
	    ++indent;
	    break;
	case WidgetFactory::Grid:
	    closeTag = makeIndent( indent ) + "</grid>";
	    ts << makeIndent( indent ) << "<grid>" << endl;
	    ++indent;
	    grid = (QDesignerGridLayout*) layout;
	    break;
	default:
	    break;
	}

	// save properties of layout
	if ( lay != WidgetFactory::NoLayout )
	    saveObjectProperties( TQT_TQOBJECT(layout), ts, indent );

    }

    for ( TQPtrListIterator<TQObject> it ( l ); it.current(); ++it )
	saveObject( it.current(), grid, ts, indent );
    if ( !closeTag.isEmpty() ) {
	indent--;
	ts << closeTag << endl;
    }
}

void Resource::saveObjectProperties( TQObject *w, TQTextStream &ts, int indent )
{
    TQStringList saved;
    TQStringList changed;
    changed = MetaDataBase::changedProperties( w );
    if ( w->isWidgetType() ) {
	if ( w->inherits( "Spacer" ) ) {
	    if ( !changed.contains( "sizeHint" ) )
		changed << "sizeHint";
	    if ( !changed.contains( "geometry" ) )
		changed << "geometry";
	}
    } else if ( w->inherits( TQLAYOUT_OBJECT_NAME_STRING ) ) { // #### should be cleaner (RS)
	changed << "margin" << "spacing";
    }

    if ( TQT_BASE_OBJECT(w) == TQT_BASE_OBJECT(formwindow->mainContainer()) ) {
	if ( changed.findIndex( "geometry" ) == -1 )
	    changed << "geometry";
	if ( changed.findIndex( "caption" ) == -1 )
	    changed << "caption";
    }

    if ( changed.isEmpty() )
	    return;

    bool inLayout = TQT_BASE_OBJECT(w) != TQT_BASE_OBJECT(formwindow->mainContainer()) && !copying && w->isWidgetType() && ( (TQWidget*)w )->parentWidget() &&
		    WidgetFactory::layoutType( ( (TQWidget*)w )->parentWidget() ) != WidgetFactory::NoLayout;

    TQStrList lst = w->metaObject()->propertyNames( !w->inherits( "Spacer" ) );
    for ( TQPtrListIterator<char> it( lst ); it.current(); ++it ) {
	if ( changed.find( TQString::fromLatin1( it.current() ) ) == changed.end() )
	    continue;
	if ( saved.find( TQString::fromLatin1( it.current() ) ) != saved.end() )
	    continue;
	saved << TQString::fromLatin1( it.current() );
	const TQMetaProperty* p = w->metaObject()->
				 property( w->metaObject()->findProperty( it.current(), true ), true );
	if ( !p || !p->stored( w ) || ( inLayout && qstrcmp( p->name(), "geometry" ) == 0 ) )
	    continue;
	if ( w->inherits( TQLABEL_OBJECT_NAME_STRING ) && qstrcmp( p->name(), "pixmap" ) == 0 &&
	     ( !( (TQLabel*)w )->pixmap() || ( (TQLabel*)w )->pixmap()->isNull() ) )
	    continue;
	if ( w->inherits( "QDesignerMenuBar" ) &&
	     ( qstrcmp( p->name(), "itemName" ) == 0 || qstrcmp( p->name(), "itemNumber" ) == 0 ||
	       qstrcmp( p->name(), "itemText" ) == 0 ) )
	    continue;
	if ( qstrcmp( p->name(), "name" ) == 0 )
	    knownNames << w->property( "name" ).toString();
	ts << makeIndent( indent ) << "<property";
	ts << " name=\"" << it.current() << "\"";
	if ( !p->stdSet() )
	    ts << " stdset=\"0\"";
	ts << ">" << endl;
	indent++;
	if ( p->isSetType() ) {
	    saveSetProperty( w, it.current(), TQVariant::nameToType( p->type() ), ts, indent );
	} else if ( p->isEnumType() ) {
	    saveEnumProperty( w, it.current(), TQVariant::nameToType( p->type() ), ts, indent );
	} else {
	    saveProperty( w, it.current(), w->property( p->name() ), TQVariant::nameToType( p->type() ), ts, indent );
	}
	indent--;
	ts << makeIndent( indent ) << "</property>" << endl;
    }

    if ( w->isWidgetType() && MetaDataBase::fakeProperties( w ) ) {
	TQMap<TQString, TQVariant>* fakeProperties = MetaDataBase::fakeProperties( w );
	for ( TQMap<TQString, TQVariant>::Iterator fake = fakeProperties->begin();
	      fake != fakeProperties->end(); ++fake ) {
	    if ( MetaDataBase::isPropertyChanged( w, fake.key() ) ) {
		if ( w->inherits( "CustomWidget" ) ) {
		    MetaDataBase::CustomWidget *cw = ( (CustomWidget*)w )->customWidget();
		    if ( cw && !cw->hasProperty( fake.key().latin1() ) && fake.key() != "toolTip" && fake.key() != "whatsThis" )
			continue;
		}

		ts << makeIndent( indent ) << "<property name=\"" << fake.key() << "\" stdset=\"0\">" << endl;
		indent++;
		saveProperty( w, fake.key(), *fake, (*fake).type(), ts, indent );
		indent--;
		ts << makeIndent( indent ) << "</property>" << endl;
	    }
	}
    }
}

void Resource::saveSetProperty( TQObject *w, const TQString &name, TQVariant::Type, TQTextStream &ts, int indent )
{
    const TQMetaProperty *p = w->metaObject()->property( w->metaObject()->findProperty( name, true ), true );
    TQStrList l( p->valueToKeys( w->property( name ).toInt() ) );
    TQString v;
    for ( uint i = 0; i < l.count(); ++i ) {
	v += l.at( i );
	if ( i < l.count() - 1 )
	    v += "|";
    }
    ts << makeIndent( indent ) << "<set>" << v << "</set>" << endl;
}

void Resource::saveEnumProperty( TQObject *w, const TQString &name, TQVariant::Type, TQTextStream &ts, int indent )
{
    const TQMetaProperty *p = w->metaObject()->property( w->metaObject()->findProperty( name, true ), true );
    ts << makeIndent( indent ) << "<enum>" << p->valueToKey( w->property( name ).toInt() ) << "</enum>" << endl;
}

void Resource::saveProperty( TQObject *w, const TQString &name, const TQVariant &value, TQVariant::Type t, TQTextStream &ts, int indent )
{
    if ( name == "hAlign" || name =="vAlign" || name == "wordwrap" || name == "layoutMargin" || name =="layoutSpacing" )
	return;
    int num, unum;
    double dob;
    TQString comment;
    if ( w && formwindow->widgets()->find( (TQWidget*)w ) )
	comment = MetaDataBase::propertyComment( w, name );
    switch ( t ) {
    case TQVariant::String:
	ts << makeIndent( indent ) << "<string>" << entitize( value.toString() ) << "</string>" << endl;
	if ( !comment.isEmpty() )
	    ts << makeIndent( indent ) << "<comment>" << entitize( comment ) << "</comment>" << endl;
	break;
    case TQVariant::CString:
	ts << makeIndent( indent ) << "<cstring>" << entitize( value.toCString() ).latin1() << "</cstring>" << endl;
	break;
    case TQVariant::Bool:
	ts << makeIndent( indent ) << "<bool>" << mkBool( value.toBool() ) << "</bool>" << endl;
	break;
    case TQVariant::Int:
	num = value.toInt();
	if ( w && w->inherits( TQLAYOUT_OBJECT_NAME_STRING ) ) {
	    if ( name == "spacing" )
		num = MetaDataBase::spacing( TQT_TQOBJECT(WidgetFactory::containerOfWidget( WidgetFactory::layoutParent( TQT_TQLAYOUT(TQT_BASE_OBJECT(w)) ) )) );
	    else if ( name == "margin" )
		num = MetaDataBase::margin( TQT_TQOBJECT(WidgetFactory::containerOfWidget( WidgetFactory::layoutParent( TQT_TQLAYOUT(TQT_BASE_OBJECT(w)) ) )) );
	}
	ts << makeIndent( indent ) << "<number>" << TQString::number( num ) << "</number>" << endl;
	break;
    case TQVariant::Double:
	dob = value.toDouble();
	ts << makeIndent( indent ) << "<number>" << TQString::number( dob ) << "</number>" << endl;
	break;
    case TQVariant::KeySequence:
	num = value.toInt();
	ts << makeIndent( indent ) << "<number>" << TQString::number( num ) << "</number>" << endl;
	break;
    case TQVariant::UInt:
	unum = value.toUInt();
	if ( w && w->inherits( TQLAYOUT_OBJECT_NAME_STRING ) ) {
	    if ( name == "spacing" )
		num = MetaDataBase::spacing( TQT_TQOBJECT(WidgetFactory::layoutParent( TQT_TQLAYOUT(TQT_BASE_OBJECT(w)) )) );
	    else if ( name == "margin" )
		num = MetaDataBase::margin( TQT_TQOBJECT(WidgetFactory::layoutParent( TQT_TQLAYOUT(TQT_BASE_OBJECT(w)) )) );
	}
	ts << makeIndent( indent ) << "<number>" << TQString::number( unum ) << "</number>" << endl;
	break;
    case TQVariant::Rect: {
	TQVariant v( value );
	ts << makeIndent( indent ) << "<rect>" << endl;
	indent++;
	ts << makeIndent( indent ) << "<x>" << TQString::number( v.toRect().x() ) << "</x>" << endl;
	ts << makeIndent( indent ) << "<y>" << TQString::number( v.toRect().y() ) << "</y>" << endl;
	ts << makeIndent( indent ) << "<width>" << TQString::number( v.toRect().width() ) << "</width>" << endl;
	ts << makeIndent( indent ) << "<height>" << TQString::number( v.toRect().height() ) << "</height>" << endl;
	indent--;
	ts << makeIndent( indent ) << "</rect>" << endl;
    } break;
    case TQVariant::Point: {
	TQVariant v( value );
	ts << makeIndent( indent ) << "<point>" << endl;
	indent++;
	ts << makeIndent( indent ) << "<x>" << TQString::number( v.toPoint().x() ) << "</x>" << endl;
	ts << makeIndent( indent ) << "<y>" << TQString::number( v.toPoint().y() ) << "</y>" << endl;
	indent--;
	ts << makeIndent( indent ) << "</point>" << endl;
    } break;
    case TQVariant::Size: {
	TQVariant v( value );
	ts << makeIndent( indent ) << "<size>" << endl;
	indent++;
	ts << makeIndent( indent ) << "<width>" << TQString::number( v.toSize().width() ) << "</width>" << endl;
	ts << makeIndent( indent ) << "<height>" << TQString::number( v.toSize().height() ) << "</height>" << endl;
	indent--;
	ts << makeIndent( indent ) << "</size>" << endl;
    } break;
    case TQVariant::Color: {
	TQVariant v( value );
	ts << makeIndent( indent ) << "<color>" << endl;
	indent++;
	saveColor( ts, indent, v.toColor() );
	indent--;
	ts << makeIndent( indent ) << "</color>" << endl;
    } break;
    case TQVariant::Font: {
	TQVariant v( value );
	ts << makeIndent( indent ) << "<font>" << endl;
	TQFont f( tqApp->font() );
	if ( w && w->isWidgetType() && ((TQWidget*)w)->parentWidget() )
	    f = ((TQWidget*)w)->parentWidget()->font();
	TQFont f2( v.toFont() );
	indent++;
	if ( f.family() != f2.family() )
	    ts << makeIndent( indent ) << "<family>" << f2.family() << "</family>" << endl;
	if ( f.pointSize() != f2.pointSize() )
	    ts << makeIndent( indent ) << "<pointsize>" << TQString::number( f2.pointSize() ) << "</pointsize>" << endl;
	if ( f.bold() != f2.bold() )
	    ts << makeIndent( indent ) << "<bold>" << TQString::number( (int)f2.bold() ) << "</bold>" << endl;
	if ( f.italic() != f2.italic() )
	    ts << makeIndent( indent ) << "<italic>" << TQString::number( (int)f2.italic() ) << "</italic>" << endl;
	if ( f.underline() != f2.underline() )
	    ts << makeIndent( indent ) << "<underline>" << TQString::number( (int)f2.underline() ) << "</underline>" << endl;
	if ( f.strikeOut() != f2.strikeOut() )
	    ts << makeIndent( indent ) << "<strikeout>" << TQString::number( (int)f2.strikeOut() ) << "</strikeout>" << endl;
	indent--;
	ts << makeIndent( indent ) << "</font>" << endl;
    } break;
    case TQVariant::SizePolicy: {
	TQSizePolicy sp( value.toSizePolicy() );
	ts << makeIndent( indent ) << "<sizepolicy>" << endl;
	indent++;
	ts << makeIndent( indent ) << "<hsizetype>" << (int)sp.horData() << "</hsizetype>" << endl;
	ts << makeIndent( indent ) << "<vsizetype>" << (int)sp.verData() << "</vsizetype>" << endl;
	ts << makeIndent( indent ) << "<horstretch>" << (int)sp.horStretch() << "</horstretch>" << endl;
	ts << makeIndent( indent ) << "<verstretch>" << (int)sp.verStretch() << "</verstretch>" << endl;
	indent--;
	ts << makeIndent( indent ) << "</sizepolicy>" << endl;
	break;
    }
    case TQVariant::Pixmap:
	savePixmap( value.toPixmap(), ts, indent );
	break;
    case TQVariant::IconSet:
	savePixmap( value.toIconSet().pixmap(), ts, indent, "iconset" );
	break;
    case TQVariant::Image:
	ts << makeIndent( indent ) << "<image>" << saveInCollection( value.toImage() ) << "</image>" << endl;
    break;
    case TQVariant::Palette: {
	TQPalette p( value.toPalette() );
	ts << makeIndent( indent ) << "<palette>" << endl;
	indent++;

	ts << makeIndent( indent ) << "<active>" << endl;
	indent++;
	saveColorGroup( ts, indent, p.active() );
	indent--;
	ts << makeIndent( indent ) << "</active>" << endl;

	ts << makeIndent( indent ) << "<disabled>" << endl;
	indent++;
	saveColorGroup( ts, indent, p.disabled() );
	indent--;
	ts << makeIndent( indent ) << "</disabled>" << endl;

	ts << makeIndent( indent ) << "<inactive>" << endl;
	indent++;
	saveColorGroup( ts, indent, p.inactive() );
	indent--;
	ts << makeIndent( indent ) << "</inactive>" << endl;

	indent--;
	ts << makeIndent( indent ) << "</palette>" << endl;
    } break;
    case TQVariant::Cursor:
	ts << makeIndent( indent ) << "<cursor>" << value.toCursor().shape() << "</cursor>" << endl;
	break;
    case TQVariant::StringList: {
	TQStringList lst = value.toStringList();
	uint i = 0;
	ts << makeIndent( indent ) << "<stringlist>" << endl;
	indent++;
	if ( !lst.isEmpty() ) {
	    for ( i = 0; i < lst.count(); ++i )
		ts << makeIndent( indent ) << "<string>" << entitize( lst[ i ] ) << "</string>" << endl;
	}
	indent--;
	ts << makeIndent( indent ) << "</stringlist>" << endl;
    } break;
    case TQVariant::Date: {
	TQDate d = value.toDate();
	ts << makeIndent( indent ) << "<date>" << endl;
	indent++;
	ts << makeIndent( indent ) << "<year>" << d.year() << "</year>"  << endl;
	ts << makeIndent( indent ) << "<month>" << d.month() << "</month>"  << endl;
	ts << makeIndent( indent ) << "<day>" << d.day() << "</day>"  << endl;
	indent--;
	ts << makeIndent( indent ) << "</date>" << endl;
	break;
    }
    case TQVariant::Time: {
	TQTime t = value.toTime();
	ts << makeIndent( indent ) << "<time>" << endl;
	indent++;
	ts << makeIndent( indent ) << "<hour>" << t.hour() << "</hour>"  << endl;
	ts << makeIndent( indent ) << "<minute>" << t.minute() << "</minute>"  << endl;
	ts << makeIndent( indent ) << "<second>" << t.second() << "</second>"  << endl;
	indent--;
	ts << makeIndent( indent ) << "</time>" << endl;
	break;
    }
    case TQVariant::DateTime: {
	TQDateTime dt = value.toDateTime();
	ts << makeIndent( indent ) << "<datetime>" << endl;
	indent++;
	ts << makeIndent( indent ) << "<year>" << dt.date().year() << "</year>"  << endl;
	ts << makeIndent( indent ) << "<month>" << dt.date().month() << "</month>"  << endl;
	ts << makeIndent( indent ) << "<day>" << dt.date().day() << "</day>"  << endl;
	ts << makeIndent( indent ) << "<hour>" << dt.time().hour() << "</hour>"  << endl;
	ts << makeIndent( indent ) << "<minute>" << dt.time().minute() << "</minute>"  << endl;
	ts << makeIndent( indent ) << "<second>" << dt.time().second() << "</second>"  << endl;
	indent--;
	ts << makeIndent( indent ) << "</datetime>" << endl;
	break;
    }
    default:
	qWarning( "saving the property %s of type %d not supported yet", name.latin1(), (int)t );
    }
}

void Resource::saveColorGroup( TQTextStream &ts, int indent, const TQColorGroup &cg )
{
    for( int r = 0 ; r < TQColorGroup::NColorRoles ; r++ ) {
	ts << makeIndent( indent ) << "<color>" << endl;
	indent++;
	saveColor( ts, indent, cg.color( (TQColorGroup::ColorRole)r ) );
	indent--;
	ts << makeIndent( indent ) << "</color>" << endl;
	TQPixmap* pm = cg.brush( (TQColorGroup::ColorRole)r ).pixmap();
	if ( pm && !pm->isNull() )
	    savePixmap( *pm, ts, indent );
    }
}

void Resource::saveColor( TQTextStream &ts, int indent, const TQColor &c )
{
    ts << makeIndent( indent ) << "<red>" << TQString::number( c.red() ) << "</red>" << endl;
    ts << makeIndent( indent ) << "<green>" << TQString::number( c.green() ) << "</green>" << endl;
    ts << makeIndent( indent ) << "<blue>" << TQString::number( c.blue() ) << "</blue>" << endl;
}

TQObject *Resource::createObject( const TQDomElement &e, TQWidget *parent, TQLayout* layout )
{
    lastItem = 0;
    TQDomElement n = e.firstChild().toElement();
    TQWidget *w = 0; // the widget that got created
    TQObject *obj = 0; // gets the properties

    int row = e.attribute( "row" ).toInt();
    int col = e.attribute( "column" ).toInt();
    int rowspan = e.attribute( "rowspan" ).toInt();
    int colspan = e.attribute( "colspan" ).toInt();
    if ( rowspan < 1 )
	rowspan = 1;
    if ( colspan < 1 )
	colspan = 1;

    TQString className = e.attribute( "class", TQWIDGET_OBJECT_NAME_STRING );

    if ( !className.isNull() ) {
	obj = TQT_TQOBJECT(WidgetFactory::create( WidgetDatabase::idFromClassName( className ), parent, 0, false ));
	if ( !obj )
	    return 0;
	if ( !mainContainerSet ) {
	    if ( formwindow )
		formwindow->setMainContainer( (TQWidget*)obj );
	    mainContainerSet = true;
	}
	w = (TQWidget*)obj;
	if ( w->inherits( TQMAINWINDOW_OBJECT_NAME_STRING ) )
	    w = ( (TQMainWindow*)w )->centralWidget();
	if ( layout ) {
	    switch ( WidgetFactory::layoutType( layout ) ) {
	    case WidgetFactory::HBox:
		( (TQHBoxLayout*)layout )->addWidget( w );
		break;
	    case WidgetFactory::VBox:
		( (TQVBoxLayout*)layout )->addWidget( w );
		break;
	    case WidgetFactory::Grid:
		( (QDesignerGridLayout*)layout )->addMultiCellWidget( w, row, row + rowspan - 1,
								      col, col + colspan - 1 );
		break;
	    default:
		break;
	    }
	}

	if ( !toplevel )
	    toplevel = w;
	layout = 0;

	if ( w && formwindow ) {
	    if ( !parent || ( !parent->inherits( TQTOOLBOX_OBJECT_NAME_STRING ) && !parent->inherits( TQTABWIDGET_OBJECT_NAME_STRING ) && !parent->inherits( TQWIZARD_OBJECT_NAME_STRING ) ) )
		formwindow->insertWidget( w, pasting );
	    else if ( parent && ( parent->inherits( TQTOOLBOX_OBJECT_NAME_STRING ) || parent->inherits( TQTABWIDGET_OBJECT_NAME_STRING ) || parent->inherits( TQWIZARD_OBJECT_NAME_STRING ) ) )
		MetaDataBase::addEntry( TQT_TQOBJECT(w) );
        if (w->inherits("Dialog"))
          dynamic_cast<Dialog*>(w)->setUseInternalParser(false);
	}
    }

    while ( !n.isNull() ) {
	if ( n.tagName() == "spacer" ) {
	    createSpacer( n, w, layout, Qt::Horizontal );
	} else if ( n.tagName() == "widget" ) {
	    createObject( n, w, layout );
	} else if ( n.tagName() == "hbox" ) {
	    layout = WidgetFactory::createLayout( w, layout, WidgetFactory::HBox );
	    obj = TQT_TQOBJECT(layout);
	    n = n.firstChild().toElement();
	    continue;
	} else if ( n.tagName() == "grid" ) {
	    layout = WidgetFactory::createLayout( w, layout, WidgetFactory::Grid );
	    obj = TQT_TQOBJECT(layout);
	    n = n.firstChild().toElement();
	    continue;
	} else if ( n.tagName() == "vbox" ) {
	    layout = WidgetFactory::createLayout( w, layout, WidgetFactory::VBox );
	    obj = TQT_TQOBJECT(layout);
	    n = n.firstChild().toElement();
	    continue;
	} else if ( n.tagName() == "property" && obj ) {
	    setObjectProperty( obj, n.attribute( "name" ), n.firstChild().toElement() );
	} else if ( n.tagName() == "attribute" && w ) {
	    TQString attrib = n.attribute( "name" );
	    TQVariant v = DomTool::elementToVariant( n.firstChild().toElement(), TQVariant() );
	    if ( parent->inherits( TQTABWIDGET_OBJECT_NAME_STRING ) ) {
		if ( attrib == "title" )
		    ( (TQTabWidget*)parent )->insertTab( w, v.toString() );
	    } else
        if ( parent->inherits( TQTOOLBOX_OBJECT_NAME_STRING ) ) {
        if ( attrib == "label" )
            ( (TQToolBox*)parent )->addItem( w, v.toString() );
        } else if ( parent->inherits( TQWIZARD_OBJECT_NAME_STRING ) ) {
		if ( attrib == "title" )
		    ( (TQWizard*)parent )->addPage( w, v.toString() );
	    }
	} else if ( n.tagName() == "item" ) {
	    createItem( n, w );
	} else if ( n.tagName() == "column" || n.tagName() =="row" ) {
	    createColumn( n, w );
	} else if ( n.tagName() == "event" ) {
	// qDebug("Resource : Tag == event");
#ifndef KOMMANDER
	    MetaDataBase::setEventFunctions( obj, formwindow, MainWindow::self->currProject()->language(),
					     n.attribute( "name" ), TQStringList::split( ',', n.attribute( "functions" ) ), false );
#endif
	}

	n = n.nextSibling().toElement();
    }

    return TQT_TQOBJECT(w);
}

void Resource::createColumn( const TQDomElement &e, TQWidget *widget )
{
    if ( !widget )
	return;

    if ( widget->inherits( TQLISTVIEW_OBJECT_NAME_STRING ) && e.tagName() == "column" ) {
	TQListView *lv = (TQListView*)widget;
	TQDomElement n = e.firstChild().toElement();
	TQPixmap pix;
	bool hasPixmap = false;
	TQString txt;
	bool clickable = true, resizeable = true;
	while ( !n.isNull() ) {
	    if ( n.tagName() == "property" ) {
		TQString attrib = n.attribute( "name" );
		TQVariant v = DomTool::elementToVariant( n.firstChild().toElement(), TQVariant() );
		if ( attrib == "text" )
		    txt = v.toString();
		else if ( attrib == "pixmap" ) {
		    pix = loadPixmap( n.firstChild().toElement().toElement() );
		    hasPixmap = !pix.isNull();
		} else if ( attrib == "clickable" )
		    clickable = v.toBool();
		else if ( attrib == "resizeable" )
		    resizeable = v.toBool();
	    }
	    n = n.nextSibling().toElement();
	}
	lv->addColumn( txt );
	int i = lv->header()->count() - 1;
	if ( hasPixmap ) {
	    lv->header()->setLabel( i, pix, txt );
	}
	if ( !clickable )
	    lv->header()->setClickEnabled( clickable, i );
	if ( !resizeable )
	    lv->header()->setResizeEnabled( resizeable, i );
    }
#ifndef TQT_NO_TABLE
    else if ( widget->inherits( TQTABLE_OBJECT_NAME_STRING ) ) {
	TQTable *table = (TQTable*)widget;
	bool isRow;
	if ( ( isRow = e.tagName() == "row" ) )
	    table->setNumRows( table->numRows() + 1 );
	else
	    table->setNumCols( table->numCols() + 1 );

	TQDomElement n = e.firstChild().toElement();
	TQPixmap pix;
	bool hasPixmap = false;
	TQString txt;
	TQString field;
	TQMap<TQString, TQString> fieldMap = MetaDataBase::columnFields( TQT_TQOBJECT(table) );
	while ( !n.isNull() ) {
	    if ( n.tagName() == "property" ) {
		TQString attrib = n.attribute( "name" );
		TQVariant v = DomTool::elementToVariant( n.firstChild().toElement(), TQVariant() );
		if ( attrib == "text" )
		    txt = v.toString();
		else if ( attrib == "pixmap" ) {
		    hasPixmap = !n.firstChild().firstChild().toText().data().isEmpty();
		    if ( hasPixmap )
			pix = loadPixmap( n.firstChild().toElement() );
		} else if ( attrib == "field" )
		    field = v.toString();
	    }
	    n = n.nextSibling().toElement();
	}

	int i = isRow ? table->numRows() - 1 : table->numCols() - 1;
	TQHeader *h = !isRow ? table->horizontalHeader() : table->verticalHeader();
	if ( hasPixmap )
	    h->setLabel( i, pix, txt );
	else
	    h->setLabel( i, txt );
	if ( !isRow && !field.isEmpty() )
	    fieldMap.insert( txt, field );
	MetaDataBase::setColumnFields( TQT_TQOBJECT(table), fieldMap );
    }
#endif
}

void Resource::loadItem( const TQDomElement &e, TQPixmap &pix, TQString &txt, bool &hasPixmap )
{
    TQDomElement n = e;
    hasPixmap = false;
    while ( !n.isNull() ) {
	if ( n.tagName() == "property" ) {
	    TQString attrib = n.attribute( "name" );
	    TQVariant v = DomTool::elementToVariant( n.firstChild().toElement(), TQVariant() );
	    if ( attrib == "text" )
		txt = v.toString();
	    else if ( attrib == "pixmap" ) {
		pix = loadPixmap( n.firstChild().toElement() );
		hasPixmap = !pix.isNull();
	    }
	}
	n = n.nextSibling().toElement();
    }
}

void Resource::createItem( const TQDomElement &e, TQWidget *widget, TQListViewItem *i )
{
    if ( !widget || !WidgetFactory::hasItems( WidgetDatabase::idFromClassName( WidgetFactory::classNameOf( TQT_TQOBJECT(widget) ) ) ) )
	return;

    if ( widget->inherits( TQLISTBOX_OBJECT_NAME_STRING ) || widget->inherits( TQCOMBOBOX_OBJECT_NAME_STRING ) ) {
	TQDomElement n = e.firstChild().toElement();
	TQPixmap pix;
	bool hasPixmap = false;
	TQString txt;
	loadItem( n, pix, txt, hasPixmap );
	TQListBox *lb = 0;
	if ( widget->inherits( TQLISTBOX_OBJECT_NAME_STRING ) )
	    lb = (TQListBox*)widget;
	else
	    lb = ( (TQComboBox*)widget)->listBox();
	if ( hasPixmap ) {
	    new TQListBoxPixmap( lb, pix, txt );
	} else {
	    new TQListBoxText( lb, txt );
	}
    } else if ( widget->inherits( TQICONVIEW_OBJECT_NAME_STRING ) ) {
	TQDomElement n = e.firstChild().toElement();
	TQPixmap pix;
	bool hasPixmap = false;
	TQString txt;
	loadItem( n, pix, txt, hasPixmap );

	TQIconView *iv = (TQIconView*)widget;
	new TQIconViewItem( iv, txt, pix );
    } else if ( widget->inherits( TQLISTVIEW_OBJECT_NAME_STRING ) ) {
	TQDomElement n = e.firstChild().toElement();
	TQPixmap pix;
	TQValueList<TQPixmap> pixmaps;
	TQStringList textes;
	TQListViewItem *item = 0;
	TQListView *lv = (TQListView*)widget;
	if ( i )
	    item = new TQListViewItem( i, lastItem );
	else
	    item = new TQListViewItem( lv, lastItem );
	while ( !n.isNull() ) {
	    if ( n.tagName() == "property" ) {
		TQString attrib = n.attribute( "name" );
		TQVariant v = DomTool::elementToVariant( n.firstChild().toElement(), TQVariant() );
		if ( attrib == "text" )
		    textes << v.toString();
		else if ( attrib == "pixmap" ) {
		    TQString s = v.toString();
		    if ( s.isEmpty() ) {
			pixmaps << TQPixmap();
		    } else {
			pix = loadPixmap( n.firstChild().toElement() );
			pixmaps << pix;
		    }
		}

	    } else if ( n.tagName() == "item" ) {
		item->setOpen( true );
		createItem( n, widget, item );
	    }

	    n = n.nextSibling().toElement();
	}

	for ( int i = 0; i < lv->columns(); ++i ) {
	    item->setText( i, textes[ i ] );
	    item->setPixmap( i, pixmaps[ i ] );
	}
	lastItem = item;
    }
}

TQWidget *Resource::createSpacer( const TQDomElement &e, TQWidget *parent, TQLayout *layout, Qt::Orientation o )
{
    TQDomElement n = e.firstChild().toElement();
    int row = e.attribute( "row" ).toInt();
    int col = e.attribute( "column" ).toInt();
    int rowspan = e.attribute( "rowspan" ).toInt();
    int colspan = e.attribute( "colspan" ).toInt();
    if ( rowspan < 1 )
	rowspan = 1;
    if ( colspan < 1 )
	colspan = 1;

    Spacer *spacer = (Spacer*) WidgetFactory::create( WidgetDatabase::idFromClassName("Spacer"),
						      parent, "spacer", false);
    spacer->setOrientation( o );
    spacer->setInteraciveMode( false );
    while ( !n.isNull() ) {
	if ( n.tagName() == "property" )
	    setObjectProperty( TQT_TQOBJECT(spacer), n.attribute( "name" ), n.firstChild().toElement() );
	n = n.nextSibling().toElement();
    }
    spacer->setInteraciveMode( true );
    if ( formwindow )
	formwindow->insertWidget( spacer, pasting );
    if ( layout ) {
	if ( layout->inherits( TQBOXLAYOUT_OBJECT_NAME_STRING ) )
	    ( (TQBoxLayout*)layout )->addWidget( spacer, 0, spacer->alignment() );
	else
	    ( (QDesignerGridLayout*)layout )->addMultiCellWidget( spacer, row, row + rowspan - 1, col, col + colspan - 1,
								  spacer->alignment() );
    }
    return spacer;
}

/*!
  Attention: this function has to be in sync with Uic::setObjectProperty(). If you change one, change both.
*/
void Resource::setObjectProperty( TQObject* obj, const TQString &prop, const TQDomElement &e )
{
    const TQMetaProperty *p = obj->metaObject()->property( obj->metaObject()->findProperty( prop, true ), true );

    if ( !obj->inherits( TQLAYOUT_OBJECT_NAME_STRING )  ) {// no layouts in metadatabase... (RS)
	if ( obj->inherits( "CustomWidget" ) ) {
	    MetaDataBase::CustomWidget *cw = ( (CustomWidget*)obj )->customWidget();
	    if ( cw && !cw->hasProperty( prop.latin1() ) && !p && prop != "toolTip" && prop != "whatsThis" )
		return;
	}
	MetaDataBase::setPropertyChanged( obj, prop, true );
    }

    TQVariant defVarient;
    if ( e.tagName() == "font" ) {
	TQFont f( tqApp->font() );
	if ( obj->isWidgetType() && ( (TQWidget*)obj )->parentWidget() )
	    f = ( (TQWidget*)obj )->parentWidget()->font();
	defVarient = TQVariant( f );
    }

    TQString comment;
    TQVariant v( DomTool::elementToVariant( e, defVarient, comment ) );

    if ( !comment.isEmpty() ) {
	MetaDataBase::addEntry( obj );
	MetaDataBase::setPropertyComment( obj, prop, comment );
    }

    if ( e.tagName() == "pixmap" ) {
	TQPixmap pix = loadPixmap( e );
	if ( pix.isNull() )
	    return;
	v = TQVariant( pix );
    } else if ( e.tagName() == "iconset" ) {
	TQPixmap pix = loadPixmap( e, "iconset" );
	if ( pix.isNull() )
	    return;
	v = TQVariant( TQIconSet( pix ) );
    } else if ( e.tagName() == "image" ) {
	v = TQVariant( loadFromCollection( v.toString() ) );
    }

    if ( !p ) {
	MetaDataBase::setFakeProperty( obj, prop, v );
	if ( obj->isWidgetType() ) {
	    if ( prop == "database" && TQT_BASE_OBJECT(obj) != TQT_BASE_OBJECT(toplevel) ) {
		TQStringList lst = MetaDataBase::fakeProperty( obj, "database" ).toStringList();
		if ( lst.count() > 2 )
		    dbControls.insert( obj->name(), lst[ 2 ] );
		else if ( lst.count() == 2 )
		    dbTables.insert( obj->name(), lst );
	    }
	    return;
	}
    }


    if ( e.tagName() == "palette" ) {
	TQDomElement n = e.firstChild().toElement();
	TQPalette p;
	while ( !n.isNull() ) {
	    TQColorGroup cg;
	    if ( n.tagName() == "active" ) {
		cg = loadColorGroup( n );
		p.setActive( cg );
	    } else if ( n.tagName() == "inactive" ) {
		cg = loadColorGroup( n );
		p.setInactive( cg );
	    } else if ( n.tagName() == "disabled" ) {
		cg = loadColorGroup( n );
		p.setDisabled( cg );
	    }
	    n = n.nextSibling().toElement();
	}
	v = TQPalette( p );
    } else if ( e.tagName() == "enum" && p && p->isEnumType() ) {
	TQString key( v.toString() );
	v = TQVariant( p->keyToValue( key ) );
    } else if ( e.tagName() == "set" && p && p->isSetType() ) {
	TQString keys( v.toString() );
	TQStringList lst = TQStringList::split( '|', keys );
	TQStrList l;
	for ( TQStringList::Iterator it = lst.begin(); it != lst.end(); ++it )
	    l.append( *it );
	v = TQVariant( p->keysToValue( l ) );
    }

    if ( prop == "caption" ) {
	TQCString s1 = v.toCString();
	TQString s2 = v.toString();
	if ( !s1.isEmpty() )
	    formwindow->setCaption( s1 );
	else if ( !s2.isEmpty() )
	    formwindow->setCaption( s2 );
    }
    if ( prop == "icon" ) {
	formwindow->setIcon( v.toPixmap() );
    }

    if ( prop == "geometry" ) {
	if ( TQT_BASE_OBJECT(obj) == toplevel ) {
	    hadGeometry = true;
	    toplevel->resize( v.toRect().size() );
	    return;
	} else if ( TQT_BASE_OBJECT(obj) == formwindow->mainContainer() ) {
	    hadGeometry = true;
	    formwindow->resize( v.toRect().size() );
	    return;
	}
    }

    if ( obj->inherits( TQLAYOUT_OBJECT_NAME_STRING ) ) {
	if ( prop == "spacing" ) {
	    MetaDataBase::setSpacing( TQT_TQOBJECT(WidgetFactory::containerOfWidget( WidgetFactory::layoutParent( (TQLayout*)obj ) )), v.toInt() );
	    return;
	}
	if ( prop == "margin" ) {
	    MetaDataBase::setMargin( TQT_TQOBJECT(WidgetFactory::containerOfWidget( WidgetFactory::layoutParent( (TQLayout*)obj ) )), v.toInt() );
	    return;
	}
    }

    if ( prop == "name" ) {
	if ( pasting ) {
	    TQString s = v.toString();
	    formwindow->unify( TQT_TQOBJECT(obj), s, true );
	    obj->setName( s );
	    return;
	} else if ( formwindow && TQT_BASE_OBJECT(obj) == formwindow->mainContainer() ) {
	    formwindow->setName( v.toCString() );
	}
    }

    if ( prop == "sizePolicy" ) {
	TQSizePolicy sp = v.toSizePolicy();
	sp.setHeightForWidth( ( (TQWidget*)obj )->sizePolicy().hasHeightForWidth() );
    }

    obj->setProperty( prop, v );
}


TQString Resource::saveInCollection( const TQImage &img )
{
    TQString imgName = "none";
    TQValueList<Image>::Iterator it = images.begin();
    for ( ; it != images.end(); ++it ) {
	if ( img == ( *it ).img ) {
	    imgName = ( *it ).name;
	    break;
	}
    }

    if ( imgName == "none" ) {
	Image i;
	imgName = "image" + TQString::number( images.count() );
	i.name = imgName;
	i.img = img;
	images.append( i );
    }
    return imgName;
}

void Resource::saveImageData( const TQImage &img, TQTextStream &ts, int indent )
{
    TQByteArray ba;
    TQBuffer buf( ba );
    buf.open( IO_WriteOnly | IO_Translate );
    TQImageIO iio( &buf, "XPM" );
    iio.setImage( img );
    iio.write();
    buf.close();
    ulong len = ba.size() * 2;
    TQByteArray bazip( len );
    ::compress(  (uchar*) bazip.data(), &len, (uchar*) ba.data(), ba.size() );
    TQString res;
    ts << makeIndent( indent ) << "<data format=\"XPM.GZ\" length=\"" << ba.size() << "\">";
    static const char hexchars[] = "0123456789abcdef";
    for ( int i = 0; i < (int)len; ++i ) {
	uchar s = (uchar) bazip[i];
	ts << hexchars[s >> 4];
	ts << hexchars[s & 0x0f];
    }
    ts << "</data>" << endl;
}

void Resource::saveImageCollection( TQTextStream &ts, int indent )
{
    ts << makeIndent( indent ) << "<images>" << endl;
    indent++;

    TQValueList<Image>::Iterator it = images.begin();
    for ( ; it != images.end(); ++it ) {
	ts << makeIndent( indent ) << "<image name=\"" << (*it).name << "\">" << endl;
	indent++;
	saveImageData( (*it).img, ts, indent );
	indent--;
	ts << makeIndent( indent ) << "</image>" << endl;
    }

    indent--;
    ts << makeIndent( indent ) << "</images>" << endl;
}

static TQImage loadImageData( TQDomElement &n2 )
{
    TQImage img;
    TQString data = n2.firstChild().toText().data();
    char *ba = new char[ data.length() / 2 ];
    for ( int i = 0; i < (int)data.length() / 2; ++i ) {
	char h = data[ 2 * i ].latin1();
	char l = data[ 2 * i  + 1 ].latin1();
	uchar r = 0;
	if ( h <= '9' )
	    r += h - '0';
	else
	    r += h - 'a' + 10;
	r = r << 4;
	if ( l <= '9' )
	    r += l - '0';
	else
	    r += l - 'a' + 10;
	ba[ i ] = r;
    }
    TQString format = n2.attribute( "format", "PNG" );
    if ( format == "XPM.GZ" ) {
	ulong len = n2.attribute( "length" ).toULong();
	if ( len < data.length() * 5 )
	    len = data.length() * 5;
	TQByteArray baunzip( len );
	::uncompress( (uchar*) baunzip.data(), &len, (uchar*) ba, data.length()/2 );
	img.loadFromData( (const uchar*)baunzip.data(), len, "XPM" );
    }  else {
	img.loadFromData( (const uchar*)ba, data.length() / 2, format );
    }
    delete [] ba;

    return img;
}

void Resource::loadImageCollection( const TQDomElement &e )
{
    TQDomElement n = e.firstChild().toElement();
    while ( !n.isNull() ) {
	if ( n.tagName() == "image" ) {
	    Image img;
	    img.name =  n.attribute( "name" );
	    TQDomElement n2 = n.firstChild().toElement();
	    while ( !n2.isNull() ) {
		if ( n2.tagName() == "data" )
		    img.img = loadImageData( n2 );
		n2 = n2.nextSibling().toElement();
	    }
	    images.append( img );
	    n = n.nextSibling().toElement();
	}
    }
}

TQImage Resource::loadFromCollection( const TQString &name )
{
    TQValueList<Image>::Iterator it = images.begin();
    for ( ; it != images.end(); ++it ) {
	if ( ( *it ).name == name )
	    return ( *it ).img;
    }
    return TQImage();
}

void Resource::saveConnections( TQTextStream &ts, int indent )
{
    ts << makeIndent( indent ) << "<connections>" << endl;
    indent++;
    TQValueList<MetaDataBase::Connection> connections = MetaDataBase::connections( TQT_TQOBJECT(formwindow) );
    TQValueList<MetaDataBase::Connection>::Iterator it = connections.begin();
    for ( ; it != connections.end(); ++it ) {
	MetaDataBase::Connection conn = *it;
	if ( ( knownNames.findIndex( TQString( conn.sender->name() ) ) == -1 &&
	       qstrcmp( conn.sender->name(), "this" ) != 0 ) ||
	     ( knownNames.findIndex( TQString( conn.receiver->name() ) ) == -1 &&
	       qstrcmp( conn.receiver->name(), "this" ) != 0 ) )
	    continue;
	if ( formwindow->isMainContainer( TQT_TQOBJECT((*it).receiver) ) &&
	     !MetaDataBase::hasSlot( TQT_TQOBJECT(formwindow), MetaDataBase::normalizeSlot( (*it).slot ).latin1() ) )
	    continue;

	if ( conn.sender->inherits( "CustomWidget" ) ) {
	    MetaDataBase::CustomWidget *cw = ( (CustomWidget*)conn.sender )->customWidget();
	    if ( cw && !cw->hasSignal( conn.signal ) )
		continue;
	}

	if ( conn.receiver->inherits( "CustomWidget" ) && !formwindow->isMainContainer( conn.receiver ) ) {
	    MetaDataBase::CustomWidget *cw = ( (CustomWidget*)conn.receiver )->customWidget();
	    if ( cw && !cw->hasSlot( MetaDataBase::normalizeSlot( conn.slot ).latin1() ) )
		continue;
	}

	ts << makeIndent( indent ) << "<connection>" << endl;
	indent++;
	ts << makeIndent( indent ) << "<sender>" << entitize( conn.sender->name() ) << "</sender>" << endl;
	ts << makeIndent( indent ) << "<signal>" << entitize( conn.signal ) << "</signal>" << endl;
	ts << makeIndent( indent ) << "<receiver>" << entitize( conn.receiver->name() ) << "</receiver>" << endl;
	ts << makeIndent( indent ) << "<slot>" << entitize( MetaDataBase::normalizeSlot( conn.slot ) ) << "</slot>" << endl;
	indent--;
	ts << makeIndent( indent ) << "</connection>" << endl;
    }

#ifndef KOMMANDER
    TQString lang = formwindow->project()->language();
    LanguageInterface *iface = langIface;
    if ( iface && MetaDataBase::hasEvents( lang ) ) {
	TQObjectList *l = formwindow->queryList( TQWIDGET_OBJECT_NAME_STRING );
	l->append( formwindow );
	TQPtrList<TQAction> lst = formwindow->actionList();
	for ( TQAction *a = lst.first(); a; a = lst.next() )
	    l->append( a );
	for ( TQObject *o = l->first(); o; o = l->next() ) {
	    if ( !MetaDataBase::hasObject( o ) )
		continue;
	    TQMap<TQString, TQStringList> eventFunctions = MetaDataBase::eventFunctions( o );
	    TQMap<TQString, TQStringList>::ConstIterator it = eventFunctions.begin();
	    for ( ; it != eventFunctions.end(); ++it ) {
		TQString sls = (*it).join( "," );
		MetaDataBase::Connection conn;
		conn.sender = o;
		conn.signal = it.key();
		conn.slot = sls;
		langConnections[ lang ].append( conn );
	    }
	}
	delete l;
    }
#endif

    indent--;
    ts << makeIndent( indent ) << "</connections>" << endl;
}

void Resource::loadConnections( const TQDomElement &e )
{
    TQDomElement n = e.firstChild().toElement();
    while ( !n.isNull() ) {
	if ( n.tagName() == "connection" ) {
	    TQString lang = n.attribute( "language", "C++" );
	    TQDomElement n2 = n.firstChild().toElement();
	    MetaDataBase::Connection conn;
	    while ( !n2.isNull() ) {
		if ( n2.tagName() == "sender" ) {
		    conn.sender = 0;
		    TQString name = n2.firstChild().toText().data();
		    if ( name == "this" || qstrcmp( toplevel->name(), name ) == 0 ) {
			conn.sender = TQT_TQOBJECT(toplevel);
		    } else {
			if ( name == "this" )
			    name = toplevel->name();
			TQObjectList *l = toplevel->queryList( 0, name, false );
			if ( l ) {
			    if ( l->first() )
				conn.sender = l->first();
			    delete l;
			}
			if ( !conn.sender )
			    conn.sender = formwindow->findAction(  name );
		    }
		} else if ( n2.tagName() == "signal" ) {
		    conn.signal = n2.firstChild().toText().data();
		} else if ( n2.tagName() == "receiver" ) {
		    TQString name = n2.firstChild().toText().data();
		    if ( name == "this" || qstrcmp( toplevel->name(), name ) == 0 ) {
			conn.receiver = TQT_TQOBJECT(toplevel);
		    } else {
			TQObjectList *l = toplevel->queryList( 0, name, false );
			if ( l ) {
			    if ( l->first() )
				conn.receiver = l->first();
			    delete l;
			}
		    }
		} else if ( n2.tagName() == "slot" ) {
		    conn.slot = n2.firstChild().toText().data();
		}
		n2 = n2.nextSibling().toElement();
	    }
	    if ( formwindow ) {
		if ( TQT_BASE_OBJECT(conn.sender) == TQT_BASE_OBJECT(formwindow) )
		    conn.sender = TQT_TQOBJECT(formwindow->mainContainer());
		if ( TQT_BASE_OBJECT(conn.receiver) == TQT_BASE_OBJECT(formwindow) )
		    conn.receiver = TQT_TQOBJECT(formwindow->mainContainer());
	    }
	    if ( conn.sender && conn.receiver ) {
#ifndef KOMMANDER
		if ( lang == "C++" ) {
		    MetaDataBase::addConnection( formwindow ? formwindow : toplevel,
						 conn.sender, conn.signal, conn.receiver, conn.slot );
		}
		else if ( MetaDataBase::hasEvents( lang ) ) {
		    MetaDataBase::setEventFunctions( conn.sender, formwindow, lang, conn.signal,
						     TQStringList::split( ',', conn.slot ), false );
		}
#else
		    MetaDataBase::addConnection( formwindow ? TQT_TQOBJECT(formwindow) : TQT_TQOBJECT(toplevel),
						 conn.sender, conn.signal, conn.receiver, conn.slot );
#endif
	    }
	} else if ( n.tagName() == "slot" ) { // compatibility with 2.x
	    MetaDataBase::Slot slot;
	    slot.specifier = n.attribute( "specifier", "virtual" );
	    if ( slot.specifier.isEmpty() )
		slot.specifier = "virtual";
	    slot.access = n.attribute( "access", "public" );
	    if ( slot.access.isEmpty() )
		slot.access = "public";
	    slot.language = n.attribute( "language", "C++" );
	    slot.returnType = n.attribute( "returnType", "void" );
	    if ( slot.returnType.isEmpty() )
		slot.returnType = "void";
	    slot.slot = n.firstChild().toText().data();
	    if ( !MetaDataBase::hasSlot( TQT_TQOBJECT(formwindow), slot.slot, true ) )
		MetaDataBase::addSlot( TQT_TQOBJECT(formwindow), slot.slot, slot.specifier,
				       slot.access, slot.language, slot.returnType );
	    else
		MetaDataBase::changeSlotAttributes( TQT_TQOBJECT(formwindow), slot.slot,
						    slot.specifier, slot.access,
						    slot.language, slot.returnType );
	}
	n = n.nextSibling().toElement();
    }
}

void Resource::saveCustomWidgets( TQTextStream &ts, int indent )
{
    ts << makeIndent( indent ) << "<customwidgets>" << endl;
    indent++;

    TQPtrList<MetaDataBase::CustomWidget> *lst = MetaDataBase::customWidgets();
    for ( MetaDataBase::CustomWidget *w = lst->first(); w; w = lst->next() ) {
	if ( usedCustomWidgets.findIndex( w->className ) == -1 )
	    continue;
	ts << makeIndent( indent ) << "<customwidget>" << endl;
	indent++;
	ts << makeIndent( indent ) << "<class>" << w->className << "</class>" << endl;
	ts << makeIndent( indent ) << "<header location=\""
	   << ( w->includePolicy == MetaDataBase::CustomWidget::Local ? "local" : "global" )
	   << "\">" << w->includeFile << "</header>" << endl;
	ts << makeIndent( indent ) << "<sizehint>" << endl;
	indent++;
	ts << makeIndent( indent ) << "<width>" << w->sizeHint.width() << "</width>" << endl;
	ts << makeIndent( indent ) << "<height>" << w->sizeHint.height() << "</height>" << endl;
	indent--;
	ts << makeIndent( indent ) << "</sizehint>" << endl;
	ts << makeIndent( indent ) << "<container>" << (int)w->isContainer << "</container>" << endl;
	ts << makeIndent( indent ) << "<sizepolicy>" << endl;
	indent++;
	ts << makeIndent( indent ) << "<hordata>" << (int)w->sizePolicy.horData() << "</hordata>" << endl;
	ts << makeIndent( indent ) << "<verdata>" << (int)w->sizePolicy.verData() << "</verdata>" << endl;
	ts << makeIndent( indent ) << "<horstretch>" << (int)w->sizePolicy.horStretch() << "</horstretch>" << endl;
	ts << makeIndent( indent ) << "<verstretch>" << (int)w->sizePolicy.verStretch() << "</verstretch>" << endl;
	indent--;
	ts << makeIndent( indent ) << "</sizepolicy>" << endl;
	ts << makeIndent( indent ) << "<pixmap>" << saveInCollection( w->pixmap->convertToImage() ) << "</pixmap>" << endl;
	if ( !w->lstSignals.isEmpty() ) {
	    for ( TQValueList<TQCString>::Iterator it = w->lstSignals.begin(); it != w->lstSignals.end(); ++it )
		ts << makeIndent( indent ) << "<signal>" << entitize( *it ) << "</signal>" << endl;
	}
	if ( !w->lstSlots.isEmpty() ) {
	    for ( TQValueList<MetaDataBase::Slot>::Iterator it = w->lstSlots.begin(); it != w->lstSlots.end(); ++it )
		ts << makeIndent( indent ) << "<slot access=\"" << (*it).access << "\" specifier=\""
		   << (*it).specifier << "\">" << entitize( (*it).slot ) << "</slot>" << endl;
	}
	if ( !w->lstProperties.isEmpty() ) {
	    for ( TQValueList<MetaDataBase::Property>::Iterator it = w->lstProperties.begin(); it != w->lstProperties.end(); ++it )
		ts << makeIndent( indent ) << "<property type=\"" << (*it).type << "\">" << entitize( (*it).property ) << "</property>" << endl;
	}
	indent--;
	ts << makeIndent( indent ) << "</customwidget>" << endl;
    }


    indent--;
    ts << makeIndent( indent ) << "</customwidgets>" << endl;
}

void Resource::loadCustomWidgets( const TQDomElement &e, Resource *r )
{
    TQDomElement n = e.firstChild().toElement();
    while ( !n.isNull() ) {
	if ( n.tagName() == "customwidget" ) {
	    TQDomElement n2 = n.firstChild().toElement();
	    MetaDataBase::CustomWidget *w = new MetaDataBase::CustomWidget;
	    while ( !n2.isNull() ) {
		if ( n2.tagName() == "class" ) {
		    w->className = n2.firstChild().toText().data();
		} else if ( n2.tagName() == "header" ) {
		    w->includeFile = n2.firstChild().toText().data();
		    TQString s = n2.attribute( "location" );
		    if ( s != "local" )
			w->includePolicy = MetaDataBase::CustomWidget::Global;
		    else
			w->includePolicy = MetaDataBase::CustomWidget::Local;
		} else if ( n2.tagName() == "sizehint" ) {
		    TQDomElement n3 = n2.firstChild().toElement();
		    while ( !n3.isNull() ) {
			if ( n3.tagName() == "width" )
			    w->sizeHint.setWidth( n3.firstChild().toText().data().toInt() );
			else if ( n3.tagName() == "height" )
			    w->sizeHint.setHeight( n3.firstChild().toText().data().toInt() );
			n3 = n3.nextSibling().toElement();
		    }
		} else if ( n2.tagName() == "sizepolicy" ) {
		    TQDomElement n3 = n2.firstChild().toElement();
		    while ( !n3.isNull() ) {
			if ( n3.tagName() == "hordata" )
			    w->sizePolicy.setHorData( (TQSizePolicy::SizeType)n3.firstChild().toText().data().toInt() );
			else if ( n3.tagName() == "verdata" )
			    w->sizePolicy.setVerData( (TQSizePolicy::SizeType)n3.firstChild().toText().data().toInt() );
			else if ( n3.tagName() == "horstretch" )
			    w->sizePolicy.setHorStretch( n3.firstChild().toText().data().toInt() );
			else if ( n3.tagName() == "verstretch" )
			    w->sizePolicy.setVerStretch( n3.firstChild().toText().data().toInt() );
			n3 = n3.nextSibling().toElement();
		    }
		} else if ( n2.tagName() == "pixmap" ) {
		    TQPixmap pix;
		    if ( r ) {
			pix = r->loadPixmap( n2 );
		    } else {
			TQDomElement n3 = n2.firstChild().toElement();
			TQImage img;
			while ( !n3.isNull() ) {
			    if ( n3.tagName() == "data" ) {
				img = loadImageData( n3 );
			    }
			    n3 = n3.nextSibling().toElement();
			}
			pix.convertFromImage( img );
		    }
		    w->pixmap = new TQPixmap( pix );
		} else if ( n2.tagName() == "signal" ) {
		    w->lstSignals.append( n2.firstChild().toText().data().latin1() );
		} else if ( n2.tagName() == "container" ) {
		    w->isContainer = (bool)n2.firstChild().toText().data().toInt();
		} else if ( n2.tagName() == "slot" ) {
		    MetaDataBase::Slot slot;
		    slot.slot = n2.firstChild().toText().data().latin1();
		    slot.access = n2.attribute( "access" );
		    w->lstSlots.append( slot );
		} else if ( n2.tagName() == "property" ) {
		    MetaDataBase::Property property;
		    property.property = n2.firstChild().toText().data().latin1();
		    property.type = n2.attribute( "type" );
		    w->lstProperties.append( property );
		}
		n2 = n2.nextSibling().toElement();
	    }
	    MetaDataBase::addCustomWidget( w );
	}
	n = n.nextSibling().toElement();
    }
}

void Resource::saveTabOrder( TQTextStream &ts, int indent )
{
    TQWidgetList l = MetaDataBase::tabOrder( toplevel );
    if ( l.isEmpty() )
	return;

    ts << makeIndent( indent ) << "<tabstops>" << endl;
    indent++;

    for ( TQWidget *w = l.first(); w; w = l.next() ) {
	if ( w->testWState( TQt::WState_ForceHide ) || knownNames.findIndex( w->name() ) == -1 )
	    continue;
	ts << makeIndent( indent ) << "<tabstop>" << w->name() << "</tabstop>" << endl;
    }

    indent--;
    ts << makeIndent( indent ) << "</tabstops>" << endl;
}

void Resource::loadTabOrder( const TQDomElement &e )
{
    TQWidget *last = 0;
    TQDomElement n = e.firstChild().toElement();
    TQWidgetList widgets;
    while ( !n.isNull() ) {
	if ( n.tagName() == "tabstop" ) {
	    TQString name = n.firstChild().toText().data();
	    if ( name.isEmpty() )
		continue;
	    TQObjectList *l = toplevel->queryList( 0, name, false );
	    if ( l ) {
		if ( l->first() ) {
		    TQWidget *w = (TQWidget*)l->first();
		    widgets.append( w );
		    if ( last )
			toplevel->setTabOrder( last, w );
		    last = w;
		}
		delete l;
	    }
	}
	n = n.nextSibling().toElement();
    }

    if ( !widgets.isEmpty() )
	MetaDataBase::setTabOrder( toplevel, widgets );
}

void Resource::saveMetaInfoBefore( TQTextStream &ts, int indent )
{
  MetaDataBase::MetaInfo info = MetaDataBase::metaInfo(TQT_TQOBJECT(formwindow));
  TQString cn = formwindow->name();
  ts << makeIndent(indent) << "<class>" << entitize(cn) << "</class>" << endl;
  if (!info.comment.isEmpty())
    ts << makeIndent(indent) << "<comment>" << entitize(info.comment) << "</comment>" << endl;
  if (!info.author.isEmpty())
    ts << makeIndent(indent) << "<author>" << entitize(info.author) << "</author>" << endl;
  if (!info.license.isEmpty())
    ts << makeIndent(indent) << "<license>" << entitize(info.license) << "</license>" << endl;
  if (!info.version.isEmpty())
    ts << makeIndent(indent) << "<version>" << entitize(info.version) << "</version>" << endl;
}

void Resource::saveMetaInfoAfter( TQTextStream &ts, int indent )
{
    MetaDataBase::MetaInfo info = MetaDataBase::metaInfo( TQT_TQOBJECT(formwindow) );
	TQValueList<MetaDataBase::Include> includes = MetaDataBase::includes( TQT_TQOBJECT(formwindow) );
	TQString extensionInclude;
	bool needExtensionInclude = false;
	if ( !includes.isEmpty() || needExtensionInclude ) {
	    ts << makeIndent( indent ) << "<includes>" << endl;
	    indent++;

	    for ( TQValueList<MetaDataBase::Include>::Iterator it = includes.begin(); it != includes.end(); ++it ) {
		ts << makeIndent( indent ) << "<include location=\"" << (*it).location
		   << "\" impldecl=\"" << (*it).implDecl << "\">" << (*it).header << "</include>" << endl;
		if ( needExtensionInclude )
		    needExtensionInclude = (*it).header != extensionInclude;
	    }

	    if ( needExtensionInclude )
		ts << makeIndent( indent ) << "<include location=\"local\" impldecl=\"in implementation\">"
		   << extensionInclude << "</include>" << endl;
	    indent--;
	    ts << makeIndent( indent ) << "</includes>" << endl;
	}

	TQStringList forwards = MetaDataBase::forwards( TQT_TQOBJECT(formwindow) );
	if ( !forwards.isEmpty() ) {
	    ts << makeIndent( indent ) << "<forwards>" << endl;
	    indent++;
	    for ( TQStringList::Iterator it2 = forwards.begin(); it2 != forwards.end(); ++it2 )
		ts << makeIndent( indent ) << "<forward>" << entitize( *it2 ) << "</forward>" << endl;
	    indent--;
	    ts << makeIndent( indent ) << "</forwards>" << endl;
	}
	TQStringList sigs = MetaDataBase::signalList( TQT_TQOBJECT(formwindow) );
	if ( !sigs.isEmpty() ) {
	    ts << makeIndent( indent ) << "<signals>" << endl;
	    indent++;
	    for ( TQStringList::Iterator it3 = sigs.begin(); it3 != sigs.end(); ++it3 )
		ts << makeIndent( indent ) << "<signal>" << entitize( *it3 ) << "</signal>" << endl;
	    indent--;
	    ts << makeIndent( indent ) << "</signals>" << endl;

	TQValueList<MetaDataBase::Slot> slotList = MetaDataBase::slotList( TQT_TQOBJECT(formwindow) );
	if ( !slotList.isEmpty() ) {
	    ts << makeIndent( indent ) << "<slots>" << endl;
	    indent++;
#ifndef KOMMANDER
	    TQString lang = formwindow->project()->language();
#endif
	    TQValueList<MetaDataBase::Slot>::Iterator it = slotList.begin();
	    for ( ; it != slotList.end(); ++it ) {
		MetaDataBase::Slot slot = *it;
		ts << makeIndent( indent ) << "<slot";
		if ( slot.access != "public" )
		    ts << " access=\"" << slot.access << "\"";
		if ( slot.specifier != "virtual" )
		    ts << " specifier=\"" << slot.specifier << "\"";
		if ( slot.language != "C++" )
		    ts << " language=\"" << slot.language<< "\"";
		if ( slot.returnType != "void" )
		    ts << " returnType=\"" << entitize( slot.returnType ) << "\"";
		ts << ">" << entitize( slot.slot ) << "</slot>" << endl;
	    }
	    indent--;
	    ts << makeIndent( indent ) << "</slots>" << endl;
	}
    }

    if ( formwindow && formwindow->savePixmapInline() )
	;
    else
	ts << makeIndent( indent ) << "<pixmapfunction>" << formwindow->pixmapLoaderFunction() << "</pixmapfunction>" << endl;
    if ( !( exportMacro = MetaDataBase::exportMacro( TQT_TQOBJECT(formwindow->mainContainer()) ) ).isEmpty() )
	ts << makeIndent( indent ) << "<exportmacro>" << exportMacro << "</exportmacro>" << endl;
    if ( formwindow )
	ts << makeIndent( indent ) << "<layoutdefaults spacing=\"" << formwindow->layoutDefaultSpacing()
	   << "\" margin=\"" << formwindow->layoutDefaultMargin() << "\"/>" << endl;
}

TQColorGroup Resource::loadColorGroup( const TQDomElement &e )
{
    TQColorGroup cg;
    int r = -1;
    TQDomElement n = e.firstChild().toElement();
    TQColor col;
    while ( !n.isNull() ) {
	if ( n.tagName() == "color" ) {
	    r++;
	    cg.setColor( (TQColorGroup::ColorRole)r, (col = DomTool::readColor( n ) ) );
	} else if ( n.tagName() == "pixmap" ) {
	    TQPixmap pix = loadPixmap( n );
	    cg.setBrush( (TQColorGroup::ColorRole)r, TQBrush( col, pix ) );
	}
	n = n.nextSibling().toElement();
    }
    return cg;
}

void Resource::saveChildActions( TQAction *a, TQTextStream &ts, int indent )
{
    TQObjectList clo = a->childrenListObject();
    if ( clo.isEmpty() )
	return;
    TQObjectListIt it( clo );
    while ( it.current() ) {
	TQObject *o = it.current();
	++it;
	if ( !o->inherits( TQACTION_OBJECT_NAME_STRING ) )
	    continue;
	TQAction *ac = (TQAction*)o;
	bool isGroup = ac->inherits( TQACTIONGROUP_OBJECT_NAME_STRING );
	if ( isGroup )
	    ts << makeIndent( indent ) << "<actiongroup>" << endl;
	else
	    ts << makeIndent( indent ) << "<action>" << endl;
	indent++;
	saveObjectProperties( ac, ts, indent );
	indent--;
	if ( isGroup ) {
	    indent++;
	    saveChildActions( ac, ts, indent );
	    indent--;
	}
	if ( isGroup )
	    ts << makeIndent( indent ) << "</actiongroup>" << endl;
	else
	    ts << makeIndent( indent ) << "</action>" << endl;
    }
}

void Resource::saveActions( const TQPtrList<TQAction> &actions, TQTextStream &ts, int indent )
{
    if ( actions.isEmpty() )
	return;
    ts << makeIndent( indent ) << "<actions>" << endl;
    indent++;
    TQPtrListIterator<TQAction> it( actions );
    while ( it.current() ) {
	TQAction *a = it.current();
	bool isGroup = a->inherits( TQACTIONGROUP_OBJECT_NAME_STRING );
	if ( isGroup )
	    ts << makeIndent( indent ) << "<actiongroup>" << endl;
	else
	    ts << makeIndent( indent ) << "<action>" << endl;
	indent++;
	saveObjectProperties( a, ts, indent );
	indent--;
	if ( isGroup ) {
	    indent++;
	    saveChildActions( a, ts, indent );
	    indent--;
	}
	if ( isGroup )
	    ts << makeIndent( indent ) << "</actiongroup>" << endl;
	else
	    ts << makeIndent( indent ) << "</action>" << endl;
	++it;
    }
    indent--;
    ts << makeIndent( indent ) << "</actions>" << endl;
}

void Resource::loadChildAction( TQObject *parent, const TQDomElement &e )
{
    TQDomElement n = e;
    TQAction *a = 0;
    if ( n.tagName() == "action" ) {
	a = new QDesignerAction( parent );
	MetaDataBase::addEntry( a );
	TQDomElement n2 = n.firstChild().toElement();
	while ( !n2.isNull() ) {
	    if ( n2.tagName() == "property" ) {
		setObjectProperty( a, n2.attribute( "name" ), n2.firstChild().toElement() );
	    } else if ( n2.tagName() == "event" ) {
	    	// qDebug("Resource::loadChildAction : Tag == Event");
#ifndef KOMMANDER
		MetaDataBase::setEventFunctions( a, formwindow, MainWindow::self->currProject()->language(),
						 n2.attribute( "name" ),
						 TQStringList::split( ',', n2.attribute( "functions" ) ), false );
#endif
	    }
	    n2 = n2.nextSibling().toElement();
	}
	if ( !parent->inherits( TQACTION_OBJECT_NAME_STRING ) )
	    formwindow->actionList().append( a );
    } else if ( n.tagName() == "actiongroup" ) {
	a = new QDesignerActionGroup( parent );
	MetaDataBase::addEntry( a );
	TQDomElement n2 = n.firstChild().toElement();
	while ( !n2.isNull() ) {
	    if ( n2.tagName() == "property" ) {
		setObjectProperty( a, n2.attribute( "name" ), n2.firstChild().toElement() );
	    } else if ( n2.tagName() == "action" ||
			n2.tagName() == "actiongroup" ) {
		loadChildAction( a, n2 );
	    } else if ( n2.tagName() == "event" ) {
	    	// qDebug("Resource::loadChildAction : Tag == Event");
#ifndef KOMMANDER
		MetaDataBase::setEventFunctions( a, formwindow, MainWindow::self->currProject()->language(),
						 n2.attribute( "name" ),
						 TQStringList::split( ',', n2.attribute( "functions" ) ), false );
#endif
	    }
	    n2 = n2.nextSibling().toElement();
	}
	if ( !parent->inherits( TQACTION_OBJECT_NAME_STRING ) )
	    formwindow->actionList().append( a );
    }
}

void Resource::loadActions( const TQDomElement &e )
{
    TQDomElement n = e.firstChild().toElement();
    while ( !n.isNull() ) {
	if ( n.tagName() == "action" ) {
	    loadChildAction( TQT_TQOBJECT(formwindow), n );
	} else if ( n.tagName() == "actiongroup" ) {
	    loadChildAction( TQT_TQOBJECT(formwindow), n );
	}
	n = n.nextSibling().toElement();
    }
}

void Resource::saveToolBars( TQMainWindow *mw, TQTextStream &ts, int indent )
{
    ts << makeIndent( indent ) << "<toolbars>" << endl;
    indent++;

    TQPtrList<TQToolBar> tbList;
    for ( int i = 0; i <= (int)TQt::DockMinimized; ++i ) {
	tbList = mw->toolBars( (TQt::Dock)i );
	if ( tbList.isEmpty() )
	    continue;
	for ( TQToolBar *tb = tbList.first(); tb; tb = tbList.next() ) {
	    if ( tb->isHidden() )
		continue;
	    ts << makeIndent( indent ) << "<toolbar dock=\"" << i << "\">" << endl;
	    indent++;
	    saveObjectProperties( TQT_TQOBJECT(tb), ts, indent );
	    TQPtrList<TQAction> actionList = ( (QDesignerToolBar*)tb )->insertedActions();
	    for ( TQAction *a = actionList.first(); a; a = actionList.next() ) {
		if ( a->inherits( "QSeparatorAction" ) ) {
		    ts << makeIndent( indent ) << "<separator/>" << endl;
		} else {
		    if ( a->inherits( "QDesignerAction" ) && !( (QDesignerAction*)a )->supportsMenu() ) {
			TQWidget *w = ( (QDesignerAction*)a )->widget();
			ts <<  makeIndent( indent ) << "<widget class=\""
			   << WidgetFactory::classNameOf( TQT_TQOBJECT(w) ) << "\">" << endl;
			indent++;
			saveObjectProperties( TQT_TQOBJECT(w), ts, indent );
			const char *className = WidgetFactory::classNameOf( TQT_TQOBJECT(w) );
			if ( w->isA( "CustomWidget" ) )
			    usedCustomWidgets << TQString( className );
			if ( WidgetFactory::hasItems( WidgetDatabase::idFromClassName( WidgetFactory::classNameOf( TQT_TQOBJECT(w) ) ) ) )
			    saveItems( TQT_TQOBJECT(w), ts, indent );
			indent--;
			ts << makeIndent( indent ) << "</widget>" << endl;
		    } else {
			ts <<  makeIndent( indent ) << "<action name=\"" << a->name() << "\"/>" << endl;
		    }
		}
	    }
	    indent--;
	    ts << makeIndent( indent ) << "</toolbar>" << endl;
	}
    }
    indent--;
    ts << makeIndent( indent ) << "</toolbars>" << endl;
}

void Resource::saveMenuBar( TQMainWindow *mw, TQTextStream &ts, int indent )
{
    if ( !mw->child( 0, TQMENUBAR_OBJECT_NAME_STRING ) )
	return;
    ts << makeIndent( indent ) << "<menubar>" << endl;
    indent++;
    saveObjectProperties( TQT_TQOBJECT(mw->menuBar()), ts, indent );

    for ( int i = 0; i < (int)mw->menuBar()->count(); ++i ) {
	ts << makeIndent( indent ) << "<item text=\"" << entitize( mw->menuBar()->text( mw->menuBar()->idAt( i ) ) )
	   << "\" name=\"" << entitize( mw->menuBar()->findItem( mw->menuBar()->idAt( i ) )->popup()->name() ) << "\">" << endl;
	indent++;
	TQMenuItem *m = mw->menuBar()->findItem( mw->menuBar()->idAt( i ) );
	if ( !m )
	    continue;
	TQPtrList<TQAction> actionList = ( (QDesignerPopupMenu*)m->popup() )->insertedActions();
	for ( TQAction *a = actionList.first(); a; a = actionList.next() ) {
	    if ( a->inherits( "QSeparatorAction" ) )
		ts <<  makeIndent( indent ) << "<separator/>" << endl;
	    else
		ts <<  makeIndent( indent ) << "<action name=\"" << a->name() << "\"/>" << endl;
	}
	indent--;
	ts << makeIndent( indent ) << "</item>" << endl;
    }
    indent--;
    ts << makeIndent( indent ) << "</menubar>" << endl;
}

void Resource::loadToolBars( const TQDomElement &e )
{
    TQDomElement n = e.firstChild().toElement();
    TQMainWindow *mw = ( (TQMainWindow*)formwindow->mainContainer() );
    QDesignerToolBar *tb = 0;
    while ( !n.isNull() ) {
	if ( n.tagName() == "toolbar" ) {
	    TQt::Dock dock = (TQt::Dock)n.attribute( "dock" ).toInt();
	    tb = new QDesignerToolBar( mw, dock );
	    TQDomElement n2 = n.firstChild().toElement();
	    while ( !n2.isNull() ) {
		if ( n2.tagName() == "action" ) {
		    TQAction *a = formwindow->findAction( n2.attribute( "name" ) );
		    if ( a ) {
			a->addTo( tb );
			tb->addAction( a );
		    }
		} else if ( n2.tagName() == "separator" ) {
		    TQAction *a = new QSeparatorAction( 0 );
		    a->addTo( tb );
		    tb->addAction( a );
		} else if ( n2.tagName() == "widget" ) {
		    TQWidget *w = (TQWidget*)createObject( n2, tb );
		    QDesignerAction *a = new QDesignerAction( w, TQT_TQOBJECT(tb) );
		    a->addTo( tb );
		    tb->addAction( a );
		    tb->installEventFilters( w );
		} else if ( n2.tagName() == "property" ) {
		    setObjectProperty( TQT_TQOBJECT(tb), n2.attribute( "name" ), n2.firstChild().toElement() );
		}
		n2 = n2.nextSibling().toElement();
	    }
	}
	n = n.nextSibling().toElement();
    }
}

void Resource::loadMenuBar( const TQDomElement &e )
{
    TQDomElement n = e.firstChild().toElement();
    TQMainWindow *mw = ( (TQMainWindow*)formwindow->mainContainer() );
    QDesignerMenuBar *mb = new QDesignerMenuBar( mw );
    while ( !n.isNull() ) {
	if ( n.tagName() == "item" ) {
	    QDesignerPopupMenu *popup = new QDesignerPopupMenu( mw );
	    popup->setName( n.attribute( "name" ) );
	    TQDomElement n2 = n.firstChild().toElement();
	    while ( !n2.isNull() ) {
		if ( n2.tagName() == "action" ) {
		    TQAction *a = formwindow->findAction( n2.attribute( "name" ) );
		    if ( a ) {
			a->addTo( popup );
			popup->addAction( a );
		    }
		} else if ( n2.tagName() == "separator" ) {
		    TQAction *a = new QSeparatorAction( 0 );
		    a->addTo( popup );
		    popup->addAction( a );
		}
		n2 = n2.nextSibling().toElement();
	    }
	    mb->insertItem( n.attribute( "text" ), popup );
	} else if ( n.tagName() == "property" ) {
	    setObjectProperty( TQT_TQOBJECT(mb), n.attribute( "name" ), n.firstChild().toElement() );
	}
	n = n.nextSibling().toElement();
    }
}

#ifndef KOMMANDER
void Resource::saveFormCode()
{
#ifndef KOMMANDER
	TQString lang = formwindow->project()->language();
#else
	TQString lang = "C++";
#endif
    LanguageInterface *iface = langIface;
    if ( !iface )
	return;
    TQMap<TQString, TQString> functionBodies = MetaDataBase::functionBodies( formwindow );
    if ( langIface->supports( LanguageInterface::StoreFormCodeSeperate ) ) {
	if ( formwindow->formFile()->hasTempFileName() ||
	     formwindow->formFile()->code().isEmpty() ||
	     !formwindow->formFile()->hasFormCode() )
	    return;
#ifndef KOMMANDER
	TQFile f( formwindow->project()->makeAbsolute( formwindow->formFile()->codeFile() ) );
#else
	TQFile f( formwindow->formFile()->codeFile() );
#endif
	if ( f.open( IO_WriteOnly | IO_Translate ) ) {
	    TQTextStream ts( &f );
	    ts << formwindow->formFile()->code();
	}
    } else {
	TQValueList<LanguageInterface::Function> funcs;
	TQValueList<MetaDataBase::Slot> slotList = MetaDataBase::slotList( formwindow );
	TQValueList<MetaDataBase::Slot>::Iterator sit = slotList.begin();
	for ( ; sit != slotList.end(); ++sit ) {
	    MetaDataBase::Slot slot = *sit;
	    TQMap<TQString, TQString>::Iterator it =
		functionBodies.find( MetaDataBase::normalizeSlot( (*sit).slot ) );
	    LanguageInterface::Function func;
	    func.name = slot.slot;
	    func.body = *it;
	    func.comments = MetaDataBase::functionComments( TQT_TQOBJECT(formwindow), func.name );
	    func.returnType = slot.returnType;
	    funcs.append( func );
	}

	TQValueList<LanguageInterface::Connection> conns;
	TQValueList<MetaDataBase::Connection> mconns = langConnections[ lang ];
	for ( TQValueList<MetaDataBase::Connection>::Iterator it = mconns.begin();
	      it != mconns.end(); ++it ) {
	    LanguageInterface::Connection conn;
	    conn.sender = (*it).sender->name();
	    conn.signal = (*it).signal;
	    conn.slot = (*it).slot;
	    conns.append( conn );
	}

	iface->saveFormCode( formwindow->name(), currFileName + iface->formCodeExtension(),
			     funcs, TQStringList(), TQStringList(), TQStringList(),
			     MetaDataBase::variables( formwindow ), conns );
    }
}
#endif

 // compatibility with early 3.0 betas

static TQString make_function_pretty( const TQString &s )
{
    TQString res = s;
    if ( res.find( ")" ) - res.find( "(" ) == 1 )
	return res;
    res.replace( TQRegExp( "[(]" ), "( " );
    res.replace( TQRegExp( "[)]" ), " )" );
    res.replace( TQRegExp( "&" ), " &" );
    res.replace( TQRegExp( "[*]" ), " *" );
    res.replace( TQRegExp( "," ), ", " );
    res.replace( TQRegExp( ":" ), " : " );
    res = res.simplifyWhiteSpace();
    return res;
}

#ifndef KOMMANDER
void Resource::loadFunctions( const TQDomElement &e )
{
    TQDomElement n = e.firstChild().toElement();
    TQMap<TQString, TQString> bodies;
    while ( !n.isNull() ) {
	if ( n.tagName() == "function" ) {
	    TQString name = n.attribute( "name" );
	    TQString body = n.firstChild().toText().data();
	    bodies.insert( name, body );
	}
	n = n.nextSibling().toElement();
    }
    MetaDataBase::setFunctionBodies( TQT_TQOBJECT(formwindow), bodies, TQString(), TQString() );
    if ( !bodies.isEmpty() ) {
	LanguageInterface *iface = langIface;
	if ( !iface )
	    return;
	TQString code;
	TQValueList<MetaDataBase::Slot> slotList = MetaDataBase::slotList( formwindow );
	for ( TQValueList<MetaDataBase::Slot>::Iterator it = slotList.begin(); it != slotList.end(); ++it ) {
#ifndef KOMMANDER
	    if ( (*it).language != formwindow->project()->language() )
#else
	    if ( (*it).language != "C++" )
#endif
		continue;
	    TQString sl( (*it).slot );
	    TQString comments = MetaDataBase::functionComments( TQT_TQOBJECT(formwindow), sl );
	    if ( !comments.isEmpty() )
		code += comments + "\n";
	    code += iface->createFunctionStart( formwindow->name(), make_function_pretty( sl ),
					       ( (*it).returnType.isEmpty() ?
						 TQString( "void" ) :
						 (*it).returnType ) );
	    TQMap<TQString, TQString>::Iterator bit = bodies.find( MetaDataBase::normalizeSlot( (*it).slot ) );
	    if ( bit != bodies.end() )
		code += "\n" + *bit + "\n\n";
	    else
		code += "\n" + iface->createEmptyFunction() + "\n\n";
	}
	if ( !code.isEmpty() ) {
	    formwindow->formFile()->setCode( code );
	    hasFunctions = true;
	}
    }

}

void Resource::loadExtraSource()
{
#ifndef KOMMANDER
	TQString lang = MainWindow::self->currProject()->language();
#else
	TQString lang = "C++";
#endif
    LanguageInterface *iface = langIface;
    if ( hasFunctions || !iface )
	return;
    TQValueList<LanguageInterface::Function> functions;
    TQStringList forwards;
    TQStringList includesImpl;
    TQStringList includesDecl;
    TQStringList vars;
    TQValueList<LanguageInterface::Connection> connections;

    iface->loadFormCode( formwindow->name(), currFileName + iface->formCodeExtension(),
			 functions, forwards, includesImpl, includesDecl, vars, connections );

    if ( iface->supports( LanguageInterface::StoreFormCodeSeperate ) ) {
#ifndef KOMMANDER
	TQFile f( formwindow->project()->makeAbsolute( formwindow->formFile()->codeFile() ) );
#else
	TQFile f( formwindow->formFile()->codeFile()  );
#endif
	TQString code;
	if ( f.open( IO_ReadOnly ) ) {
	    TQTextStream ts( &f );
	    code = ts.read();
	}
	formwindow->formFile()->setCode( code );
    }

    for ( TQValueList<LanguageInterface::Connection>::Iterator cit = connections.begin();
	  cit != connections.end(); ++cit ) {
	TQObject *sender  = 0;
	TQString name = (*cit).sender;
	if ( name == "this" || qstrcmp( toplevel->name(), name ) == 0 ) {
	    sender = ( (FormWindow*)toplevel )->mainContainer();
	} else {
	    if ( name == "this" )
		name = toplevel->name();
	    TQObjectList *l = toplevel->queryList( 0, name, false );
	    if ( l ) {
		if ( l->first() )
		    sender = l->first();
		delete l;
	    }
	    if ( !sender )
		sender = formwindow->findAction(  name );
	}
	MetaDataBase::setEventFunctions( sender, formwindow, lang, (*cit).signal,
					 TQStringList::split( ',', (*cit).slot ), false );
    }

    TQMap<TQString, TQString> bodies = MetaDataBase::functionBodies( formwindow );

    for ( TQValueList<LanguageInterface::Function>::Iterator fit = functions.begin();
	  fit != functions.end(); ++fit ) {
	if ( MetaDataBase::hasSlot( TQT_TQOBJECT(formwindow), (*fit).name.latin1() ) )
	    MetaDataBase::changeSlot( TQT_TQOBJECT(formwindow), (*fit).name.latin1(), (*fit).name.latin1() );
	else
	    MetaDataBase::addSlot( TQT_TQOBJECT(formwindow), (*fit).name.latin1(), "virtual", "public", lang, (*fit).returnType );
	MetaDataBase::setFunctionComments( TQT_TQOBJECT(formwindow), (*fit).name, (*fit).comments );
	bodies.insert( MetaDataBase::normalizeSlot( (*fit).name ), (*fit).body );
    }
    MetaDataBase::setFunctionBodies( TQT_TQOBJECT(formwindow), bodies, TQString(), TQString() );

    TQStringList v = MetaDataBase::variables( formwindow );
    TQStringList::Iterator vit;
    for ( vit = vars.begin(); vit != vars.end(); ++vit )
	v << *vit;
    MetaDataBase::setVariables( TQT_TQOBJECT(formwindow), v );

    TQStringList f = MetaDataBase::forwards( formwindow );
    for ( vit = forwards.begin(); vit != forwards.end(); ++vit )
	f << *vit;
    MetaDataBase::setForwards( TQT_TQOBJECT(formwindow), f );

    TQValueList<MetaDataBase::Include> incls = MetaDataBase::includes( formwindow );
    for ( vit = includesImpl.begin(); vit != includesImpl.end(); ++vit ) {
	MetaDataBase::Include inc;
	inc.header = *vit;
	inc.location = "in implementation";
	incls << inc;
    }
    for ( vit = includesDecl.begin(); vit != includesDecl.end(); ++vit ) {
	MetaDataBase::Include inc;
	inc.header = *vit;
	inc.location = "in declaration";
	incls << inc;
    }
    MetaDataBase::setIncludes( TQT_TQOBJECT(formwindow), incls );
}
#endif
