/**********************************************************************
 This file is based on TQt Designer, Copyright (C) 2000 Trolltech AS. All rights reserved.

 This file may be distributed and/or modified under the terms of the
 GNU General Public License version 2 as published by the Free Software
 Foundation and appearing in the file LICENSE.GPL included in the
 packaging of this file.

 This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.

 See http://www.trolltech.com/gpl/ for GPL licensing information.

 Modified for Kommander:
  (C) 2002-2003 Marc Britton <consume@optusnet.com.au>
  (C) 2004      Michal Rudolf <mrudolf@kdewebdev.org>

**********************************************************************/

#include "metadatabase.h"
#include "widgetfactory.h"
#include "formwindow.h"
#include "parser.h"
#include "widgetdatabase.h"
#include "formfile.h"

#include <tqapplication.h>
#include <tqobject.h>
#include <tqlayout.h>
#include <tqptrdict.h>
#include <tqobjectlist.h>
#include <tqstrlist.h>
#include <tqmetaobject.h>
#include <tqwidgetlist.h>
#include <tqmainwindow.h>
#include <tqregexp.h>
#include <tqpluginmanager_p.h>
#include <tqdatetime.h>
#include <tqfile.h>
#include <tqfileinfo.h>
#include <tqtextstream.h>

#include <stdlib.h>


class MetaDataBaseRecord
{
public:
    TQObject *object;
    TQStringList changedProperties;
    TQMap<TQString,TQVariant> fakeProperties;
    TQMap<TQString, TQString> propertyComments;
    int spacing, margin;
    TQValueList<MetaDataBase::Connection> connections;
    TQValueList<MetaDataBase::Slot> slotList;
    TQValueList<MetaDataBase::Include> includes;
    TQStringList forwards, variables, sigs;
    TQWidgetList tabOrder;
    MetaDataBase::MetaInfo metaInfo;
    TQCursor cursor;
    TQMap<int, TQString> pixmapArguments;
    TQMap<int, TQString> pixmapKeys;
    TQMap<TQString, TQString> columnFields;
    TQMap<TQString, TQStringList> eventFunctions;
    TQMap<TQString, TQString> functionBodies;
    TQMap<TQString, TQString> functionComments;
    TQValueList<int> breakPoints;
    TQString exportMacro;
};

static TQPtrDict<MetaDataBaseRecord> *db = 0;
static TQPtrList<MetaDataBase::CustomWidget> *cWidgets = 0;
static bool doUpdate = true;
static TQStringList langList;
static TQStringList editorLangList;

/*!
  \class MetaDataBase metadatabase.h
  \brief Database which stores meta data of widgets

  The MetaDataBase stores meta information of widgets, which are not
  stored directly in widgets (properties). This is e.g. the
  information which properties have been modified.
*/

MetaDataBase::MetaDataBase()
{
}

inline void setupDataBase()
{
    if ( !db || !cWidgets ) {
        db = new TQPtrDict<MetaDataBaseRecord>( 1481 );
        db->setAutoDelete( true );
        cWidgets = new TQPtrList<MetaDataBase::CustomWidget>;
        cWidgets->setAutoDelete( true );
    }
}

void MetaDataBase::clearDataBase()
{
    delete db;
    db = 0;
    delete cWidgets;
    cWidgets = 0;
}

void MetaDataBase::addEntry( TQObject *o )
{
    if ( !o )
        return;
    setupDataBase();
    if ( db->find( o ) )
        return;
    MetaDataBaseRecord *r = new MetaDataBaseRecord;
    r->object = o;
    r->spacing = r->margin = -1;
    db->insert( (void*)o, r );

    WidgetFactory::initChangedProperties( o );
}

void MetaDataBase::removeEntry( TQObject *o )
{
    setupDataBase();
    db->remove( o );
}

void MetaDataBase::setPropertyChanged( TQObject *o, const TQString &property, bool changed )
{
    setupDataBase();
    MetaDataBaseRecord *r = db->find( (void*)o );
    if ( !r ) {
        qWarning( "No entry for %p (%s, %s) found in MetaDataBase",
                  o, o->name(), o->className() );
        return;
    }

    if ( changed ) {
        if ( r->changedProperties.findIndex( property ) == -1 )
            r->changedProperties.append( property );
    } else {
        if ( r->changedProperties.findIndex( property ) != -1 )
            r->changedProperties.remove( property );
    }

    if ( doUpdate &&
         ( property == "hAlign" || property == "vAlign" || property == "wordwrap" ) ) {
        doUpdate = false;
        setPropertyChanged( o, "alignment", changed ||
                            isPropertyChanged( o, "hAlign" ) ||
                            isPropertyChanged( o, "vAlign" ) ||
                            isPropertyChanged( o, "wordwrap" ) );
        doUpdate = true;
    }

    if ( doUpdate && property == "alignment" ) {
        doUpdate = false;
        setPropertyChanged( o, "hAlign", changed );
        setPropertyChanged( o, "vAlign", changed );
        setPropertyChanged( o, "wordwrap", changed );
        doUpdate = true;
    }
}

bool MetaDataBase::isPropertyChanged( TQObject *o, const TQString &property )
{
    setupDataBase();
    MetaDataBaseRecord *r = db->find( (void*)o );
    if ( !r ) {
        qWarning( "No entry for %p (%s, %s) found in MetaDataBase",
                  o, o->name(), o->className() );
        return false;
    }

    return r->changedProperties.findIndex( property ) != -1;
}

TQStringList MetaDataBase::changedProperties( TQObject *o )
{
    setupDataBase();
    MetaDataBaseRecord *r = db->find( (void*)o );
    if ( !r ) {
        qWarning( "No entry for %p (%s, %s) found in MetaDataBase",
                  o, o->name(), o->className() );
        return TQStringList();
    }

    TQStringList lst( r->changedProperties );
    return lst;
}

void MetaDataBase::setPropertyComment( TQObject *o, const TQString &property, const TQString &comment )
{
    setupDataBase();
    MetaDataBaseRecord *r = db->find( (void*)o );
    if ( !r ) {
        qWarning( "No entry for %p (%s, %s) found in MetaDataBase",
                  o, o->name(), o->className() );
        return;
    }

    r->propertyComments.insert( property, comment );
}

TQString MetaDataBase::propertyComment( TQObject *o, const TQString &property )
{
    setupDataBase();
    MetaDataBaseRecord *r = db->find( (void*)o );
    if ( !r ) {
        qWarning( "No entry for %p (%s, %s) found in MetaDataBase",
                  o, o->name(), o->className() );
        return TQString();
    }

    return *r->propertyComments.find( property );
}

void MetaDataBase::setFakeProperty( TQObject *o, const TQString &property, const TQVariant& value )
{
    setupDataBase();
    MetaDataBaseRecord *r = db->find( (void*)o );
    if ( !r ) {
        qWarning( "No entry for %p (%s, %s) found in MetaDataBase",
                  o, o->name(), o->className() );
        return;
    }
    r->fakeProperties[property] = value;
}

TQVariant MetaDataBase::fakeProperty( TQObject * o, const TQString &property)
{
    setupDataBase();
    MetaDataBaseRecord *r = db->find( (void*)o );
    if ( !r ) {
        qWarning( "No entry for %p (%s, %s) found in MetaDataBase",
                  o, o->name(), o->className() );
        return TQVariant();
    }
    TQMap<TQString, TQVariant>::Iterator it = r->fakeProperties.find( property );
    if ( it != r->fakeProperties.end() )
        return r->fakeProperties[property];
    return WidgetFactory::defaultValue( o, property );

}

TQMap<TQString,TQVariant>* MetaDataBase::fakeProperties( TQObject* o )
{
    setupDataBase();
    MetaDataBaseRecord *r = db->find( (void*)o );
    if ( !r ) {
        qWarning( "No entry for %p (%s, %s) found in MetaDataBase",
                  o, o->name(), o->className() );
        return 0;
    }
    return &r->fakeProperties;
}

void MetaDataBase::setSpacing( TQObject *o, int spacing )
{
    if ( !o )
        return;
    setupDataBase();
    MetaDataBaseRecord *r = db->find( (void*)o );
    if ( !r || !o->isWidgetType() ) {
        qWarning( "No entry for %p (%s, %s) found in MetaDataBase",
                  o, o->name(), o->className() );
        return;
    }

    r->spacing = spacing;
    TQLayout * layout = 0;
    WidgetFactory::layoutType( (TQWidget*)o, layout );
    if ( layout )
        layout->setSpacing( spacing );
}

int MetaDataBase::spacing( TQObject *o )
{
    if ( !o )
        return -1;
    setupDataBase();
    if ( o->inherits( TQMAINWINDOW_OBJECT_NAME_STRING ) )
        o = TQT_TQOBJECT(( (TQMainWindow*)o )->centralWidget());
    MetaDataBaseRecord *r = db->find( TQT_TQOBJECT(o) );
    if ( !r || !o->isWidgetType() ) {
        qWarning( "No entry for %p (%s, %s) found in MetaDataBase",
                  o, o->name(), o->className() );
        return -1;
    }

    return r->spacing;
}

void MetaDataBase::setMargin( TQObject *o, int margin )
{
    if ( !o )
        return;
    setupDataBase();
    MetaDataBaseRecord *r = db->find( (void*)o );
    if ( !r || !o->isWidgetType() ) {
        qWarning( "No entry for %p (%s, %s) found in MetaDataBase",
                  o, o->name(), o->className() );
        return;
    }

    r->margin = margin;
    TQLayout * layout = 0;
    WidgetFactory::layoutType( (TQWidget*)o, layout );
    if ( margin < 1 )
        margin = 1;
    if ( layout )
        layout->setMargin( margin );
}

int MetaDataBase::margin( TQObject *o )
{
    if ( !o )
        return -1;
    setupDataBase();
    if ( o->inherits( TQMAINWINDOW_OBJECT_NAME_STRING ) )
        o = TQT_TQOBJECT(( (TQMainWindow*)o )->centralWidget());
    MetaDataBaseRecord *r = db->find( (void*)o );
    if ( !r || !o->isWidgetType() ) {
        qWarning( "No entry for %p (%s, %s) found in MetaDataBase",
                  o, o->name(), o->className() );
        return -1;
    }

    return r->margin;
}

void MetaDataBase::addConnection( TQObject *o, TQObject *sender, const TQCString &signal,
                                  TQObject *receiver, const TQCString &slot )
{
    setupDataBase();
    MetaDataBaseRecord *r = db->find( (void*)o );
    if ( !r ) {
        qWarning( "No entry for %p (%s, %s) found in MetaDataBase",
                  o, o->name(), o->className() );
        return;
    }
    Connection conn;
    conn.sender = sender;
    conn.signal = signal;
    conn.receiver = receiver;
    conn.slot = slot;
    r->connections.append( conn );
}

void MetaDataBase::removeConnection( TQObject *o, TQObject *sender, const TQCString &signal,
                                     TQObject *receiver, const TQCString &slot )
{
    setupDataBase();
    MetaDataBaseRecord *r = db->find( (void*)o );
    if ( !r ) {
        qWarning( "No entry for %p (%s, %s) found in MetaDataBase",
                  o, o->name(), o->className() );
        return;
    }

    for ( TQValueList<Connection>::Iterator it = r->connections.begin(); it != r->connections.end(); ++it ) {
        Connection conn = *it;
        if ( conn.sender == sender &&
             conn.signal == signal &&
             conn.receiver == receiver &&
             conn.slot == slot ) {
            r->connections.remove( it );
            break;
        }
    }
}

TQValueList<MetaDataBase::Connection> MetaDataBase::connections( TQObject *o )
{
    setupDataBase();
    MetaDataBaseRecord *r = db->find( (void*)o );
    if ( !r ) {
        qWarning( "No entry for %p (%s, %s) found in MetaDataBase",
                  o, o->name(), o->className() );
        return TQValueList<Connection>();
    }
    return r->connections;
}

TQValueList<MetaDataBase::Connection> MetaDataBase::connections( TQObject *o, TQObject *sender,
                                                                TQObject *receiver )
{
    setupDataBase();
    MetaDataBaseRecord *r = db->find( (void*)o );
    if ( !r ) {
        qWarning( "No entry for %p (%s, %s) found in MetaDataBase",
                  o, o->name(), o->className() );
        return TQValueList<Connection>();
    }
    TQValueList<Connection>::Iterator it = r->connections.begin();
    TQValueList<Connection> ret;
    TQValueList<Connection>::Iterator conn;
    while ( ( conn = it ) != r->connections.end() ) {
        ++it;
        if ( (*conn).sender == sender &&
             (*conn).receiver == receiver )
            ret << *conn;
    }

    return ret;
}

TQValueList<MetaDataBase::Connection> MetaDataBase::connections( TQObject *o, TQObject *object )
{
    setupDataBase();
    MetaDataBaseRecord *r = db->find( (void*)o );
    if ( !r ) {
        qWarning( "No entry for %p (%s, %s) found in MetaDataBase",
                  o, o->name(), o->className() );
        return TQValueList<Connection>();
    }
    TQValueList<Connection>::Iterator it = r->connections.begin();
    TQValueList<Connection> ret;
    TQValueList<Connection>::Iterator conn;
    while ( ( conn = it ) != r->connections.end() ) {
        ++it;
        if ( (*conn).sender == object ||
             (*conn).receiver == object )
            ret << *conn;
    }
    return ret;
}

void MetaDataBase::doConnections( TQObject *o )
{
    setupDataBase();
    MetaDataBaseRecord *r = db->find( (void*)o );
    if ( !r ) {
        qWarning( "No entry for %p (%s, %s) found in MetaDataBase",
                  o, o->name(), o->className() );
        return;
    }

    TQObject *sender = 0, *receiver = 0;
    TQObjectList *l = 0;
    TQValueList<Connection>::Iterator it = r->connections.begin();
    for ( ; it != r->connections.end(); ++it ) {
        Connection conn = *it;
        if ( qstrcmp( conn.sender->name(), o->name() ) == 0 ) {
            sender = o;
        } else {
            l = o->queryList( 0, conn.sender->name(), false );
            if ( !l || !l->first() ) {
                delete l;
                continue;
            }
            sender = l->first();
            delete l;
        }
        if ( qstrcmp( conn.receiver->name(), o->name() ) == 0 ) {
            receiver = o;
        } else {
            l = o->queryList( 0, conn.receiver->name(), false );
            if ( !l || !l->first() ) {
                delete l;
                continue;
            }
            receiver = l->first();
            delete l;
        }
        TQString s = "2""%1";
        s = s.arg( conn.signal.data() );
        TQString s2 = "1""%1";
        s2 = s2.arg( conn.slot.data() );

        TQStrList signalList = sender->metaObject()->signalNames( true );
        TQStrList slotList = receiver->metaObject()->slotNames( true );

        // avoid warnings
        if ( signalList.find( conn.signal ) == -1 ||
             slotList.find( conn.slot ) == -1 )
            continue;

        TQObject::connect( sender, s, receiver, s2 );
    }
}

void MetaDataBase::addSlot( TQObject *o, const TQCString &slot, const TQString& specifier,
                            const TQString &access, const TQString &language, const TQString &returnType )
{
    setupDataBase();
    MetaDataBaseRecord *r = db->find( (void*)o );
    if ( !r ) {
        qWarning( "No entry for %p (%s, %s) found in MetaDataBase",
                  o, o->name(), o->className() );
        return;
    }

    Slot s;
    s.slot = slot;
    s.specifier = specifier;
    s.access = access;
    s.language = language;
    s.returnType = returnType;
    TQValueList<MetaDataBase::Slot>::Iterator it = r->slotList.find( s );
    if ( it != r->slotList.end() )
        r->slotList.remove( it );
    r->slotList.append( s );
#ifndef KOMMANDER
    ( (FormWindow*)o )->formFile()->addSlotCode( s );
#endif
}

void MetaDataBase::setSlotList( TQObject *o, const TQValueList<Slot> &slotList )
{
    setupDataBase();
    MetaDataBaseRecord *r = db->find( (void*)o );
    if ( !r ) {
        qWarning( "No entry for %p (%s, %s) found in MetaDataBase",
                  o, o->name(), o->className() );
        return;
    }
    r->slotList = slotList;
}

void MetaDataBase::removeSlot( TQObject *o, const TQCString &slot, const TQString& specifier,
                               const TQString &access, const TQString &language, const TQString &returnType )
{
    setupDataBase();
    MetaDataBaseRecord *r = db->find( (void*)o );
    if ( !r ) {
        qWarning( "No entry for %p (%s, %s) found in MetaDataBase",
                  o, o->name(), o->className() );
        return;
    }

    for ( TQValueList<Slot>::Iterator it = r->slotList.begin(); it != r->slotList.end(); ++it ) {
        Slot s = *it;
        if ( s.slot == slot &&
             s.specifier == specifier &&
             s.access == access &&
             ( language.isEmpty() || s.language == language ) &&
               ( returnType.isEmpty() || s.returnType == returnType ) ) {
            r->slotList.remove( it );
            break;
        }
    }
}

void MetaDataBase::removeSlot( TQObject *o, const TQString &slot )
{
    setupDataBase();
    MetaDataBaseRecord *r = db->find( (void*)o );
    if ( !r ) {
        qWarning( "No entry for %p (%s, %s) found in MetaDataBase",
                  o, o->name(), o->className() );
        return;
    }

    for ( TQValueList<Slot>::Iterator it = r->slotList.begin(); it != r->slotList.end(); ++it ) {
        Slot s = *it;
        if ( normalizeSlot( s.slot ) == normalizeSlot( slot ) ) {
            r->slotList.remove( it );
            break;
        }
    }
}

TQValueList<MetaDataBase::Slot> MetaDataBase::slotList( TQObject *o )
{
    setupDataBase();
    MetaDataBaseRecord *r = db->find( (void*)o );
    if ( !r ) {
        qWarning( "No entry for %p (%s, %s) found in MetaDataBase",
                  o, o->name(), o->className() );
        return TQValueList<Slot>();
    }

    return r->slotList;
}

bool MetaDataBase::isSlotUsed( TQObject *o, const TQCString &slot )
{
    setupDataBase();
    MetaDataBaseRecord *r = db->find( (void*)o );
    if ( !r ) {
        qWarning( "No entry for %p (%s, %s) found in MetaDataBase",
                  o, o->name(), o->className() );
        return false;
    }

    TQValueList<Connection> conns = connections( o );
    for ( TQValueList<Connection>::Iterator it = conns.begin(); it != conns.end(); ++it ) {
        if ( (*it).slot == slot )
            return true;
    }
    return false;
}

void MetaDataBase::changeSlot( TQObject *o, const TQCString &slot, const TQCString &newName )
{
    setupDataBase();
    MetaDataBaseRecord *r = db->find( (void*)o );
    if ( !r ) {
        qWarning( "No entry for %p (%s, %s) found in MetaDataBase",
                  o, o->name(), o->className() );
        return;
    }

    for ( TQValueList<Slot>::Iterator it = r->slotList.begin(); it != r->slotList.end(); ++it ) {
        Slot s = *it;
        if ( normalizeSlot( s.slot ) == normalizeSlot( slot ) ) {
            (*it).slot = newName;
            return;
        }
    }
}

void MetaDataBase::changeSlotAttributes( TQObject *o, const TQCString &slot,
                                         const TQString& specifier, const TQString &access,
                                         const TQString &language, const TQString &returnType )
{
    setupDataBase();
    MetaDataBaseRecord *r = db->find( (void*)o );
    if ( !r ) {
        qWarning( "No entry for %p (%s, %s) found in MetaDataBase",
                  o, o->name(), o->className() );
        return;
    }

    for ( TQValueList<Slot>::Iterator it = r->slotList.begin(); it != r->slotList.end(); ++it ) {
        Slot s = *it;
        if ( normalizeSlot( s.slot ) == normalizeSlot( slot ) ) {
            (*it).specifier = specifier;
            (*it).access = access;
            (*it).language = language;
            (*it).returnType = returnType;
            return;
        }
    }
}

bool MetaDataBase::hasSlot( TQObject *o, const TQCString &slot, bool onlyCustom )
{
    setupDataBase();
    MetaDataBaseRecord *r = db->find( (void*)o );
    if ( !r ) {
        qWarning( "No entry for %p (%s, %s) found in MetaDataBase",
                  o, o->name(), o->className() );
        return false;
    }

    if ( !onlyCustom ) {
        TQStrList slotList = o->metaObject()->slotNames( true );
        if ( slotList.find( slot ) != -1 )
            return true;

        if ( o->inherits( "FormWindow" ) ) {
            o = TQT_TQOBJECT(( (FormWindow*)o )->mainContainer());
            slotList = o->metaObject()->slotNames( true );
            if ( slotList.find( slot ) != -1 )
                return true;
        }

        if ( o->inherits( "CustomWidget" ) ) {
            MetaDataBase::CustomWidget *w = ( (::CustomWidget*)o )->customWidget();
            for ( TQValueList<Slot>::Iterator it = w->lstSlots.begin(); it != w->lstSlots.end(); ++it ) {
                TQCString s = (*it).slot;
                if ( !s.data() )
                    continue;
                if ( s == slot )
                    return true;
            }
        }
    }

    for ( TQValueList<Slot>::Iterator it = r->slotList.begin(); it != r->slotList.end(); ++it ) {
        Slot s = *it;
        if ( normalizeSlot( s.slot ) == normalizeSlot( slot ) )
            return true;
    }

    return false;
}

TQString MetaDataBase::languageOfSlot( TQObject *o, const TQCString &slot )
{
    setupDataBase();
    MetaDataBaseRecord *r = db->find( (void*)o );
    if ( !r ) {
        qWarning( "No entry for %p (%s, %s) found in MetaDataBase",
                  o, o->name(), o->className() );
        return TQString();
    }

    TQString sl = slot;
    sl = normalizeSlot( sl );
    for ( TQValueList<Slot>::Iterator it = r->slotList.begin(); it != r->slotList.end(); ++it ) {
        Slot s = *it;
        TQString sl2 = s.slot;
        sl2 = normalizeSlot( sl2 );
        if ( sl == sl2 )
            return s.language;
    }
    return TQString();
}

bool MetaDataBase::addCustomWidget( CustomWidget *wid )
{
    setupDataBase();

    for ( CustomWidget *w = cWidgets->first(); w; w = cWidgets->next() ) {
        if ( *wid == *w ) {
            for ( TQValueList<TQCString>::ConstIterator it = wid->lstSignals.begin(); it != wid->lstSignals.end(); ++it ) {
                if ( !w->hasSignal( *it ) )
                    w->lstSignals.append( *it );
            }
            for ( TQValueList<Slot>::ConstIterator it2 = wid->lstSlots.begin(); it2 != wid->lstSlots.end(); ++it2 ) {
                if ( !w->hasSlot( MetaDataBase::normalizeSlot( (*it2).slot ).latin1() ) )
                    w->lstSlots.append( *it2 );
            }
            for ( TQValueList<Property>::ConstIterator it3 = wid->lstProperties.begin(); it3 != wid->lstProperties.end(); ++it3 ) {
                if ( !w->hasProperty( (*it3).property ) )
                    w->lstProperties.append( *it3 );
            }
            delete wid;
            return false;
        }
    }


    WidgetDatabaseRecord *r = new WidgetDatabaseRecord;
    r->name = wid->className;
    r->group = WidgetDatabase::widgetGroup( "Custom" );
    r->toolTip = wid->className;
    r->iconSet = new TQIconSet( *wid->pixmap, *wid->pixmap );
    r->isContainer = wid->isContainer;
    wid->id = WidgetDatabase::addCustomWidget( r );
    cWidgets->append( wid );
    return true;
}

void MetaDataBase::removeCustomWidget( CustomWidget *w )
{
    cWidgets->removeRef( w );
}

TQPtrList<MetaDataBase::CustomWidget> *MetaDataBase::customWidgets()
{
    setupDataBase();
    return cWidgets;
}

MetaDataBase::CustomWidget *MetaDataBase::customWidget( int id )
{
    for ( CustomWidget *w = cWidgets->first(); w; w = cWidgets->next() ) {
        if ( id == w->id )
            return w;
    }
    return 0;
}

bool MetaDataBase::isWidgetNameUsed( CustomWidget *wid )
{
    for ( CustomWidget *w = cWidgets->first(); w; w = cWidgets->next() ) {
        if ( w == wid )
            continue;
        if ( wid->className == w->className )
            return true;
    }
    return false;
}

bool MetaDataBase::hasCustomWidget( const TQString &className )
{
    for ( CustomWidget *w = cWidgets->first(); w; w = cWidgets->next() ) {
        if ( w->className == className )
            return true;
    }
    return false;
}

void MetaDataBase::setTabOrder( TQWidget *w, const TQWidgetList &order )
{
    setupDataBase();
    MetaDataBaseRecord *r = db->find( (void*) w );
    if ( !r ) {
        qWarning( "No entry for %p (%s, %s) found in MetaDataBase",
                  w, w->name(), w->className() );
        return;
    }

    r->tabOrder = order;
}

TQWidgetList MetaDataBase::tabOrder( TQWidget *w )
{
    setupDataBase();
    MetaDataBaseRecord *r = db->find( (void*) w );
    if ( !r ) {
        qWarning( "No entry for %p (%s, %s) found in MetaDataBase",
                  w, w->name(), w->className() );
        return TQWidgetList();
    }

    return r->tabOrder;
}

void MetaDataBase::setIncludes( TQObject *o, const TQValueList<Include> &incs )
{
    setupDataBase();
    MetaDataBaseRecord *r = db->find( (void*)o );
    if ( !r ) {
        qWarning( "No entry for %p (%s, %s) found in MetaDataBase",
                  o, o->name(), o->className() );
        return;
    }

    r->includes = incs;
}

TQValueList<MetaDataBase::Include> MetaDataBase::includes( TQObject *o )
{
    setupDataBase();
    MetaDataBaseRecord *r = db->find( (void*)o );
    if ( !r ) {
        qWarning( "No entry for %p (%s, %s) found in MetaDataBase",
                  o, o->name(), o->className() );
        return TQValueList<Include>();
    }

    return r->includes;
}

void MetaDataBase::setForwards( TQObject *o, const TQStringList &fwds )
{
    setupDataBase();
    MetaDataBaseRecord *r = db->find( (void*)o );
    if ( !r ) {
        qWarning( "No entry for %p (%s, %s) found in MetaDataBase",
                  o, o->name(), o->className() );
        return;
    }

    r->forwards = fwds;
}

TQStringList MetaDataBase::forwards( TQObject *o )
{
    setupDataBase();
    MetaDataBaseRecord *r = db->find( (void*)o );
    if ( !r ) {
        qWarning( "No entry for %p (%s, %s) found in MetaDataBase",
                  o, o->name(), o->className() );
        return TQStringList();
    }

    return r->forwards;
}

void MetaDataBase::setSignalList( TQObject *o, const TQStringList &sigs )
{
    setupDataBase();
    MetaDataBaseRecord *r = db->find( (void*)o );
    if ( !r ) {
        qWarning( "No entry for %p (%s, %s) found in MetaDataBase",
                  o, o->name(), o->className() );
        return;
    }

    r->sigs.clear();

    for ( TQStringList::ConstIterator it = sigs.begin(); it != sigs.end(); ++it ) {
        TQString s = (*it).simplifyWhiteSpace();
        bool hasSemicolon = s.endsWith( ";" );
        if ( hasSemicolon )
            s = s.left( s.length() - 1 );
        int p = s.find( '(' );
        if ( p < 0 )
            p = s.length();
        int sp = s.find( ' ' );
        if ( sp >= 0 && sp < p ) {
            s = s.mid( sp+1 );
            p -= sp + 1;
        }
        if ( p == (int) s.length() )
            s += "()";
        if ( hasSemicolon )
            s += ";";
        r->sigs << s;
    }
}

TQStringList MetaDataBase::signalList( TQObject *o )
{
    setupDataBase();
    MetaDataBaseRecord *r = db->find( (void*)o );
    if ( !r ) {
        qWarning( "No entry for %p (%s, %s) found in MetaDataBase",
                  o, o->name(), o->className() );
        return TQStringList();
    }

    return r->sigs;
}

void MetaDataBase::setMetaInfo( TQObject *o, MetaInfo mi )
{
    setupDataBase();
    MetaDataBaseRecord *r = db->find( (void*)o );
    if ( !r ) {
        qWarning( "No entry for %p (%s, %s) found in MetaDataBase",
                  o, o->name(), o->className() );
        return;
    }

    r->metaInfo = mi;
}

MetaDataBase::MetaInfo MetaDataBase::metaInfo( TQObject *o )
{
    setupDataBase();
    MetaDataBaseRecord *r = db->find( (void*)o );
    if ( !r ) {
        qWarning( "No entry for %p (%s, %s) found in MetaDataBase",
                  o, o->name(), o->className() );
        return MetaInfo();
    }

    return r->metaInfo;
}




MetaDataBase::CustomWidget::CustomWidget()
{
    className = "MyCustomWidget";
    includeFile = "mywidget.h";
    includePolicy = Local;
    sizeHint = TQSize( -1, -1 );
    pixmap = new TQPixmap( PixmapChooser::loadPixmap( "customwidget.xpm" ) );
    id = -1;
    sizePolicy = TQSizePolicy( TQSizePolicy::Preferred, TQSizePolicy::Preferred );
    isContainer = false;
}

MetaDataBase::CustomWidget::CustomWidget( const CustomWidget &w )
{
    className = w.className;
    includeFile = w.includeFile;
    includePolicy = w.includePolicy;
    sizeHint = w.sizeHint;
    if ( w.pixmap )
        pixmap = new TQPixmap( *w.pixmap );
    else
        pixmap = 0;
    id = w.id;
    isContainer = w.isContainer;
}

void MetaDataBase::setCursor( TQWidget *w, const TQCursor &c )
{
    setupDataBase();
    MetaDataBaseRecord *r = db->find( (void*)w );
    if ( !r ) {
        qWarning( "No entry for %p (%s, %s) found in MetaDataBase",
                  w, w->name(), w->className() );
        return;
    }

    r->cursor = c;
}

TQCursor MetaDataBase::cursor( TQWidget *w )
{
    setupDataBase();
    MetaDataBaseRecord *r = db->find( (void*)w );
    if ( !r ) {
        w->unsetCursor();
        return w->cursor();
    }

    return r->cursor;
}

bool MetaDataBase::CustomWidget::operator==( const CustomWidget &w ) const
{
    return className == w.className;
}

MetaDataBase::CustomWidget &MetaDataBase::CustomWidget::operator=( const CustomWidget &w )
{
    delete pixmap;
    className = w.className;
    includeFile = w.includeFile;
    includePolicy = w.includePolicy;
    sizeHint = w.sizeHint;
    if ( w.pixmap )
        pixmap = new TQPixmap( *w.pixmap );
    else
        pixmap = 0;
    lstSignals = w.lstSignals;
    lstSlots = w.lstSlots;
    lstProperties = w.lstProperties;
    id = w.id;
    isContainer = w.isContainer;
    return *this;
}

bool MetaDataBase::CustomWidget::hasSignal( const TQCString &signal ) const
{
    TQStrList sigList = TQWidget::staticMetaObject()->signalNames( true );
    if ( sigList.find( signal ) != -1 )
        return true;
    for ( TQValueList<TQCString>::ConstIterator it = lstSignals.begin(); it != lstSignals.end(); ++it ) {
        if ( normalizeSlot( *it ) == normalizeSlot( signal ) )
            return true;
    }
    return false;
}

bool MetaDataBase::CustomWidget::hasSlot( const TQCString &slot ) const
{
    TQStrList slotList = TQWidget::staticMetaObject()->slotNames( true );
    if ( slotList.find( normalizeSlot( slot ) ) != -1 )
        return true;

    for ( TQValueList<MetaDataBase::Slot>::ConstIterator it = lstSlots.begin(); it != lstSlots.end(); ++it ) {
        if ( normalizeSlot( (*it).slot ) == normalizeSlot( slot ) )
            return true;
    }
    return false;
}

bool MetaDataBase::CustomWidget::hasProperty( const TQCString &prop ) const
{
    TQStrList propList = TQWidget::staticMetaObject()->propertyNames( true );
    if ( propList.find( prop ) != -1 )
        return true;

    for ( TQValueList<MetaDataBase::Property>::ConstIterator it = lstProperties.begin(); it != lstProperties.end(); ++it ) {
        if ( (*it).property == prop )
            return true;
    }
    return false;
}

void MetaDataBase::setPixmapArgument( TQObject *o, int pixmap, const TQString &arg )
{
    if ( !o )
        return;
    setupDataBase();
    MetaDataBaseRecord *r = db->find( (void*)o );
    if ( !r ) {
        qWarning( "No entry for %p (%s, %s) found in MetaDataBase",
                  o, o->name(), o->className() );
        return;
    }

    r->pixmapArguments.remove( pixmap );
    r->pixmapArguments.insert( pixmap, arg );
}

TQString MetaDataBase::pixmapArgument( TQObject *o, int pixmap )
{
    if ( !o )
        return TQString();
    setupDataBase();
    MetaDataBaseRecord *r = db->find( (void*)o );
    if ( !r ) {
        qWarning( "No entry for %p (%s, %s) found in MetaDataBase",
                  o, o->name(), o->className() );
        return TQString();
    }

    return *r->pixmapArguments.find( pixmap );
}

void MetaDataBase::clearPixmapArguments( TQObject *o )
{
    if ( !o )
        return;
    setupDataBase();
    MetaDataBaseRecord *r = db->find( (void*)o );
    if ( !r ) {
        qWarning( "No entry for %p (%s, %s) found in MetaDataBase",
                  o, o->name(), o->className() );
        return;
    }

    r->pixmapArguments.clear();
}


void MetaDataBase::setPixmapKey( TQObject *o, int pixmap, const TQString &arg )
{
    if ( !o )
        return;
    setupDataBase();
    MetaDataBaseRecord *r = db->find( (void*)o );
    if ( !r ) {
        qWarning( "No entry for %p (%s, %s) found in MetaDataBase",
                  o, o->name(), o->className() );
        return;
    }

    r->pixmapKeys.remove( pixmap );
    r->pixmapKeys.insert( pixmap, arg );
}

TQString MetaDataBase::pixmapKey( TQObject *o, int pixmap )
{
    if ( !o )
        return TQString();
    setupDataBase();
    MetaDataBaseRecord *r = db->find( (void*)o );
    if ( !r ) {
        qWarning( "No entry for %p (%s, %s) found in MetaDataBase",
                  o, o->name(), o->className() );
        return TQString();
    }

    return *r->pixmapKeys.find( pixmap );
}

void MetaDataBase::clearPixmapKeys( TQObject *o )
{
    if ( !o )
        return;
    setupDataBase();
    MetaDataBaseRecord *r = db->find( (void*)o );
    if ( !r ) {
        qWarning( "No entry for %p (%s, %s) found in MetaDataBase",
                  o, o->name(), o->className() );
        return;
    }

    r->pixmapKeys.clear();
}



void MetaDataBase::setColumnFields( TQObject *o, const TQMap<TQString, TQString> &columnFields )
{
    if ( !o )
        return;
    setupDataBase();
    MetaDataBaseRecord *r = db->find( (void*)o );
    if ( !r ) {
        qWarning( "No entry for %p (%s, %s) found in MetaDataBase",
                  o, o->name(), o->className() );
        return;
    }

    r->columnFields = columnFields;
}

TQMap<TQString, TQString> MetaDataBase::columnFields( TQObject *o )
{
    if ( !o )
        return TQMap<TQString, TQString>();
    setupDataBase();
    MetaDataBaseRecord *r = db->find( (void*)o );
    if ( !r ) {
        qWarning( "No entry for %p (%s, %s) found in MetaDataBase",
                  o, o->name(), o->className() );
        return TQMap<TQString, TQString>();
    }

    return r->columnFields;
}

TQString MetaDataBase::normalizeSlot( const TQString &s )
{
    return Parser::cleanArgs( s );
}

void MetaDataBase::clear( TQObject *o )
{
    if ( !o )
        return;
    setupDataBase();
    db->remove( (void*)o );
    for ( TQPtrDictIterator<TQWidget> it( *( (FormWindow*)o )->widgets() ); it.current(); ++it )
        db->remove( (void*)it.current() );
}

void MetaDataBase::setExportMacro( TQObject *o, const TQString &macro )
{
    if ( !o )
        return;
    setupDataBase();
    MetaDataBaseRecord *r = db->find( (void*)o );
    if ( !r ) {
        qWarning( "No entry for %p (%s, %s) found in MetaDataBase",
                  o, o->name(), o->className() );
        return;
    }

    r->exportMacro = macro;
}

TQString MetaDataBase::exportMacro( TQObject *o )
{
    if ( !o )
        return "";
    setupDataBase();
    MetaDataBaseRecord *r = db->find( (void*)o );
    if ( !r ) {
        qWarning( "No entry for %p (%s, %s) found in MetaDataBase",
                  o, o->name(), o->className() );
        return "";
    }

    return r->exportMacro;
}

bool MetaDataBase::hasObject( TQObject *o )
{
    return !!db->find( o );
}

