/***************************************************************************
 *   Copyright (C) 2005 by Robert Gruber                                   *
 *   rgruber@users.sourceforge.net                                         *
 *                                                                         *
 *   This file has been taken from cervisia an adapted to fit my needs:    *
 *   Copyright (C) 1999-2002 Bernd Gehrmann <bernd@mail.berlios.de>        *
 *   Copyright (c) 2003-2005 André Wöbbeking <Woebbeking@web.de>           *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "annotateview.h"

#include <tqheader.h>
#include <tqdatetime.h>
#include <tqpainter.h>
#include <kglobalsettings.h>
#include <kglobal.h>
#include <klocale.h>
#include <kdebug.h>

#include "annotatepage.h"

class AnnotateViewItem : public TQListViewItem
{
    friend class AnnotateView;

public:
    enum { LineNumberColumn, AuthorColumn, DateColumn,ContentColumn };

    AnnotateViewItem(AnnotateView *parent, TQString rev, TQString author, 
            TQDateTime date, TQString content, TQString comment, 
            bool odd, int linenumber);

    virtual int compare(TQListViewItem *item, int col, bool ascending) const;
    virtual int width(const TQFontMetrics &, const TQListView *, int col) const;
    virtual TQString text(int col) const;
    virtual void paintCell(TQPainter *, const TQColorGroup &, int, int, int);

private:
    TQString m_revision;
    TQString m_author;
    TQString m_content;
    TQString m_comment;
    TQDateTime m_logDate;
    bool m_odd;
    int m_lineNumber;

    static const int BORDER;
};


const int AnnotateViewItem::BORDER = 4;


AnnotateViewItem::AnnotateViewItem(AnnotateView *parent, TQString rev, 
    TQString author, TQDateTime date, TQString content, TQString comment, 
    bool odd, int linenumber)
    : TQListViewItem(parent)
    , m_revision(rev)
    , m_author(author)
    , m_content(content)
    , m_comment(comment)
    , m_logDate(date)
    , m_odd(odd)
    , m_lineNumber(linenumber)
{}


int AnnotateViewItem::compare(TQListViewItem *item, int, bool) const
{
    int linenum1 = m_lineNumber;
    int linenum2 = static_cast<AnnotateViewItem*>(item)->m_lineNumber;

    return (linenum2 > linenum1)? -1 : (linenum2 < linenum1)? 1 : 0;
}


TQString AnnotateViewItem::text(int col) const
{
    switch (col)
    {
    case LineNumberColumn:
        return TQString::number(m_lineNumber);
    case AuthorColumn:
        return (m_revision + TQChar(' ') + m_author);
    case DateColumn:
        return KGlobal::locale()->formatDate(m_logDate.date(), true);
    case ContentColumn:
        return m_content;
    default:
        ;
    };

    return TQString();
}


void AnnotateViewItem::paintCell(TQPainter *p, const TQColorGroup &, int col, int width, int align)
{
    TQColor backgroundColor;

    switch (col)
    {
    case LineNumberColumn:
        backgroundColor = KGlobalSettings::highlightColor();
        p->setPen(KGlobalSettings::highlightedTextColor());
        break;
    default:
        backgroundColor = m_odd ? KGlobalSettings::baseColor()
                                : KGlobalSettings::alternateBackgroundColor();
        p->setPen(KGlobalSettings::textColor());
        break;
    };

    p->fillRect(0, 0, width, height(), backgroundColor);

    TQString str = text(col);
    if (str.isEmpty())
        return;

    if (align & (AlignTop || AlignBottom) == 0)
            align |= AlignVCenter;

    p->drawText(BORDER, 0, width - 2*BORDER, height(), align, str);
}


int AnnotateViewItem::width(const TQFontMetrics &fm, const TQListView *, int col) const
{
    return fm.width(text(col)) + 2*BORDER;
}


/******************************************************************************/
/*****************Definition of class AnnotateView ****************************/
/******************************************************************************/

AnnotateView::AnnotateView(AnnotatePage *parent, const char *name)
    : KListView(parent, name), TQToolTip( viewport() ), 
    m_page(parent)
{
    setFrameStyle(TQFrame::WinPanel | TQFrame::Sunken);
    setAllColumnsShowFocus(true);
    setShowToolTips(false);
    header()->hide();

    addColumn(TQString());
    addColumn(TQString());
    addColumn(TQString());
    addColumn(TQString());

    setSorting(AnnotateViewItem::LineNumberColumn);
    setColumnAlignment(AnnotateViewItem::LineNumberColumn, TQt::AlignRight);

    connect( this, TQT_SIGNAL(executed(TQListViewItem*)),
             this, TQT_SLOT(itemClicked(TQListViewItem*)) );
}


void AnnotateView::addLine(TQString rev, TQString author, TQDateTime date, 
        TQString content, TQString comment, bool odd)
{
    new AnnotateViewItem(this, rev, author, date, content, comment, 
            odd, childCount()+1);
}


TQSize AnnotateView::sizeHint() const
{
    TQFontMetrics fm(fontMetrics());
    return TQSize(100 * fm.width("0"), 20 * fm.lineSpacing());
}


void AnnotateView::maybeTip( const TQPoint & p )
{
    AnnotateViewItem * item = dynamic_cast<AnnotateViewItem*>( itemAt( p ) );
    if (!item)
        return;

    const int column(header()->sectionAt(p.x()));
    if (column != AnnotateViewItem::AuthorColumn &&
        column != AnnotateViewItem::DateColumn) {
        return;
    }

    TQRect r = itemRect( item );
    //get the dimension of the author + the date column
    TQRect headerRect = header()->sectionRect(AnnotateViewItem::AuthorColumn);
    headerRect = headerRect.unite(header()->sectionRect(AnnotateViewItem::DateColumn));

    r.setLeft(headerRect.left());
    r.setWidth(headerRect.width());

    if (r.isValid())
    {
        tip( r, "<nobr><b>"+item->text(AnnotateViewItem::AuthorColumn)+"</b></nobr><br>"
                "<nobr>"+item->text(AnnotateViewItem::DateColumn)+"</nobr>"
                "<pre>"+item->m_comment+"</pre>");
    }
}

void AnnotateView::itemClicked(TQListViewItem *item)
{
    kdDebug(9006) << "itemClicked()" << endl;

    AnnotateViewItem * line = dynamic_cast<AnnotateViewItem*>(item);
    if (line) {
        kdDebug(9006) << "requesting annotate for revision " << line->m_revision << endl;
        emit m_page->requestAnnotate(line->m_revision);
    } else {
        kdDebug(9006) << "This is not an AnnotateViewItem" << endl;
    }
}

#include "annotateview.moc"
