/***************************************************************************
 *   Copyright (C) 2003 by Mario Scalas                                    *
 *   mario.scalas@libero.it                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <tqlineedit.h>
#include <tqtextedit.h>
#include <tqpushbutton.h>
#include <tqtooltip.h>
#include <tqlayout.h>
#include <tqwhatsthis.h>
#include <tqlabel.h>

#include <klistview.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kdebug.h>
#include <kcombobox.h>
#include <kservicetype.h> 

#include "partexplorerformbase.h"
#include "partexplorerform.h"

///////////////////////////////////////////////////////////////////////////////
// class PropertyItem
///////////////////////////////////////////////////////////////////////////////
namespace PartExplorer{

class PropertyItem : public KListViewItem
{
public:
    PropertyItem( KListViewItem *parent, const TQString &propertyName,
        const TQString &propertyType, const TQString &propertyValue )
        : KListViewItem( parent )
    {
        setText( 0, propertyName );
        setText( 1, propertyType );
        setText( 2, propertyValue );
    }

    TQString tipText() const
    {
        TQString tip = i18n("Name: %1 | Type: %2 | Value: %3");
        return tip.arg( text(0) ).arg( text(1) ).arg( text(2) );
    }
};

}
///////////////////////////////////////////////////////////////////////////////
// class ResultsList
///////////////////////////////////////////////////////////////////////////////

class ResultList;

class ResultsToolTip: public TQToolTip
{
public:
    ResultsToolTip( ResultsList* parent );
    virtual void maybeTip( const TQPoint& p );

private:
    ResultsList* m_resultsList;
};

class ResultsList : public KListView
{
public:
    ResultsList( TQWidget *parent )
        : KListView( parent, "resultslist" )
    {
        this->setShowToolTips( false );
        new ResultsToolTip( this );
    }

    virtual ~ResultsList() {}

    void clear()
    {
        KListView::clear();
    }
};

ResultsToolTip::ResultsToolTip( ResultsList* parent )
    : TQToolTip( parent->viewport() ), m_resultsList( parent )
{
}

void ResultsToolTip::maybeTip( const TQPoint& p )
{
    PartExplorer::PropertyItem *item = dynamic_cast<PartExplorer::PropertyItem*>( m_resultsList->itemAt( p ) );
    if ( item )
    {
        TQRect r = m_resultsList->itemRect( item );
        if ( r.isValid() )
            tip( r, item->tipText() );
    }
}


///////////////////////////////////////////////////////////////////////////////
// class PartExplorerForm
///////////////////////////////////////////////////////////////////////////////

PartExplorerForm::PartExplorerForm( TQWidget *parent )
    : KDialogBase( parent, "parteplorerform", false,
        i18n("Part Explorer - A Services Lister"), User1 | Close, User1, true )
{
    m_base = new PartExplorerFormBase( this, "partexplorerformbase", 0 );
    m_resultsList = new ResultsList( m_base );
    m_resultsList->addColumn( i18n( "Property" ) );
    m_resultsList->addColumn( i18n( "Type" ) );
    m_resultsList->addColumn( i18n( "Value" ) );
    m_resultsList->setSizePolicy( TQSizePolicy( (TQSizePolicy::SizeType)3,
        (TQSizePolicy::SizeType)3, 0, 0,
        m_resultsList->sizePolicy().hasHeightForWidth() ) );
    TQWhatsThis::add( m_resultsList, i18n("<b>Matching services</b><p>Results (if any) are shown grouped by matching service name.") );
    m_base->resultsLabel->setBuddy(m_resultsList);
    m_base->layout()->add( m_resultsList );
    setMainWidget( m_base );
    m_base->typeCombo->lineEdit()->setFocus();

    // User1 button text
    setButtonText( User1, i18n("&Search") );

    // Resize dialog
    resize( 480, 512 );

//    connect( m_base->typeCombo->lineEdit(), TQT_SIGNAL(returnPressed()), this, TQT_SLOT(slotSearchRequested()) );
//    connect( m_base->constraintsText, TQT_SIGNAL(returnPressed()), this, TQT_SLOT(slotSearchRequested()) );

    connect( actionButton(User1), TQT_SIGNAL(clicked()), this, TQT_SLOT(slotSearchRequested()) );
//    connect( m_base->typeCombo->lineEdit(), TQT_SIGNAL( textChanged ( const TQString & ) ), this,  TQT_SLOT( slotServicetypeChanged( const TQString&  ) ) );
//    slotServicetypeChanged( m_base->typeCombo->lineEdit()->text() );

	// populating with all known servicetypes
	KServiceType::List serviceList = KServiceType::allServiceTypes();
	TQStringList list;
	KServiceType::List::Iterator it = serviceList.begin();
	while( it != serviceList.end() )
	{
		list << (*it)->name();
		++it;
	}
	list.sort();
	m_base->typeCombo->insertStringList( list );
}

///////////////////////////////////////////////////////////////////////////////

PartExplorerForm::~PartExplorerForm()
{
}

///////////////////////////////////////////////////////////////////////////////

void PartExplorerForm::slotSearchRequested()
{
	TQString serviceType = m_base->typeCombo->lineEdit()->text();
	TQString constraints = m_base->constraintsText->text();

    kdDebug(9000) << "===> PartExplorerForm::slotSearchRequested(): " <<
        " serviceType = " << serviceType << ", constraints = " << constraints << endl;

    // Query for requested services
    KTrader::OfferList foundServices = KTrader::self()->query( serviceType, constraints );
    fillServiceList( foundServices );
}

///////////////////////////////////////////////////////////////////////////////

void PartExplorerForm::slotDisplayError( TQString errorMessage )
{
    if (errorMessage.isEmpty())
    {
        errorMessage = i18n("Unknown error.");
    }
    KMessageBox::error( this, errorMessage );
}

///////////////////////////////////////////////////////////////////////////////

void PartExplorerForm::fillServiceList( const KTrader::OfferList &services )
{
    this->m_resultsList->clear();

    if ( services.isEmpty())
    {
        slotDisplayError( i18n("No service found matching the criteria.") );
        return;
    }

    this->m_resultsList->setRootIsDecorated( true );

    KListViewItem *rootItem = 0;

    KTrader::OfferList::ConstIterator it = services.begin();
    for ( ; it != services.end(); ++it )
    {
        KService::Ptr service = (*it);
        KListViewItem *serviceItem = new KListViewItem( this->m_resultsList, rootItem, service->name() );

        TQStringList propertyNames = service->propertyNames();
        for ( TQStringList::const_iterator it = propertyNames.begin(); it != propertyNames.end(); ++it )
        {
            TQString propertyName = (*it);
            TQVariant property = service->property( propertyName );
            TQString propertyType = property.typeName();
            TQString propertyValue;
            if (propertyType == TQSTRINGLIST_OBJECT_NAME_STRING) {
                propertyValue = property.toStringList().join(", ");
            }
            else {
                propertyValue = property.toString();
            }

            TQString dProperty = " *** Found property < %1, %2, %3 >";
            dProperty = dProperty.arg( propertyName ).arg( propertyType ).arg( propertyValue );
            kdDebug( 9000 ) << dProperty << endl;

            new PartExplorer::PropertyItem( serviceItem, propertyName, propertyType, propertyValue );
        }
    }
}

#include "partexplorerform.moc"
