/***************************************************************************
 *   Copyright (C) 1999 by Jonas Nordin                                    *
 *   jonas.nordin@syncom.se                                                *
 *   Copyright (C) 2000-2001 by Bernd Gehrmann                             *
 *   bernd@kdevelop.org                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "classtooldlg.h"

#include <tqlistbox.h>
#include <tqpushbutton.h>
#include <tqtooltip.h>
#include <kdialog.h>
#include <kiconloader.h>
#include <klocale.h>

#include "kdevlanguagesupport.h"
#include "classstore.h"
#include "parsedclass.h"
#include "classviewpart.h"


ClassToolDialog::ClassToolDialog( ClassViewPart *part )
    : TQWidget(0, "class tool dialog")
{
    currentOperation = ViewNone;
    comboAccess = (PIAccess)-1;
    currentClass = 0;
    m_part = part;

    class_combo = new TQComboBox(false, this);

    TQPushButton *close_button = new KPushButton(KStdGuiItem::close(), this);

    parents_button = new TQToolButton(this);
    parents_button->setPixmap( UserIcon("CTparents", KIcon::DefaultState, ClassViewFactory::instance()) );
    parents_button->setToggleButton(true);
    parents_button->setFixedSize(parents_button->sizeHint());
    TQToolTip::add(parents_button, i18n("Show parents"));

    children_button = new TQToolButton(this);
    children_button->setPixmap( UserIcon("CTchildren", KIcon::DefaultState, ClassViewFactory::instance()) );
    children_button->setToggleButton(true);
    children_button->setFixedSize(children_button->sizeHint());
    TQToolTip::add(children_button, i18n("Show children"));

    clients_button = new TQToolButton(this);
    clients_button->setPixmap( UserIcon("CTclients", KIcon::DefaultState, ClassViewFactory::instance()) );
    clients_button->setToggleButton(true);
    clients_button->setFixedSize(clients_button->sizeHint());
    TQToolTip::add(clients_button, i18n("Show clients"));

    suppliers_button = new TQToolButton(this);
    suppliers_button->setPixmap( UserIcon("CTsuppliers", KIcon::DefaultState, ClassViewFactory::instance()) );
    suppliers_button->setToggleButton(true);
    suppliers_button->setFixedSize(suppliers_button->sizeHint());
    TQToolTip::add(suppliers_button, i18n("Show suppliers"));

    methods_button = new TQToolButton(this);
    methods_button->setPixmap( UserIcon("CVpublic_meth", KIcon::DefaultState, ClassViewFactory::instance()) );
    methods_button->setToggleButton(true);
    methods_button->setFixedSize(methods_button->sizeHint());
    TQToolTip::add(methods_button, i18n("Show methods"));

    attributes_button = new TQToolButton(this);
    attributes_button->setPixmap( UserIcon("CVpublic_var", KIcon::DefaultState, ClassViewFactory::instance()) );
    attributes_button->setToggleButton(true);
    attributes_button->setFixedSize(attributes_button->sizeHint());
    TQToolTip::add(attributes_button, i18n("Show attributes"));

    access_combo = new TQComboBox(false, this, "access combo");
    access_combo->setMinimumWidth(100);
    access_combo->setSizeLimit(10);
    access_combo->insertItem(i18n("member access","All"));
    access_combo->insertItem(i18n("member access","Public"));
    access_combo->insertItem(i18n("member access","Protected"));
    access_combo->insertItem(i18n("member access","Private"));
    access_combo->insertItem(i18n("member access","Package"));

    class_tree = new ClassToolWidget(part, this);
    //    classTree->setMinimumSize( 500, 400 );

    TQVBoxLayout *layout = new TQVBoxLayout(this, KDialog::marginHint(), KDialog::spacingHint());
    TQHBoxLayout *firstrowLayout = new TQHBoxLayout();
    TQHBoxLayout *secondrowLayout = new TQHBoxLayout();
    layout->addLayout(firstrowLayout);
    layout->addLayout(secondrowLayout);

    firstrowLayout->addWidget(class_combo, 1);
    firstrowLayout->addWidget(close_button, 0);

    secondrowLayout->addWidget(parents_button);
    secondrowLayout->addWidget(children_button);
    secondrowLayout->addWidget(clients_button);
    secondrowLayout->addWidget(suppliers_button);
    secondrowLayout->addStretch();
    secondrowLayout->addWidget(methods_button);
    secondrowLayout->addWidget(attributes_button);
    secondrowLayout->addWidget(access_combo);

    layout->addWidget(class_tree, 10);

    connect( class_combo, TQT_SIGNAL(activated(const TQString&)),
             this, TQT_SLOT(slotClassComboChoice(const TQString&)) );
    connect( close_button, TQT_SIGNAL(clicked()),
             this, TQT_SLOT(slotClose()) );
    connect( access_combo, TQT_SIGNAL(activated(const TQString&)),
             this, TQT_SLOT(slotAccessComboChoice(const TQString&)) );
    connect( parents_button, TQT_SIGNAL(clicked()), TQT_SLOT(viewParents()));
    connect( children_button, TQT_SIGNAL(clicked()), TQT_SLOT(viewChildren()));
    connect( clients_button, TQT_SIGNAL(clicked()), TQT_SLOT(viewClients()));
    connect( suppliers_button, TQT_SIGNAL(clicked()), TQT_SLOT(viewSuppliers()));
    connect( methods_button, TQT_SIGNAL(clicked()), TQT_SLOT(viewMethods()));
    connect( attributes_button, TQT_SIGNAL(clicked()), TQT_SLOT(viewAttributes()));

    connect( part, TQT_SIGNAL(setLanguageSupport(KDevLanguageSupport*)),
             this, TQT_SLOT(setLanguageSupport(KDevLanguageSupport*)) );

    m_part->registerClassToolDialog(this);
}


ClassToolDialog::~ClassToolDialog()
{
    m_part->unregisterClassToolDialog(this);
}


void ClassToolDialog::setLanguageSupport(KDevLanguageSupport *ls)
{
    if (ls) {
        disconnect(ls, 0, this, 0);
        connect(ls, TQT_SIGNAL(updatedSourceInfo()), this, TQT_SLOT(refresh()));
    } else
        refresh();

    currentOperation = ViewNone;
}


void ClassToolDialog::setClassName(const TQString &name)
{
    if ( class_combo->count() == 0 ) refresh();

    TQListBox *lb = class_combo->listBox();

    for (int i=0; i < (int)lb->count(); ++i)
        if (lb->text(i) == name) {
            class_combo->setCurrentItem(i);
            break;
        }

    if (!name.isEmpty())
        currentClass = m_part->classStore()->getClassByName(name);
    else
        currentClass = 0;
}


void ClassToolDialog::viewNone()
{
    currentOperation = ViewNone;
    refresh();
}


/** View the parents of the current class. */
void ClassToolDialog::viewParents()
{
    currentOperation = ViewParents;
    refresh();
}


/** View the children of the current class. */
void ClassToolDialog::viewChildren()
{
    currentOperation = ViewChildren;
    refresh();
}


/** View all classes that has this class as an attribute. */
void ClassToolDialog::viewClients()
{
    currentOperation = ViewClients;
    refresh();
}


/** View all classes that this class has as attributes. */
void ClassToolDialog::viewSuppliers()
{
    currentOperation = ViewSuppliers;
    refresh();
}


/** View methods in this class and parents. */
void ClassToolDialog::viewMethods()
{
    currentOperation = ViewMethods;
    refresh();
}


/** View attributes in this class and parents. */
void ClassToolDialog::viewAttributes()
{
    currentOperation = ViewAttributes;
    refresh();
}


void ClassToolDialog::slotAccessComboChoice(const TQString &str)
{
    if( str == i18n("member access","All") )
        comboAccess = (PIAccess)-1;
    else if( str == i18n("member access","Public") )
        comboAccess = PIE_PUBLIC;
    else if( str == i18n("member access","Protected") )
        comboAccess = PIE_PROTECTED;
    else if( str == i18n("member access","Private") )
        comboAccess = PIE_PRIVATE;
    else if( str == i18n("member access","Package") )
        comboAccess = PIE_PACKAGE;

    // Update the view if the choice affected the data.
    if (currentOperation == ViewMethods || currentOperation == ViewAttributes)
        buildTree();
}


void ClassToolDialog::slotClose()
{
    delete this;
    //    TQTimer::singleShot(0, this, TQT_SLOT(delayedClose()));
}


void ClassToolDialog::delayedClose()
{
    delete this;
}


void ClassToolDialog::slotClassComboChoice(const TQString &str)
{
    setClassName(str);
    refresh();
}


void ClassToolDialog::refresh()
{
    // Clear the combo box and fill it with the new items.
    // Try to select the previously selected class

    TQString oldName = class_combo->currentText();

    class_combo->clear();
    TQStringList list = m_part->classStore()->getSortedClassNameList();
    class_combo->insertStringList(list);
    setClassName(oldName);

    // Rebuild the tree and caption/button state
    buildTree();
}


/** Change the caption depending on the current operation. */
void ClassToolDialog::updateCaptionAndButtons()
{
    TQString caption;
    TQToolButton *button;

    switch (currentOperation)
        {
        case ViewParents:
            button = parents_button;
            caption = i18n("Parents");
            break;
        case ViewChildren:
            button = children_button;
            caption = i18n("Children");
            break;
        case ViewClients:
            button = clients_button;
            caption = i18n("Clients");
            break;
        case ViewSuppliers:
            button = suppliers_button;
            caption = i18n("Suppliers");
            break;
        case ViewMethods:
            button = methods_button;
            caption = i18n("%1 Methods").arg(access_combo->currentText());
            break;
        case ViewAttributes:
            button = attributes_button;
            caption = i18n("%1 Attributes").arg(access_combo->currentText());
            break;
        default:
            button = 0;
            caption = i18n("Class Tool Dialog");
            break;
        }

    parents_button->setOn(false);
    children_button->setOn(false);
    clients_button->setOn(false);
    suppliers_button->setOn(false);
    methods_button->setOn(false);
    attributes_button->setOn(false);

    if (button) {
        button->setOn(true);
        setCaption(i18n("%1 of Class %2").arg(caption).arg(currentClass->name()));
    } else
        setCaption(caption);
}


void ClassToolDialog::buildTree()
{
    if (!currentClass)
        currentOperation = ViewNone;

    updateCaptionAndButtons();

    class_tree->clear();

    switch (currentOperation)
        {
        case ViewParents:
            class_tree->insertClassAndClasses(currentClass, currentClass->parents);
            break;
        case ViewChildren:
            {
                TQValueList<ParsedClass*> list = m_part->classStore()->getClassesByParent(currentClass->name());
                class_tree->insertClassAndClasses(currentClass, list);
            }
            break;
        case ViewClients:
            {
                TQValueList<ParsedClass*> list = m_part->classStore()->getClassClients(currentClass->name());
                class_tree->insertClassAndClasses(currentClass, list);
            }
            break;
        case ViewSuppliers:
            {
                TQValueList<ParsedClass*> list = m_part->classStore()->getClassSuppliers(currentClass->name());
                class_tree->insertClassAndClasses(currentClass, list);
            }
            break;
        case ViewMethods:
            class_tree->insertAllClassMethods(currentClass, comboAccess);
            break;
        case ViewAttributes:
            class_tree->insertAllClassAttributes(currentClass, comboAccess);
            break;
        default:
            break;
        }
}

#include "classtooldlg.moc"
