/***************************************************************************
 *   Copyright (C) 2001 by Bernd Gehrmann                                  *
 *   bernd@kdevelop.org                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <tqapplication.h>
#include <tqheader.h>
#include <tqlayout.h>
#include <tqlistview.h>
#include <tqradiobutton.h>
#include <tqvbox.h>
#include <tqvbuttongroup.h>
#include <kdebug.h>
#include <klocale.h>

#include "pgioptionsplugin.h"


const char * const optimization_flags[] = {
    "-Mautopar",
    "-Mextract",
    "-Minline",
    "-Mipa",
    0
};


const char * const hpf_flags[] = {
    "-Mbackslash",
    "-Mcmf",
    "-Mdclchk",
    "-Mextend",
    "-Mf90",
    "-Mnofree",
    "-Mstandard",
    "-Mupcase",
    "-Mbyteswapio",
    "-Mdepchk",
    "-Mdlines",
    "-Mg",
    "-Mftn",
    "-Minfo",
    "-Minform",
    "-Mkeepft",
    "-Mkeepstaticn",
    "-Mmpi",
    "-Mmpl",
    "-Mlist",
    "-Mnohpfc",
    "-Mnoindependent",
    "-Mnoprelink",
    "-Moverlap",
    "-Mpreprocess",
    "-Mprof",
    "-Mpvm",
    "-Mr8",
    "-Mrecursive",
    "-Mreplicate",
    "-Mrpm",
    "-Mrpm1",
    "-M[no]sequence",
    "-Msmp",
    "-Mstats",
    0
};



class GeneralTab : public TQWidget
{
public:
    GeneralTab( TQWidget *parent=0, const char *name=0 );
    ~GeneralTab();

    void readFlags(TQStringList *str);
    void writeFlags(TQStringList *str);
};


class OptimizationTab : public TQWidget
{
public:
    OptimizationTab( PgiOptionsPlugin::Type type, TQWidget *parent=0, const char *name=0 );
    ~OptimizationTab();

    void readFlags(TQStringList *str);
    void writeFlags(TQStringList *str);

private:
    TQRadioButton *Odefault, *O0, *O1, *O2;
    TQListView *optBox;
};


class HpfTab : public TQWidget
{
public:
    HpfTab( TQWidget *parent=0, const char *name=0 );
    ~HpfTab();

    void readFlags(TQStringList *str);
    void writeFlags(TQStringList *str);

private:
    TQListView *hpfBox;
};


OptimizationTab::OptimizationTab(PgiOptionsPlugin::Type type, TQWidget *parent, const char *name)
    : TQWidget(parent, name)
{
    TQBoxLayout *layout = new TQVBoxLayout(this, KDialog::marginHint(), KDialog::spacingHint());
    layout->setAutoAdd(true);

    TQVButtonGroup *group = new TQVButtonGroup(i18n("Optimization Level"), this);
    Odefault = new TQRadioButton(i18n("Default"), group);
    Odefault->setChecked(true);
    O0 = new TQRadioButton(i18n("No optimization"), group);
    O1 = new TQRadioButton(i18n("Level 1"), group);
    O2 = new TQRadioButton(i18n("Level 2"), group);

    if (type == PgiOptionsPlugin::PGHPF) {
        optBox = new TQListView(this);
        optBox->addColumn(TQString());
        optBox->header()->hide();
        for (const char * const *p = optimization_flags; *p; ++p) {
            new TQCheckListItem(optBox, *p, TQCheckListItem::CheckBox);
            kdDebug() << (*p) << endl;
        }
    } else
        optBox = 0;

    TQApplication::sendPostedEvents(this, TQEvent::ChildInserted);
    layout->addStretch();
}


OptimizationTab::~OptimizationTab()
{}


void OptimizationTab::readFlags(TQStringList *list)
{
    if (optBox) {
        TQListViewItem *item = optBox->firstChild();
        for (; item; item = item->nextSibling()) {
            TQStringList::Iterator sli = list->find(item->text(0));
            if (sli != list->end()) {
                static_cast<TQCheckListItem*>(item)->setOn(true);
                list->remove(sli);
            }
        }
    }
    
    TQStringList::Iterator sli;
    sli = list->find("-O0");
    if (sli != list->end()) {
        O0->setChecked(true);
        list->remove(sli);
    }
    sli = list->find("-O1");
    if (sli != list->end()) {
        O1->setChecked(true);
        list->remove(sli);
    }
    sli = list->find("-O2");
    if (sli != list->end()) {
        O2->setChecked(true);
        list->remove(sli);
    }
}


void OptimizationTab::writeFlags(TQStringList *list)
{
    if (optBox) {
        TQListViewItem *item = optBox->firstChild();
        for (; item; item = item->nextSibling())
            if (static_cast<TQCheckListItem*>(item)->isOn())
                (*list) << item->text(0);
    }
    
    if (O0->isChecked())
        (*list) << "-O0";
    else if (O1->isChecked())
        (*list) << "-O1";
    else if (O2->isChecked())
        (*list) << "-O2";
}


HpfTab::HpfTab(TQWidget *parent, const char *name)
    : TQWidget(parent, name)
{
    TQBoxLayout *layout = new TQVBoxLayout(this, KDialog::marginHint(), KDialog::spacingHint());
    layout->setAutoAdd(true);

    hpfBox = new TQListView(this);
    hpfBox->addColumn(TQString());
    hpfBox->header()->hide();
    for (const char * const *p = hpf_flags; *p; ++p)
        new TQCheckListItem(hpfBox, *p, TQCheckListItem::CheckBox);

    TQApplication::sendPostedEvents(this, TQEvent::ChildInserted);
    layout->addStretch();
}


HpfTab::~HpfTab()
{}


void HpfTab::readFlags(TQStringList *list)
{
    TQListViewItem *item = hpfBox->firstChild();
    for (; item; item = item->nextSibling()) {
        TQStringList::Iterator sli = list->find(item->text(0));
        if (sli != list->end()) {
            static_cast<TQCheckListItem*>(item)->setOn(true);
            list->remove(sli);
        }
    }
}


void HpfTab::writeFlags(TQStringList *list)
{
    TQListViewItem *item = hpfBox->firstChild();
    for (; item; item = item->nextSibling()) {
        if (static_cast<TQCheckListItem*>(item)->isOn())
            (*list) << item->text(0);
    }
}


PgiOptionsDialog::PgiOptionsDialog(PgiOptionsPlugin::Type type, TQWidget *parent, const char *name)
    : KDialogBase(Tabbed, (type == PgiOptionsPlugin::PGHPF)? i18n("PGHPF Compiler Options") : i18n("PGF77 Compiler Options"),
                  Ok|Cancel, Ok, parent, name, true)
{
    TQVBox *vbox;

    //    vbox = addVBoxPage(i18n("General"));
    //    general = new GeneralTab(vbox, "general tab");

    vbox = addVBoxPage(i18n("Optimization"));
    optimization = new OptimizationTab(type, vbox, "optimization tab");

    if (type == PgiOptionsPlugin::PGHPF) {
        vbox = addVBoxPage(i18n("HPF"));
        hpf = new HpfTab(vbox, "optimization tab");
    } else
        hpf = 0;
        
}


PgiOptionsDialog::~PgiOptionsDialog()
{
}


void PgiOptionsDialog::setFlags(const TQString &flags)
{
    TQStringList flaglist = TQStringList::split(" ", flags);

    // Hand them to 'general' at last, so it can make a line edit
    // with the unprocessed items
    if (hpf)
        hpf->readFlags(&flaglist);
    optimization->readFlags(&flaglist);
    //    general->readFlags(&flaglist);
}


TQString PgiOptionsDialog::flags() const
{
    TQStringList flaglist;

    if (hpf)
        hpf->writeFlags(&flaglist);
    optimization->writeFlags(&flaglist);
    //    general->writeFlags(&flaglist);

    TQString flags;
    TQStringList::ConstIterator li;
    for (li = flaglist.begin(); li != flaglist.end(); ++li) {
        flags += (*li);
        flags += " ";
    }

    flags.truncate(flags.length()-1);
    return flags;
}


PgiOptionsPlugin::PgiOptionsPlugin(Type type, TQObject *parent, const char *name)
    : KDevCompilerOptions(parent, name)
{
    pgitype = type;
}


PgiOptionsPlugin::~PgiOptionsPlugin()
{}


TQString PgiOptionsPlugin::exec(TQWidget *parent, const TQString &flags)
{
    PgiOptionsDialog *dlg = new PgiOptionsDialog(pgitype, parent, "pgi options dialog");
    TQString newFlags = flags;
    dlg->setFlags(flags);
    if (dlg->exec() == TQDialog::Accepted)
        newFlags = dlg->flags();
    delete dlg;
    return newFlags;
}

#include "pgioptionsplugin.moc"
