/***************************************************************************
 *   Copyright (C) 2005 by Alexander Dymo                                  *
 *   adymo@kdevelop.org                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Library General Public License as       *
 *   published by the Free Software Foundation; either version 2 of the    *
 *   License, or (at your option) any later version.                       *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU Library General Public     *
 *   License along with this program; if not, write to the                 *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.             *
 ***************************************************************************/

#include <cstdlib>
#include <iostream>
#include <stdio.h>
#include "tqmakedriver.h"
#include "tqmakeastvisitor.h"

#include <tqstring.h>

#include <kdebug.h>
#include <kcmdlineargs.h>
#include <kurl.h>

static const KCmdLineOptions options[] =
{
    {"silent", "Enable Parser debug output", 0},
    {"!debug", "Disable output of the generated AST", 0},
    {"!+files", "TQMake project files", 0}
};


class PrintAST : TQMake::ASTVisitor
{
public:
  PrintAST() : TQMake::ASTVisitor()
  {
      indent = 0;
  }

  virtual void processProject( TQMake::ProjectAST* p )
  {
      TQMake::ASTVisitor::processProject(p);
  }
private:
  virtual void enterRealProject( TQMake::ProjectAST* p )
  {
      kdDebug(9024) << getIndent() << "--------- Entering Project: " << replaceWs(p->fileName()) << "| LineEnding:" << p->lineEnding() << " --------------" << endl;
      indent += 4;
      TQMake::ASTVisitor::enterRealProject(p);
  }

  virtual void leaveRealProject( TQMake::ProjectAST* p )
  {
      indent -= 4;
      kdDebug(9024) << getIndent() << "--------- Leaving Project: " << replaceWs(p->fileName()) << " --------------" << endl;
      TQMake::ASTVisitor::leaveRealProject(p);
  }

  virtual void enterScope( TQMake::ProjectAST* p )
  {
      kdDebug(9024) << getIndent() << "--------- Entering Scope: " << replaceWs(p->scopedID) << " --------------" << endl;
      indent += 4;
      TQMake::ASTVisitor::enterScope(p);
  }

  virtual void leaveScope( TQMake::ProjectAST* p )
  {
      indent -= 4;
      kdDebug(9024) << getIndent() << "--------- Leaving Scope: " << replaceWs(p->scopedID) << " --------------" << endl;
      TQMake::ASTVisitor::leaveScope(p);
  }

  virtual void enterFunctionScope( TQMake::ProjectAST* p )
  {
      kdDebug(9024) << getIndent() << "--------- Entering FunctionScope: " << replaceWs(p->scopedID) << "(" << replaceWs(p->args) << ")"<< " --------------" << endl;
      indent += 4;
      TQMake::ASTVisitor::enterFunctionScope(p);
  }

  virtual void leaveFunctionScope( TQMake::ProjectAST* p )
  {
      indent -= 4;
      kdDebug(9024) << getIndent() << "--------- Leaving FunctionScope: " << replaceWs(p->scopedID) << "(" << replaceWs(p->args) << ")"<< " --------------" << endl;
      TQMake::ASTVisitor::leaveFunctionScope(p);
  }

  TQString replaceWs(TQString s)
  {
      return s.replace("\n", "%nl").replace("\t", "%tab").replace(" ", "%spc");
  }

  virtual void processAssignment( TQMake::AssignmentAST* a)
  {
      kdDebug(9024) << getIndent() << "Assignment(" << replaceWs(a->indent) << "):" << replaceWs(a->scopedID) << " " << replaceWs(a->op) << " " << replaceWs(a->values.join("|")) << endl;
      TQMake::ASTVisitor::processAssignment(a);
  }

  virtual void processNewLine( TQMake::NewLineAST* n)
  {
      kdDebug(9024) << getIndent() << "Newline " << endl;
      TQMake::ASTVisitor::processNewLine(n);
  }

  virtual void processComment( TQMake::CommentAST* a)
  {
      kdDebug(9024) << getIndent() << "Comment: " << replaceWs(a->comment) << endl;
      TQMake::ASTVisitor::processComment(a);
  }

  virtual void processInclude( TQMake::IncludeAST* a)
  {
      kdDebug(9024) << getIndent() << "Include: " << replaceWs(a->projectName) << endl;
      TQMake::ASTVisitor::processInclude(a);
  }

  TQString getIndent()
  {
      TQString ind;
      for( int i = 0 ; i < indent ; i++)
          ind += " ";
      return ind;
  }
  int indent;
};
int main(int argc, char *argv[])
{
  KCmdLineArgs::init( argc, argv, "TQMake Parser", "qmake-parser", "Parse TQMake project files", "1.0.0");
    KCmdLineArgs::addCmdLineOptions(options);

    KCmdLineArgs *args = KCmdLineArgs::parsedArgs();

    if( args->count() < 1 )
    {
        KCmdLineArgs::usage(0);
    }

    int debug = 0;
    bool silent = false;

    if( args->isSet("silent") )
        silent = true;
    if( args->isSet("debug") )
        debug = 1;
    for( int i = 0 ; i < args->count() ; i++ )
    {
        TQMake::ProjectAST *projectAST;
        int ret = TQMake::Driver::parseFile(args->url(i).path(), &projectAST, debug);
        PrintAST pa;
        if ( ret == 0 )
            if ( !silent )
            {
                pa.processProject(projectAST);
                TQString profile;
                projectAST->writeBack(profile);
                kdDebug(9024) << "TQMake file written back:\n" << profile << endl;
            }
        return ret;
    }
    return 0;
}
