/*
    kopeteplugindataobject.cpp - Kopete Plugin Data Object

    Copyright (c) 2003-2004 by Olivier Goffart       <ogoffart @tiscalinet.be>
    Copyright (c) 2003      by Martijn Klingens      <klingens@kde.org>
    Copyright (c) 2004      by Richard Smith         <kde@metafoo.co.uk>
    Kopete    (c) 2002-2004 by the Kopete developers <kopete-devel@kde.org>

    *************************************************************************
    *                                                                       *
    * This library is free software; you can redistribute it and/or         *
    * modify it under the terms of the GNU Lesser General Public            *
    * License as published by the Free Software Foundation; either          *
    * version 2 of the License, or (at your option) any later version.      *
    *                                                                       *
    *************************************************************************
*/

#include "kopetecontactlistelement.h"

#include <kconfig.h>
#include <kdebug.h>
#include <kglobal.h>

#include "kopeteplugin.h"

namespace Kopete {

class ContactListElement::Private
{
public:
	TQMap<TQString, TQMap<TQString, TQString> > pluginData;
	TQMap<ContactListElement::IconState, TQString> icons;
	bool useCustomIcon;
};

ContactListElement::ContactListElement( TQObject *parent, const char *name )
: TQObject( parent, name )
{
	d = new Private;

	d->useCustomIcon = false;
#if 0  //TODO
	connect( Kopete::Global::onlineStatusIconCache(), TQT_SIGNAL( iconsChanged() ), TQT_SIGNAL( iconAppearanceChanged() ) );
#endif
}

ContactListElement::~ContactListElement()
{
	delete d;
}

void ContactListElement::setPluginData( Plugin *plugin, const TQMap<TQString, TQString> &pluginData )
{
	if ( pluginData.isEmpty() )
	{
		d->pluginData.remove( plugin->pluginId() );
		return;
	}

	d->pluginData[ plugin->pluginId() ] = pluginData;

	emit pluginDataChanged();
}

void ContactListElement::setPluginData( Plugin *p, const TQString &key, const TQString &value )
{
	d->pluginData[ p->pluginId() ][ key ] = value;

	emit pluginDataChanged();
}

TQMap<TQString, TQString> ContactListElement::pluginData( Plugin *plugin ) const
{
	if ( !d->pluginData.contains( plugin->pluginId() ) )
		return TQMap<TQString, TQString>();

	return d->pluginData[ plugin->pluginId() ];
}

TQString ContactListElement::pluginData( Plugin *plugin, const TQString &key ) const
{
	if ( !d->pluginData.contains( plugin->pluginId() ) || !d->pluginData[ plugin->pluginId() ].contains( key ) )
		return TQString();

	return d->pluginData[ plugin->pluginId() ][ key ];
}

const TQValueList<TQDomElement> ContactListElement::toXML()
{
	TQDomDocument pluginData;
	TQValueList<TQDomElement> pluginNodes;
	pluginData.appendChild( pluginData.createElement( TQString::fromLatin1( "plugin-data" ) ) );

	if ( !d->pluginData.isEmpty() )
	{
		TQMap<TQString, TQMap<TQString, TQString> >::ConstIterator pluginIt;
		for ( pluginIt = d->pluginData.begin(); pluginIt != d->pluginData.end(); ++pluginIt )
		{
			TQDomElement pluginElement = pluginData.createElement( TQString::fromLatin1( "plugin-data" ) );
			pluginElement.setAttribute( TQString::fromLatin1( "plugin-id" ), pluginIt.key()  );

			TQMap<TQString, TQString>::ConstIterator it;
			for ( it = pluginIt.data().begin(); it != pluginIt.data().end(); ++it )
			{
				TQDomElement pluginDataField = pluginData.createElement( TQString::fromLatin1( "plugin-data-field" ) );
				pluginDataField.setAttribute( TQString::fromLatin1( "key" ), it.key()  );
				pluginDataField.appendChild( pluginData.createTextNode(  it.data()  ) );
				pluginElement.appendChild( pluginDataField );
			}

			pluginData.documentElement().appendChild( pluginElement );
			pluginNodes.append( pluginElement );
		}
	}
	if ( !d->icons.isEmpty() )
	{
		TQDomElement iconsElement = pluginData.createElement( TQString::fromLatin1( "custom-icons" ) );
		iconsElement.setAttribute( TQString::fromLatin1( "use" ), d->useCustomIcon ?  TQString::fromLatin1( "1" ) : TQString::fromLatin1( "0" ) );

		for ( TQMap<IconState, TQString >::ConstIterator it = d->icons.begin(); it != d->icons.end(); ++it )
		{
			TQDomElement iconElement = pluginData.createElement( TQString::fromLatin1( "icon" ) );
			TQString stateStr;
			switch ( it.key() )
			{
			case Open:
				stateStr = TQString::fromLatin1( "open" );
				break;
			case Closed:
				stateStr = TQString::fromLatin1( "closed" );
				break;
			case Online:
				stateStr = TQString::fromLatin1( "online" );
				break;
			case Away:
				stateStr = TQString::fromLatin1( "away" );
				break;
			case Offline:
				stateStr = TQString::fromLatin1( "offline" );
				break;
			case Unknown:
				stateStr = TQString::fromLatin1( "unknown" );
				break;
			case None:
			default:
				stateStr = TQString::fromLatin1( "none" );
				break;
			}
			iconElement.setAttribute( TQString::fromLatin1( "state" ), stateStr );
			iconElement.appendChild( pluginData.createTextNode( it.data() )  );
			iconsElement.appendChild( iconElement );
		}
		pluginData.documentElement().appendChild( iconsElement );
		pluginNodes.append( iconsElement );
	}
	return pluginNodes;
}

bool ContactListElement::fromXML( const TQDomElement& element )
{
	if ( element.tagName() == TQString::fromLatin1( "plugin-data" ) )
	{
		TQMap<TQString, TQString> pluginData;
		TQString pluginId = element.attribute( TQString::fromLatin1( "plugin-id" ), TQString() );

		//in kopete 0.6 the AIM protocol was called OSCAR
		if ( pluginId == TQString::fromLatin1( "OscarProtocol" ) )
			pluginId = TQString::fromLatin1( "AIMProtocol" );

		TQDomNode field = element.firstChild();
		while( !field.isNull() )
		{
			TQDomElement fieldElement = field.toElement();
			if ( fieldElement.tagName() == TQString::fromLatin1( "plugin-data-field" ) )
			{
				pluginData.insert( fieldElement.attribute( TQString::fromLatin1( "key" ),
					TQString::fromLatin1( "undefined-key" ) ), fieldElement.text() );
			}
			field = field.nextSibling();
		}
		d->pluginData.insert( pluginId, pluginData );
	}
	else if ( element.tagName() == TQString::fromLatin1( "custom-icons" ) )
	{
		d->useCustomIcon= element.attribute( TQString::fromLatin1( "use" ), TQString::fromLatin1( "1" ) ) == TQString::fromLatin1( "1" );
		TQDomNode ic = element.firstChild();
		while( !ic.isNull() )
		{
			TQDomElement iconElement = ic.toElement();
			if ( iconElement.tagName() == TQString::fromLatin1( "icon" ) )
			{
				TQString stateStr = iconElement.attribute( TQString::fromLatin1( "state" ), TQString() );
				TQString icon = iconElement.text();
				IconState state = None;

				if ( stateStr == TQString::fromLatin1( "open" ) )
					state = Open;
				if ( stateStr == TQString::fromLatin1( "closed" ) )
					state = Closed;
				if ( stateStr == TQString::fromLatin1( "online" ) )
					state = Online;
				if ( stateStr == TQString::fromLatin1( "offline" ) )
					state = Offline;
				if ( stateStr == TQString::fromLatin1( "away" ) )
					state = Away;
				if ( stateStr == TQString::fromLatin1( "unknown" ) )
					state = Unknown;

				d->icons[ state ] = icon;
			}
			ic = ic.nextSibling();
		}
	}
	else
	{
		return false;
	}

	return true;
}

TQString ContactListElement::icon( ContactListElement::IconState state ) const
{
	if ( d->icons.contains( state ) )
		return d->icons[state];

	return d->icons[ None ];
}

void ContactListElement::setIcon( const TQString& icon , ContactListElement::IconState state )
{
	if ( icon.isNull() )
		d->icons.remove( state );
	else
		d->icons[ state ] = icon;

	emit iconChanged( state, icon );
	emit iconAppearanceChanged();
}

bool ContactListElement::useCustomIcon() const
{
	return d->useCustomIcon;
}

void ContactListElement::setUseCustomIcon( bool useCustomIcon )
{
	if ( d->useCustomIcon != useCustomIcon )
	{
		d->useCustomIcon = useCustomIcon;
		emit useCustomIconChanged( useCustomIcon );
	}
}

} //END namespace Kopete

#include "kopetecontactlistelement.moc"



