#include "field.h"

#include <tqwhatsthis.h>
#include <tqlayout.h>
#include <tqlabel.h>
#include <tqpushbutton.h>

#include <klocale.h>
#include <kcanvasrootpixmap.h>
#include <knotifyclient.h>
#include <kgamelcd.h>

#include "factory.h"
#include "board.h"
#include "baseprefs.h"


const char *BaseField::BUTTON_TEXTS[NB_BUTTON_TYPE] = {
    I18N_NOOP("Start"), I18N_NOOP("Resume"), I18N_NOOP("Proceed")
};

BaseField::BaseField(TQWidget *w)
    : _widget(w), _boardLayout(0), _label(0), _button(0)
{
    top = new TQGridLayout(w, 3, 5, 10);

    lcds = new TQGridLayout(7, 1, 5);
    top->addLayout(lcds, 1, 0);
    lcds->setRowStretch(1, 0);

    board = bfactory->createBoard(true, w);
    _boardRootPixmap = new KCanvasRootPixmap(board);
    _boardRootPixmap->start();
    top->addWidget(board, 1, 2);
}

void BaseField::init(bool AI, bool multiplayer, bool server, bool first,
                     const TQString &name)
{
    _flags.AI = AI;
    _flags.multiplayer = multiplayer;
    _flags.server = server;
    _flags.first = first;
	TQString text = (AI ? i18n("%1\n(AI player)").arg(name)
                    : (multiplayer ? i18n("%1\n(Human player)").arg(name)
                       : TQString()));
	if ( first && !server ) text += i18n("\nWaiting for server");
	setMessage(text, (first && server ? StartButton : NoButton));
    showScore->resetColor();
    board->init(false);
}

void BaseField::setArcade()
{
    board->init(true);
    setMessage(i18n("Stage #1"), StartButton);
}

bool BaseField::isArcade() const
{
    return board->isArcade();
}

void BaseField::setMessage(const TQString &label, ButtonType type)
{
    delete _label;
    _label = 0;
    delete _button;
    _button = 0;
    delete _boardLayout;
    _boardLayout = 0;

    if ( label.isEmpty() && type==NoButton ) {
        _widget->setFocus();
        return;
    }

    _boardLayout = new TQVBoxLayout(board);
    _boardLayout->addStretch(3);
    if ( !label.isEmpty() ) {
        TQString str = (isArcade() ? i18n("Arcade game") + '\n'
                       : TQString()) + label;
        _label = new TQLabel(str, board);
        _label->setAlignment(TQt::AlignCenter);
        _label->setFrameStyle( TQFrame::Panel | TQFrame::Sunken );
        _boardLayout->addWidget(_label, 0, TQt::AlignCenter);
        _label->show();
    }
    _boardLayout->addStretch(1);
    if ( type!=NoButton ) {
        _button = new TQPushButton(i18n(BUTTON_TEXTS[type]), board);
        _button->setFocus();
        const char *slot = (type==ResumeButton ? TQT_SLOT(pause())
                            : TQT_SLOT(start()));
        _button->connect(_button, TQT_SIGNAL(clicked()),
                              _widget->parent(), slot);
        _boardLayout->addWidget(_button, 0, TQt::AlignCenter);
        _button->show();
    }
    _boardLayout->addStretch(3);
}

void BaseField::start(const GTInitData &data)
{
    _firstScore = KExtHighscore::firstScore();
    _lastScore = KExtHighscore::lastScore();
    hideMessage();
	board->start(data);
}

void BaseField::pause(bool pause)
{
	if (pause) {
		board->pause();
        setMessage(i18n("Game paused"), ResumeButton);
	} else {
		board->unpause();
        hideMessage();
	}
}

void BaseField::stop(bool gameover)
{
	board->stop();
    ButtonType button = StartButton;
    TQString msg = (gameover ? i18n("Game over") : TQString());
    if ( board->isArcade() && board->arcadeStageDone() ) {
        if ( board->arcadeStage()==bfactory->bbi.nbArcadeStages )
            msg = i18n("The End");
        else {
            msg = i18n("Stage #%1 done").arg(board->arcadeStage());
            button = ProceedButton;
        }
    }
    setMessage(msg, button);
}

void BaseField::gameOver(const KExtHighscore::Score &score, TQWidget *parent)
{
    KNotifyClient::event(parent->winId(), "game over", i18n("Game Over"));
    KExtHighscore::submitScore(score, parent);
}

void BaseField::scoreUpdated()
{
    showScore->display( (int)board->score() );
    if (_flags.multiplayer) return;

    TQColor color;
    if ( _firstScore<currentScore() ) color = TQt::red;
    else if ( _lastScore<currentScore() ) color = TQt::blue;
    showScore->setColor(color);
}

void BaseField::settingsChanged()
{
    TQColor color = BasePrefs::fadeColor();
    double s = BasePrefs::fadeIntensity();
    _boardRootPixmap->setFadeEffect(s, color);
    board->canvas()->setBackgroundColor(color);
    board->settingsChanged();
}
