/***************************************************************************
    smb4kcore  -  The main core class of Smb4K.
                             -------------------
    begin                : Do Apr 8 2004
    copyright            : (C) 2004-2007 by Alexander Reinholdt
    email                : dustpuppy@users.berlios.de
 ***************************************************************************/

/***************************************************************************
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,   *
 *   MA  02110-1301 USA                                                    *
 ***************************************************************************/

#ifndef SMB4KCORE_H
#define SMB4KCORE_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// TQt includes
#include <tqobject.h>
#include <tqmap.h>
#include <tqtimer.h>
#include <tqvaluelist.h>

// application specific includes
#include "smb4kscanner.h"
#include "smb4kmounter.h"
#include "smb4kfileio.h"
#include "smb4kbookmarkhandler.h"
#include "smb4kprint.h"
#include "smb4ksynchronizer.h"
#include "smb4kpreviewer.h"

// forward declarations
class Smb4KWorkgroupItem;
class Smb4KHostItem;
class Smb4KShare;


/**
 * This is the top-level core class. It inherits most of the core classes.
 *
 * @author Alexander Reinholdt <dustpuppy@users.berlios.de>
 */

class Smb4KCore : public TQObject
{
  Q_OBJECT
  TQ_OBJECT

  public:
    /**
     * Returns a static pointer to this class.
     */
    static Smb4KCore *self();

    /**
     * The destructor.
     */
    ~Smb4KCore();

    /**
     * This function registers with the DCOP server and starts the core.
     * It initializes the scanning of the network, the remounting of
     * recently used shares, etc. Use this, if you want to take advantage
     * of the automatic features of Smb4K's core.
     */
    void init();

    /**
     * Returns the state the scanner is in.
     */
    static int scannerState() { return self()->m_scanner_state; }

    /**
     * Returns TRUE, if the scanner is running, otherwise FALSE.
     */
    static bool scannerIsRunning() { return self()->m_scanner->isRunning(); }

    /**
     * Returns the state the mounter is in.
     */
    static int mounterState() { return self()->m_mounter_state; }

    /**
     * Returns TRUE, if the mounter is running, otherwise FALSE.
     */
    static bool mounterIsRunning() { return self()->m_mounter->isRunning(); }

    /**
     * Returns the state the print object is in.
     */
    static int printState() { return self()->m_print_state; }

    /**
     * Returns TRUE, if the print object is running, otherwise FALSE.
     */
    static bool printIsRunning() { return self()->m_print->isRunning(); }

    /**
     * Returns the state the synchronizer is in.
     */
    static int synchronizerState() { return self()->m_syncer_state; }

    /**
     * Returns TRUE, if the synchronizer is running, otherwise FALSE.
     */
    static bool synchronizerIsRunning() { return self()->m_synchronizer->isRunning(); }

    /**
     * Returns TRUE if one of the core classes is doing something.
     */
    static bool isRunning();

    /**
     * Returns the current state the core is in.
     */
    static int currentState() { return self()->m_current_state; }

    /**
     * Returns a pointer to the scanner object.
     */
    static Smb4KScanner *scanner() { return self()->m_scanner; }

    /**
     * Returns a pointer to the mounter object.
     */
    static Smb4KMounter *mounter() { return self()->m_mounter; }

    /**
     * Returns a pointer to the bookmark handler object.
     */
    static Smb4KBookmarkHandler *bookmarkHandler() { return self()->m_bookmarkHandler; }

    /**
     * Returns a pointer to the file IO object.
     */
    static Smb4KFileIO *fileIO() { return self()->m_fileIO; }

    /**
     * Returns a pointer to the printer handler object.
     */
    static Smb4KPrint *print() { return self()->m_print; }

    /**
     * Returns a pointer to the synchronizer object.
     */
    static Smb4KSynchronizer *synchronizer() { return self()->m_synchronizer; }

    /**
     * Returns a pointer to the previewer object.
     */
    static Smb4KPreviewer *previewer() { return self()->m_previewer; }

    /**
     * Aborts any action of the core.
     */
    static void abort();

    /**
     * This enumeration determines with which application the mount point
     * of the current mounted share is to be opened.
     */
    enum OpenWith { Konqueror, Konsole };

    /**
     * Opens the mount point of a share. This function is used to run the file manager.
     *
     * @param share         The share to be opened.
     *
     * @param openWith      Integer of type Smb4KCore::OpenWith. Determines with which
     *                      application the share should be opened.
     */
    static void open( Smb4KShare *share, int openWith = Konqueror );

  signals:
    /**
     * This signal is emitted, if one of the core objects
     * starts or stops running.
     */
    void runStateChanged();

  protected:
    /**
     * The constructor
     */
    Smb4KCore();

    /**
     * A static pointer to this class
     */
    static Smb4KCore *m_self;

  protected slots:
    /**
     * This slot sets the run state of the scanner.
     *
     * @param state         The run state the scanner is in. For definition of the
     *                      run states see smb4kdefs.h header file.
     */
    void slotSetScannerState( int state );

    /**
     * This slot sets the run state of the mounter.
     *
     * @param state         The run state the mounter is in. For definition of the
     *                      run states see smb4kdefs.h header file.
     */
    void slotSetMounterState( int state );

    /**
     * This slot sets the run state of the synchronizer.
     *
     * @param state         The run state the synchronizer is in. For definition of the
     *                      run states see smb4kdefs.h header file.
     */
    void slotSetSynchronizerState( int state );

    /**
     * This slot sets the run state of the print handler.
     *
     * @param state         The run state the print handler is in. For definition of the
     *                      run states see smb4kdefs.h header file.
     */
    void slotSetPrinterHandlerState( int state );

    /**
     * This slot sets the run state for the previewer.
     *
     * @param state         The state the previewer is in. For definition of the
     *                      run states see smb4kdefs.h header file.
     */
    void slotSetPreviewerState( int state );

    /**
     * This slot is connected to the KApplication::shutDown() signal. It is invoked
     * when the application is shut down by the KDE logout or by pressing CTRL+Q, etc.
     */
    void slotShutdown();

  private:
    /**
     * The scanner object.
     */
    Smb4KScanner *m_scanner;

    /**
     * The mounter object.
     */
    Smb4KMounter *m_mounter;

    /**
     * The bookmark handler object.
     */
    Smb4KBookmarkHandler *m_bookmarkHandler;

    /**
     * The FileIO object.
     */
    Smb4KFileIO *m_fileIO;

    /**
     * The printer handler object.
     */
    Smb4KPrint *m_print;

    /**
     * The synchronizer object.
     */
    Smb4KSynchronizer *m_synchronizer;

    /**
     * The previewer object.
     */
    Smb4KPreviewer *m_previewer;

    /**
     * The state the sanner is in.
     */
    int m_scanner_state;

    /**
     * The state the mounter is in.
     */
    int m_mounter_state;

    /**
     * Holds the current state.
     */
    int m_current_state;

    /**
     * The state the printer handler is in.
     */
    int m_print_state;

    /**
     * The state the synchronizer is in.
     */
    int m_syncer_state;

    /**
     * The state of the previewer is in.
     */
    int m_previewer_state;

    /**
     * This function is used to set the current state of the core.
     *
     * @param state         One of the states defined in smb4kdefs.h
     */
    void setCurrentState( int state );

    /**
     * Searches for the needed programs and emits an error
     * if mandatory ones are missing.
     */
    void searchPrograms();

    /**
     * This is the list of workgroups known to Smb4K. It is filled by
     * the scanner and may be used by all other classes that are invoked
     * by Smb4KCore.
     */
    TQValueList<Smb4KWorkgroupItem *> m_workgroups;

    /**
     * This is the list of remote hosts known to Smb4K. It is filled by
     * the scanner and may be used by all other classes that are invoked
     * by Smb4KCore.
     */
    TQValueList<Smb4KHostItem *> m_hosts;

    /**
     * Set default values for settings that depend on the system Smb4K is
     * running on and that have to be set dynamically.
     */
    void setDefaultSettings();
};

#endif
