/***************************************************************************
    smb4kbookmarkhandler  -  This class handles the bookmarks.
                             -------------------
    begin                : Fr Jan 9 2004
    copyright            : (C) 2004 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,   *
 *   MA  02110-1301 USA                                                    *
 ***************************************************************************/

#ifndef SMB4KBOOKMARKHANDLER_H
#define SMB4KBOOKMARKHANDLER_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// TQt includes
#include <tqobject.h>
#include <tqvaluelist.h>

// forward declarations
class Smb4KHostItem;
class Smb4KBookmark;


/**
 * This class belongs the to core classes of Smb4K and manages the
 * bookmarks.
 *
 * @author         Alexander Reinholdt <dustpuppy@mail.berlios.de>
 */

class Smb4KBookmarkHandler : public TQObject
{
  Q_OBJECT
  TQ_OBJECT

  public:
    /**
     * The constructor.
     *
     * @param hosts         A list of Smb4KHostItem items. This should be used
     *                      to give the bookmark handler the list of known hosts,
     *                      so it is able to update the bookmarks if necessary.
     *                      In case a NULL pointer is passed, the bookmarks won't
     *                      be updated.
     *
     * @param parent        The parent of this object
     *
     * @param name          This object's name
     */
    Smb4KBookmarkHandler( TQValueList<Smb4KHostItem *> *hosts = 0,
                          TQObject *parent = 0,
                          const char *name = 0 );

    /**
     * The destructor.
     */
    ~Smb4KBookmarkHandler();

    /**
     * This function writes a bookmark to the bookmark file.
     *
     * NOTE: This function is more or less a wrapper around the
     * writeBookmarList() function.
     *
     * @param bookmark      A bookmark that is to be written to
     *                      the bookmark file
     */
    void addBookmark( Smb4KBookmark *bookmark );

    /**
     * This function writes a new list of bookmarks. The old list will be
     * deleted. It should be used, if you manipulated the list of bookmarks
     * i. e. by a bookmark editor. When this function finishes, the
     * bookmarksUpdated() signal will be emitted.
     *
     * @param list          The (new) list of bookmarks that is to be written
     *                      to the bookmark file
     */
    void writeBookmarkList( const TQValueList<Smb4KBookmark *> &list );

    /**
     * Get the list of bookmarks.
     *
     * @returns             The current list of bookmarks stored in the
     *                      bookmark file.
     */
    const TQValueList<Smb4KBookmark *> &getBookmarks();

    /**
     * This function searches for a bookmark using its name (//HOST/SHARE) and
     * returns a pointer to it if it is present or NULL.
     *
     * @param bookmark      The bookmark that is searched. The string you enter
     *                      must consist of the host and the share: //HOST/SHARE.
     *
     * @returns             The bookmark object that was searched for or NULL if it
     *                      wasn't found.
     */
    Smb4KBookmark *findBookmarkByName( const TQString &bookmark );

    /**
     * This function searches for a bookmark using its label and returns a pointer
     * to it if it is present or NULL.
     *
     * @param label         The label that is searched.
     *
     * @returns             The bookmark object that was searched for or NULL if it
     *                      wasn't found.
     */
    Smb4KBookmark *findBookmarkByLabel( const TQString &label );

  signals:
    /**
     * Signal emitted when the list of bookmarks has been updated.
     */
    void bookmarksUpdated();

  private:
    /**
     * The list of bookmarks.
     */
    TQValueList<Smb4KBookmark *> m_bookmarks;

    /**
     * This function loads the list of bookmarks from the bookmarks file.
     * When it finishes, the bookmarksUpdated() signal is emitted. So, if you
     * want to access the list of bookmarks immediately after they were read,
     * connect a slot to that signal.
     */
    void loadBookmarks();

    /**
     * This lis is a pointer to a global list of all known hosts. It
     * is used to update the bookmarks.
     */
    TQValueList<Smb4KHostItem *> *m_hosts;

    /**
     * This function updates the data of the bookmarks, i.e. is searches for
     * the host provided by m_hosts and sets the appropriate data, if
     * necessary.
     */
    void update();
};
#endif
