/* -*- c-basic-offset: 4 indent-tabs-mode: nil -*- vi:set ts=8 sts=4 sw=4: */

/*
    Rosegarden
    A MIDI and audio sequencer and musical notation editor.
 
    This program is Copyright 2000-2008
        Guillaume Laurent   <glaurent@telegraph-road.org>,
        Chris Cannam        <cannam@all-day-breakfast.com>,
        Richard Bown        <richard.bown@ferventsoftware.com>
 
    The moral rights of Guillaume Laurent, Chris Cannam, and Richard
    Bown to claim authorship of this work have been asserted.
 
    Other copyrights also apply to some parts of this work.  Please
    see the AUTHORS file and individual file headers for details.
 
    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License as
    published by the Free Software Foundation; either version 2 of the
    License, or (at your option) any later version.  See the file
    COPYING included with this distribution for more information.
*/


#include "StandardRuler.h"

#include <klocale.h>
#include "misc/Debug.h"
#include "MarkerRuler.h"
#include "base/RulerScale.h"
#include "document/RosegardenGUIDoc.h"
#include "document/MultiViewCommandHistory.h"
#include "gui/application/RosegardenGUIApp.h"
#include "gui/general/GUIPalette.h"
#include "gui/rulers/LoopRuler.h"
#include "document/RosegardenGUIDoc.h"
#include <tqobject.h>
#include <tqtooltip.h>
#include <tqvbox.h>
#include <tqwidget.h>


namespace Rosegarden
{

StandardRuler::StandardRuler(RosegardenGUIDoc *doc,
                             RulerScale *rulerScale,
                             double xorigin,
                             int barHeight,
                             bool invert,
                             TQWidget* parent,
                             const char* name,
                             WFlags f):
        TQVBox(parent, name, f),
        m_invert(invert),
        m_loopRulerHeight(10),
        m_currentXOffset(0),
        m_doc(doc),
        m_rulerScale(rulerScale),
        m_hButtonBar(0)
{
    setSpacing(0);

    if (!m_invert) {
        m_hButtonBar = new MarkerRuler
                       (m_doc, m_rulerScale, barHeight - m_loopRulerHeight, xorigin, this);
    }

    m_loopRuler = new LoopRuler
                  (m_doc, m_rulerScale, m_loopRulerHeight, xorigin, m_invert, this, name);

    if (m_invert) {
        m_hButtonBar = new MarkerRuler
                       (m_doc, m_rulerScale, barHeight - m_loopRulerHeight, xorigin, this);
    }

    TQObject::connect
        (doc->getCommandHistory(), TQT_SIGNAL(commandExecuted()),
         this, TQT_SLOT(update()));

}

void StandardRuler::setSnapGrid(SnapGrid *grid)
{
    m_loopRuler->setSnapGrid(grid);
}

void StandardRuler::connectRulerToDocPointer(RosegardenGUIDoc *doc)
{

    RG_DEBUG << "StandardRuler::connectRulerToDocPointer" << endl;

    // use the document as a hub for pointer and loop set related signals
    // pointer and loop drag signals are specific to the current view,
    // so they are re-emitted from the loop ruler by this widget
    //
    TQObject::connect
    (m_loopRuler, TQT_SIGNAL(setPointerPosition(timeT)),
     doc, TQT_SLOT(slotSetPointerPosition(timeT)));

    TQObject::connect
    (m_hButtonBar, TQT_SIGNAL(setPointerPosition(timeT)),
     doc, TQT_SLOT(slotSetPointerPosition(timeT)));

    TQObject::connect
    (m_hButtonBar, TQT_SIGNAL(editMarkers()),
     RosegardenGUIApp::self(), TQT_SLOT(slotEditMarkers()));

    TQObject::connect
    (m_hButtonBar, TQT_SIGNAL(addMarker(timeT)),
     RosegardenGUIApp::self(), TQT_SLOT(slotAddMarker(timeT)));

    TQObject::connect
    (m_hButtonBar, TQT_SIGNAL(deleteMarker(int, timeT, TQString, TQString)),
     RosegardenGUIApp::self(), TQT_SLOT(slotDeleteMarker(int, timeT, TQString, TQString)));

    TQObject::connect
    (m_loopRuler, TQT_SIGNAL(dragPointerToPosition(timeT)),
     this, TQT_SIGNAL(dragPointerToPosition(timeT)));

    TQObject::connect
    (m_loopRuler, TQT_SIGNAL(dragLoopToPosition(timeT)),
     this, TQT_SIGNAL(dragLoopToPosition(timeT)));

    TQObject::connect
    (m_loopRuler, TQT_SIGNAL(setPlayPosition(timeT)),
     RosegardenGUIApp::self(), TQT_SLOT(slotSetPlayPosition(timeT)));

    TQObject::connect
    (m_hButtonBar, TQT_SIGNAL(setLoop(timeT, timeT)),
     doc, TQT_SLOT(slotSetLoop(timeT, timeT)));

    TQObject::connect
    (m_loopRuler, TQT_SIGNAL(setLoop(timeT, timeT)),
     doc, TQT_SLOT(slotSetLoop(timeT, timeT)));

    TQObject::connect
    (doc, TQT_SIGNAL(loopChanged(timeT, timeT)),
     m_loopRuler,
     TQT_SLOT(slotSetLoopMarker(timeT, timeT)));

    m_loopRuler->setBackgroundColor(GUIPalette::getColour(GUIPalette::PointerRuler));
}

void StandardRuler::slotScrollHoriz(int x)
{
    m_loopRuler->scrollHoriz(x);
    m_hButtonBar->scrollHoriz(x);
}

void StandardRuler::setMinimumWidth(int width)
{
    m_hButtonBar->setMinimumWidth(width);
    m_loopRuler->setMinimumWidth(width);
}

void StandardRuler::setHScaleFactor(double dy)
{
    m_hButtonBar->setHScaleFactor(dy);
    m_loopRuler->setHScaleFactor(dy);
}

void StandardRuler::paintEvent(TQPaintEvent *e)
{
    m_hButtonBar->update();
    m_loopRuler->update();
    TQWidget::paintEvent(e);
}

}
#include "StandardRuler.moc"
