
/* -*- c-basic-offset: 4 indent-tabs-mode: nil -*- vi:set ts=8 sts=4 sw=4: */

/*
    Rosegarden
    A MIDI and audio sequencer and musical notation editor.

    This program is Copyright 2000-2008
        Guillaume Laurent   <glaurent@telegraph-road.org>,
        Chris Cannam        <cannam@all-day-breakfast.com>,
        Richard Bown        <richard.bown@ferventsoftware.com>

    The moral rights of Guillaume Laurent, Chris Cannam, and Richard
    Bown to claim authorship of this work have been asserted.

    Other copyrights also apply to some parts of this work.  Please
    see the AUTHORS file and individual file headers for details.

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License as
    published by the Free Software Foundation; either version 2 of the
    License, or (at your option) any later version.  See the file
    COPYING included with this distribution for more information.
*/

#ifndef _RG_EVENTVIEW_H_
#define _RG_EVENTVIEW_H_

#include "base/MidiTypes.h"
#include "base/NotationTypes.h"
#include "base/Segment.h"
#include "gui/general/EditViewBase.h"
#include <set>
#include <tqsize.h>
#include <tqstring.h>
#include <vector>
#include "base/Event.h"


class TQWidget;
class TQPopupMenu;
class TQPoint;
class TQListViewItem;
class TQLabel;
class TQCheckBox;
class TQButtonGroup;
class KListView;


namespace Rosegarden
{

class Segment;
class RosegardenGUIDoc;
class Event;


class EventView : public EditViewBase, public SegmentObserver
{
    Q_OBJECT
  TQ_OBJECT

    // Event filters
    //
    enum EventFilter
    {
        None               = 0x0000,
        Note               = 0x0001,
        Rest               = 0x0002,
        Text               = 0x0004,
        SystemExclusive    = 0x0008,
        Controller         = 0x0010,
        ProgramChange      = 0x0020,
        PitchBend          = 0x0040,
        ChannelPressure    = 0x0080,
        KeyPressure        = 0x0100,
        Indication         = 0x0200,
        Other              = 0x0400
    };

public:
    EventView(RosegardenGUIDoc *doc,
              std::vector<Segment *> segments,
              TQWidget *parent);

    virtual ~EventView();

    virtual bool applyLayout(int staffNo = -1);

    virtual void refreshSegment(Segment *segment,
                                timeT startTime = 0,
                                timeT endTime = 0);

    virtual void updateView();

    virtual void setupActions();
    virtual void initStatusBar();
    virtual TQSize getViewSize(); 
    virtual void setViewSize(TQSize);

    // Set the button states to the current filter positions
    //
    void setButtonsToFilter();

    // Menu creation and show
    //
    void createMenu();

public slots:

    // standard slots
    virtual void slotEditCut();
    virtual void slotEditCopy();
    virtual void slotEditPaste();

    // other edit slots
    void slotEditDelete();
    void slotEditInsert();
    void slotEditEvent();
    void slotEditEventAdvanced();

    void slotFilterSelection();
    void slotSelectAll();
    void slotClearSelection();

    void slotMusicalTime();
    void slotRealTime();
    void slotRawTime();

    // Show RMB menu
    //
    void slotPopupMenu(TQListViewItem*, const TQPoint&, int);
    void slotMenuActivated(int);

    // on double click on the event list
    //
    void slotPopupEventEditor(TQListViewItem*);

    // Change filter parameters
    //
    void slotModifyFilter(int);

    virtual void eventAdded(const Segment *, Event *) { }
    virtual void eventRemoved(const Segment *, Event *);
    virtual void endMarkerTimeChanged(const Segment *, bool) { }
    virtual void segmentDeleted(const Segment *);

signals:
    void editTriggerSegment(int);

protected slots:
    virtual void slotSaveOptions();

    void slotEditTriggerName();
    void slotEditTriggerPitch();
    void slotEditTriggerVelocity();
    void slotTriggerTimeAdjustChanged(int);
    void slotTriggerRetuneChanged();

protected:

    virtual void readOptions();
    void makeInitialSelection(timeT);
    TQString makeTimeString(timeT time, int timeMode);
    TQString makeDurationString(timeT time,
                               timeT duration, int timeMode);
    virtual Segment *getCurrentSegment();

    virtual void updateViewCaption();

    //--------------- Data members ---------------------------------

    bool         m_isTriggerSegment;
    TQLabel      *m_triggerName;
    TQLabel      *m_triggerPitch;
    TQLabel      *m_triggerVelocity;

    KListView   *m_eventList;
    int          m_eventFilter;

    static int   m_lastSetEventFilter;

    TQButtonGroup   *m_filterGroup;
    TQCheckBox      *m_noteCheckBox;
    TQCheckBox      *m_textCheckBox;
    TQCheckBox      *m_sysExCheckBox;
    TQCheckBox      *m_programCheckBox;
    TQCheckBox      *m_controllerCheckBox;
    TQCheckBox      *m_restCheckBox;
    TQCheckBox      *m_pitchBendCheckBox;
    TQCheckBox      *m_keyPressureCheckBox;
    TQCheckBox      *m_channelPressureCheckBox;
    TQCheckBox      *m_indicationCheckBox;
    TQCheckBox      *m_otherCheckBox;

    std::vector<int> m_listSelection;
    std::set<Event *> m_deletedEvents; // deleted since last refresh

    TQPopupMenu     *m_menu;

};


}

#endif
