/***************************************************************************
 *   Copyright (C) 2005-2007 Nicolas Hadacek <hadacek@kde.org>             *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/
#include "misc_gui.h"

#include <tqapplication.h>
#include <tqpushbutton.h>
#include <tqtimer.h>
#include <tqwidgetstack.h>
#include <tqobjectlist.h>
#include <tqpainter.h>
#include <tqheader.h>
#include <tqmetaobject.h>
#include <tqvariant.h>
#include <tqpopupmenu.h>

#include <kcursor.h>
#include <kiconloader.h>
#include <kmessagebox.h>
#include <kaction.h>
#include <ktabbar.h>

#include "dialog.h"
#include "common/common/number.h"
#include "common/common/misc.h"
#include "common/gui/number_gui.h"

//-----------------------------------------------------------------------------
bool PBusyCursor::_overridePaused = false;

void PBusyCursor::start()
{
  TQApplication::setOverrideCursor(KCursor::waitCursor(), true);
}

void PBusyCursor::stop()
{
  TQApplication::restoreOverrideCursor();
}

void PBusyCursor::pause()
{
  _overridePaused = TQApplication::overrideCursor();
  stop();
}

void PBusyCursor::restore()
{
  if (_overridePaused) start();
}

//-----------------------------------------------------------------------------
void MessageBox::information(const TQString &text, Log::ShowMode show, const TQString &dontShowAgainName)
{
  if ( show==Log::DontShow ) return;
  PBusyCursor::pause();
  KMessageBox::information(tqApp->mainWidget(), text, TQString(), dontShowAgainName, KMessageBox::Notify | KMessageBox::AllowLink);
  PBusyCursor::restore();
}

void MessageBox::detailedSorry(const TQString &text, const TQString &details, Log::ShowMode show)
{
  if ( show==Log::DontShow ) return;
  PBusyCursor::pause();
  if ( details.isEmpty() ) KMessageBox::sorry(tqApp->mainWidget(), text, TQString(), KMessageBox::Notify | KMessageBox::AllowLink);
  else KMessageBox::detailedSorry(tqApp->mainWidget(), text, details, TQString(), KMessageBox::Notify | KMessageBox::AllowLink);
  PBusyCursor::restore();
}

bool MessageBox::askContinue(const TQString &text, const KGuiItem &buttonContinue, const TQString &caption)
{
  ::PBusyCursor::pause();
  int res = KMessageBox::warningContinueCancel(tqApp->mainWidget(), text, caption, buttonContinue);
  ::PBusyCursor::restore();
  return ( res==KMessageBox::Continue );
}

bool MessageBox::questionYesNo(const TQString &text, const KGuiItem &yesButton,const KGuiItem &noButton, const TQString &caption)
{
  ::PBusyCursor::pause();
  int res = KMessageBox::questionYesNo(tqApp->mainWidget(), text, caption, yesButton, noButton);
  ::PBusyCursor::restore();
  return ( res==KMessageBox::Yes );
}

MessageBox::Result MessageBox::questionYesNoCancel(const TQString &text, const KGuiItem &yesButton, const KGuiItem &noButton,
                                                        const TQString &caption)
{
  ::PBusyCursor::pause();
  int res = KMessageBox::questionYesNoCancel(tqApp->mainWidget(), text, caption, yesButton, noButton);
  ::PBusyCursor::restore();
  if ( res==KMessageBox::Yes ) return Yes;
  if ( res==KMessageBox::No ) return No;
  return Cancel;
}

void MessageBox::text(const TQString &text, Log::ShowMode show)
{
  if ( show==Log::DontShow ) return;
  PBusyCursor::pause();
  TextEditorDialog dialog(text, TQString(), false, tqApp->mainWidget());
  dialog.exec();
  PBusyCursor::restore();
}

//----------------------------------------------------------------------------
PopupButton::PopupButton(TQWidget *parent, const char *name)
  : KPushButton(parent, name)
{
  init();
}

PopupButton::PopupButton(const TQString &text, TQWidget *parent, const char *name)
  : KPushButton(text, parent, name)
{
  init();
}

void PopupButton::init()
{
  _separator = false;
  setFlat(true);
  TQPopupMenu *popup = new TQPopupMenu(this);
  connect(popup, TQT_SIGNAL(activated(int)), TQT_SIGNAL(activated(int)));
  setPopup(popup);
}

void PopupButton::appendAction(KAction *action)
{
  if ( _separator && popup()->count()!=0 ) popup()->insertSeparator();
  _separator = false;
  action->plug(popup());
}

void PopupButton::appendAction(const TQString &label, const TQString &icon,
                               TQObject *receiver, const char *slot)
{
  appendAction(new KAction(label, icon, 0, receiver, slot, (KActionCollection *)0));
}

int PopupButton::appendItem(const TQString &label, const TQString &icon, int id)
{
  KIconLoader loader;
  TQPixmap pixmap = loader.loadIcon(icon, KIcon::Small);
  return appendItem(label, pixmap, id);
}

int PopupButton::appendItem(const TQString &label, const TQPixmap &icon, int id)
{
  if ( _separator && popup()->count()!=0 ) popup()->insertSeparator();
  _separator = false;
  return popup()->insertItem(icon, label, id);
}

//-----------------------------------------------------------------------------
Splitter::Splitter(const TQValueList<int> &defaultSizes, Qt::Orientation o, TQWidget *parent, const char *name)
  : TQSplitter(o, parent, name), _defaultSizes(defaultSizes)
{
  Q_ASSERT(name);
  setOpaqueResize(true);
  TQTimer::singleShot(0, this, TQT_SLOT(updateSizes()));
}

Splitter::~Splitter()
{
  GuiConfig gc;
  gc.writeEntry(TQString(name()) + "_sizes", sizes());
}

void Splitter::updateSizes()
{
  GuiConfig gc;
  TQValueList<int> sizes = gc.readIntListEntry(TQString(name()) + "_sizes");
  for (uint i=sizes.count(); i<_defaultSizes.count(); i++) sizes.append(_defaultSizes[i]);
  setSizes(sizes);
}

//-----------------------------------------------------------------------------
TabBar::TabBar(TQWidget *parent, const char *name)
  : KTabBar(parent, name), _ignoreWheelEvent(false)
{}

void TabBar::wheelEvent(TQWheelEvent *e)
{
  if (_ignoreWheelEvent) TQApplication::sendEvent(parent(), e); // #### not sure why ignoring is not enough...
  else KTabBar::wheelEvent(e);
}

TabWidget::TabWidget(TQWidget *parent, const char *name)
  : KTabWidget(parent, name)
{
  setTabBar(new TabBar(this));
}

void TabWidget::setIgnoreWheelEvent(bool ignore)
{
  static_cast<TabBar *>(tabBar())->_ignoreWheelEvent = ignore;
}

void TabWidget::wheelEvent(TQWheelEvent *e)
{
  if (static_cast<TabBar *>(tabBar())->_ignoreWheelEvent) e->ignore();
  else KTabWidget::wheelEvent(e);
}

void TabWidget::setTabBar(TabBar *tabbar)
{
  KTabWidget::setTabBar(tabbar);
  connect(tabBar(), TQT_SIGNAL(contextMenu( int, const TQPoint & )), TQT_SLOT(contextMenu( int, const TQPoint & )));
  connect(tabBar(), TQT_SIGNAL(mouseDoubleClick( int )), TQT_SLOT(mouseDoubleClick( int )));
  connect(tabBar(), TQT_SIGNAL(mouseMiddleClick( int )), TQT_SLOT(mouseMiddleClick( int )));
  connect(tabBar(), TQT_SIGNAL(initiateDrag( int )), TQT_SLOT(initiateDrag( int )));
  connect(tabBar(), TQT_SIGNAL(testCanDecode(const TQDragMoveEvent *, bool & )), TQT_SIGNAL(testCanDecode(const TQDragMoveEvent *, bool & )));
  connect(tabBar(), TQT_SIGNAL(receivedDropEvent( int, TQDropEvent * )), TQT_SLOT(receivedDropEvent( int, TQDropEvent * )));
  connect(tabBar(), TQT_SIGNAL(moveTab( int, int )), TQT_SLOT(moveTab( int, int )));
  connect(tabBar(), TQT_SIGNAL(closeRequest( int )), TQT_SLOT(closeRequest( int )));
  connect(tabBar(), TQT_SIGNAL(wheelDelta( int )), TQT_SLOT(wheelDelta( int )));
}

//-----------------------------------------------------------------------------
ComboBox::ComboBox(TQWidget *parent, const char *name)
  : TQComboBox(parent, name), _ignoreWheelEvent(false)
{}

void ComboBox::wheelEvent(TQWheelEvent *e)
{
  if (_ignoreWheelEvent) TQApplication::sendEvent(parent(), e); // #### not sure why ignoring is not enough...
  else TQComboBox::wheelEvent(e);
}
