/*
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
*/

/*
  Copyright (C) 2006 Eike Hein <hein@kde.org>
*/

#ifndef VIEWTREE_H
#define VIEWTREE_H

#include <klistview.h>


class ViewTreeItem;
class ChatWindow;

class ViewTree : public KListView
{
    Q_OBJECT
  TQ_OBJECT

    public:
        explicit ViewTree(TQWidget *parent);
        ~ViewTree();

        void selectFirstView(bool select);
        void addView(const TQString& name, ChatWindow* view, const TQIconSet &iconset, bool select = false, ChatWindow* afterView = 0);
        void setViewName(ChatWindow* view, const TQString& name);
        void setViewColor(ChatWindow* view, TQColor color);
        void setViewIcon(ChatWindow* view, const TQIconSet &iconset);

        void moveViewUp(ChatWindow* view);
        void moveViewDown(ChatWindow* view);
        bool canMoveViewUp(ChatWindow* view);
        bool canMoveViewDown(ChatWindow* view);

        TQPtrList<ChatWindow> getSortedViewList();

    public slots:
        void updateAppearance();
        void removeView(ChatWindow* view);
        void selectView(ChatWindow* view);
        void unHighlight();

    signals:
        void setViewTreeShown(bool show);
        void showView(ChatWindow* view);
        void closeView(ChatWindow* view);
        void showViewContextMenu(TQWidget* widget, const TQPoint& point);
        void sizeChanged();
        void syncTabBarToTree();

    protected:
        void contentsMousePressEvent(TQMouseEvent* e);
        void contentsMouseReleaseEvent(TQMouseEvent* e);
        void contentsMouseMoveEvent(TQMouseEvent* e);
        void contentsWheelEvent(TQWheelEvent* e);
        void contentsContextMenuEvent(TQContextMenuEvent* e);
        void keyPressEvent(TQKeyEvent* e);

        void resizeEvent(TQResizeEvent* e);

        void findDrop(const TQPoint &pos, TQListViewItem *&parent, TQListViewItem *&after);
        TQDragObject* dragObject();

        void paintEmptyArea(TQPainter* p, const TQRect& rect);

    private slots:
        void announceSelection(TQListViewItem* item);
        void slotAboutToMoveView();
        void slotMovedView();
        void enableCloseButton();

    private:
        class ToolTip;
        ViewTree::ToolTip* m_toolTip;

        void toggleSeparator();
        void selectUpper(bool wrap = false);
        void selectLower(bool wrap = false);

        ViewTreeItem* getItemForView(ChatWindow* view);
        ViewTreeItem* getParentItemForView(ChatWindow* view);
        ViewTreeItem* getLastChild(TQListViewItem* parent);

        bool canMoveItemUp(ViewTreeItem* item);
        bool canMoveItemDown(ViewTreeItem* item);

        bool isAboveIcon(TQPoint point, ViewTreeItem* item);
        void hideCloseButtons(ViewTreeItem* exception = 0);

        ViewTreeItem* m_separator;
        int m_specialViewCount;

        // Controls whether or not to select the first view added
        // to the tree.
        bool m_selectFirstView;

        // Used in mouse handling to determine whether both the press
        // and the release event occurred over a close button.
        bool m_pressedAboveCloseButton;
        ViewTreeItem* m_closeButtonItem;
        TQTimer* m_enableCloseButtonTimer;

        // Used for middle-click close.
        ViewTreeItem* m_middleClickItem;
};

#endif
