/*
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
*/

/*
  Copyright (C) 2002 Dario Abatianni <eisfuchs@tigress.com>
  Copyright (C) 2006-2008 Eike Hein <hein@kde.org>
*/

#ifndef CHATWINDOW_H
#define CHATWINDOW_H

#include "identity.h"
#include "common.h"

#include <tqvbox.h>
#include <tqfile.h>


class IRCView;
class Server;
class KonversationMainWindow;

class ChatWindow : public TQVBox
{
    Q_OBJECT
  TQ_OBJECT

    public:
        explicit ChatWindow(TQWidget* parent);
        ~ChatWindow();

        enum WindowType
        {
            Status=0,
            Channel,
            Query,
            DccChat,
            DccTransferPanel,
            RawLog,
            Notice,
            SNotice,
            ChannelList,
            Konsole,
            UrlCatcher,
            NicksOnline,
            LogFileReader
        };

        /** This should be called and set with a non-null server as soon
         *  as possibly after ChatWindow is created.
         *  @param newServer The server to set it to.
         */
        virtual void setServer(Server* newServer);
        /** This should be called if setServer is not called - e.g.
         *  in the case of konsolepanel.  This should be set as soon
         *  as possible after creation.
         */

        /** Get the server this is linked to.
         *  @return The server it is associated with, or null if none.
         */
        Server* getServer();
        void setTextView(IRCView* newView);
        IRCView* getTextView() const;
        void setLog(bool activate);

        TQString getName();

        void setType(WindowType newType);
        WindowType getType();

        virtual void append(const TQString& nickname,const TQString& message);
        virtual void appendRaw(const TQString& message, bool suppressTimestamps=false);
        virtual void appendQuery(const TQString& nickname,const TQString& message, bool inChannel = false);
        virtual void appendAction(const TQString& nickname,const TQString& message);
        virtual void appendServerMessage(const TQString& type,const TQString& message, bool parseURL = true);
        virtual void appendCommandMessage(const TQString& command, const TQString& message, bool important = true,
            bool parseURL = true, bool self = false);
        virtual void appendBacklogMessage(const TQString& firstColumn,const TQString& message);

        TQWidget* parentWidget;

        virtual TQString getTextInLine();
        /** Clean up and close this tab.  Return false if you want to cancel the close. */
        virtual bool closeYourself(bool askForConfirmation = true);
        /** Reimplement this to return true in all classes that /can/ become front view.
         */
        virtual bool canBeFrontView();

        /** Reimplement this to return true in all classes that you can search in - i.e. use "Edit->Find Text" in.
         */
        virtual bool searchView();

        virtual bool notificationsEnabled() { return m_notificationsEnabled; }

        virtual bool eventFilter(TQObject* watched, TQEvent* e);

        TQString logFileName() { return logfile.name(); }

        virtual void setChannelEncoding(const TQString& /* encoding */) {}
        virtual TQString getChannelEncoding() { return TQString(); }
        virtual TQString getChannelEncodingDefaultDesc() { return TQString(); }
        bool isChannelEncodingSupported() const;

        /** Force updateInfo(info) to be emitted.
         *  Useful for when this tab has just gained focus
         */
        virtual void emitUpdateInfo();

        /** child classes have to override this and return true if they want the
         *  "insert character" item on the menu to be enabled.
         */
        virtual bool isInsertSupported() { return false; }

        /** child classes have to override this and return true if they want the
         *  "irc color" item on the menu to be enabled.
         */
        virtual bool areIRCColorsSupported() {return false; }

        Konversation::TabNotifyType currentTabNotification() { return m_currentTabNotify; }
        TQColor highlightColor();

        signals:
        void nameChanged(ChatWindow* view, const TQString& newName);
        //void online(ChatWindow* myself, bool state);
        /** Emit this signal when you want to change the status bar text for this tab.
         *  It is ignored if this tab isn't focused.
         */
        void updateInfo(const TQString &info);
        void updateTabNotification(ChatWindow* chatWin, const Konversation::TabNotifyType& type);

        void setStatusBarTempText(const TQString&);
        void clearStatusBarTempText();

        void closing(ChatWindow* myself);

    public slots:
        void updateAppearance();

        void logText(const TQString& text);

        /**
         * This is called when a chat window gains focus.
         * It enables and disables the appropriate menu items,
         * then calls childAdjustFocus.
         * You can call this manually to focus this tab.
         */
        void adjustFocus();

        virtual void appendInputText(const TQString&, bool fromCursor);
        virtual void indicateAway(bool away);


        virtual void setNotificationsEnabled(bool enable) { m_notificationsEnabled = enable; }
        void activateTabNotification(Konversation::TabNotifyType type);
        void resetTabNotification();

    protected slots:
        ///Used to disable functions when not connected
        virtual void serverOnline(bool online);

    protected:

        /** Some children may handle the name themselves, and not want this public.
         *  Increase the visibility in the subclass if you want outsiders to call this.
         *  The name is the string that is shown in the tab.
         *  @param newName The name to show in the tab
         */
        virtual void setName(const TQString& newName);

        /** Called from adjustFocus */
        virtual void childAdjustFocus() = 0;

        void setLogfileName(const TQString& name);
        void setChannelEncodingSupported(bool enabled);
        void cdIntoLogPath();

        int spacing();
        int margin();

        bool log;
        bool firstLog;
        TQString name;
        TQString logName;

        TQFont font;

        IRCView* textView;
        /** A pointer to the server this chatwindow is part of.
         *  Not always non-null - e.g. for konsolepanel
         */
        Server* m_server;
        TQFile logfile;
        WindowType type;

        bool m_notificationsEnabled;

        bool m_channelEncodingSupported;

        Konversation::TabNotifyType m_currentTabNotify;
};
#endif
