/* This file is part of the KDE project
   Copyright (C)  2001, 2002 Montel Laurent <lmontel@mandrakesoft.com>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
*/

#include "KoFontDia.h"

#include <tqvbox.h>
#include <tqhbox.h>

#include "KoRichText.h"

#include <kcolordialog.h>
#include <klocale.h>
#include <kdebug.h>

#include <tqgroupbox.h>
#include <tqpushbutton.h>
#include <tqtabwidget.h>
#include <tqlabel.h>
#include <tqcombobox.h>
#include <tqradiobutton.h>
#include <tqcheckbox.h>
#include <knuminput.h>
#include <KoGlobal.h>
#include <tqvgroupbox.h>
#include <tqvbox.h>
#include <tqhgroupbox.h>
#include <tqhbuttongroup.h>
#include <kcolorbutton.h>
#include <kiconloader.h>

KoFontDia::KoFontDia( const KoTextFormat& initialFormat,
    KSpell2::Broker::Ptr broker, TQWidget* parent, const char* name )
    : KDialogBase( parent, name, true,
    i18n("Select Font"), Ok|Cancel|User1|Apply, Ok ),
    m_initialFormat(initialFormat),
    m_changedFlags(KoTextFormat::NoFlags)
{
    setButtonText( KDialogBase::User1, i18n("&Reset") );

    TQVBox *mainWidget = new TQVBox( this );
    TQHBox *mainHBox = new TQHBox( mainWidget );

    TQTabWidget *fontTabWidget = new TQTabWidget( mainHBox );

    // Font tab
    fontTab = new KoFontTab( KFontChooser::SmoothScalableFonts, this );
    fontTabWidget->addTab( fontTab, i18n( "Font" ) );

    connect( fontTab, TQT_SIGNAL( familyChanged() ), this, TQT_SLOT( slotFontFamilyChanged() ) );
    connect( fontTab, TQT_SIGNAL( boldChanged() ), this, TQT_SLOT( slotFontBoldChanged() ) );
    connect( fontTab, TQT_SIGNAL( italicChanged() ), this, TQT_SLOT( slotFontItalicChanged() ) );
    connect( fontTab, TQT_SIGNAL( sizeChanged() ), this, TQT_SLOT( slotFontSizeChanged() ) );

    //Highlighting tab
    highlightingTab = new KoHighlightingTab( this );
    fontTabWidget->addTab( highlightingTab, i18n( "Highlighting" ) );

    connect( highlightingTab, TQT_SIGNAL( underlineChanged( int ) ), this, TQT_SLOT( slotUnderlineChanged( int ) ) );
    connect( highlightingTab, TQT_SIGNAL( underlineStyleChanged( int ) ), this, TQT_SLOT( slotUnderlineStyleChanged( int ) ) );
    connect( highlightingTab, TQT_SIGNAL( underlineColorChanged( const TQColor & ) ), this, TQT_SLOT( slotUnderlineColorChanged( const TQColor & ) ) );
    connect( highlightingTab, TQT_SIGNAL( strikethroughChanged( int ) ), this, TQT_SLOT( slotStrikethroughChanged( int ) ) );
    connect( highlightingTab, TQT_SIGNAL( strikethroughStyleChanged( int ) ), this, TQT_SLOT( slotStrikethroughStyleChanged( int ) ) );
    connect( highlightingTab, TQT_SIGNAL( wordByWordChanged( bool ) ), this, TQT_SLOT( slotWordByWordChanged( bool ) ) );
    connect( highlightingTab, TQT_SIGNAL( capitalisationChanged( int ) ), this, TQT_SLOT( slotCapitalisationChanged( int ) ) );

    //Decoratio tab
    decorationTab = new KoDecorationTab( this );
    fontTabWidget->addTab( decorationTab, i18n( "Decoration" ) );

    connect( decorationTab, TQT_SIGNAL( fontColorChanged( const TQColor& ) ), this, TQT_SLOT( slotFontColorChanged( const TQColor& ) ) );
    connect( decorationTab, TQT_SIGNAL( backgroundColorChanged( const TQColor& ) ), this, TQT_SLOT( slotBackgroundColorChanged( const TQColor& ) ) );
    connect( decorationTab, TQT_SIGNAL( shadowColorChanged( const TQColor& ) ), this, TQT_SLOT( slotShadowColorChanged( const TQColor& ) ) );
    connect( decorationTab, TQT_SIGNAL( shadowDistanceChanged( double ) ), this, TQT_SLOT( slotShadowDistanceChanged( double ) ) );
    connect( decorationTab, TQT_SIGNAL( shadowDirectionChanged( int ) ), this, TQT_SLOT( slotShadowDirectionChanged( int ) ) );

    //Layout tab
    layoutTab = new KoLayoutTab( true, this );
    fontTabWidget->addTab( layoutTab, i18n( "Layout" ) );
    connect( layoutTab, TQT_SIGNAL( subSuperScriptChanged() ), this, TQT_SLOT( slotSubSuperChanged() ) );;
    connect( layoutTab, TQT_SIGNAL( offsetChanged( int ) ), this, TQT_SLOT( slotOffsetChanged( int ) ) );
    connect( layoutTab, TQT_SIGNAL( relativeSizeChanged( double ) ), this, TQT_SLOT( slotRelativeSizeChanged( double ) ) );
    connect( layoutTab, TQT_SIGNAL( hyphenationChanged( bool ) ), this, TQT_SLOT( slotHyphenationChanged( bool ) ) );

    //Language tab
    languageTab = new KoLanguageTab( broker, this );
    fontTabWidget->addTab( languageTab, i18n( "Language" ) );
    connect( languageTab, TQT_SIGNAL( languageChanged() ), this, TQT_SLOT( slotLanguageChanged() ) );

    //Related properties List View
    //relatedPropertiesListView = new KListView( mainHBox );

    //Preview
    fontDiaPreview = new KoFontDiaPreview( mainWidget );

    setMainWidget( mainWidget );

    init();
}

void KoFontDia::init()
{
    connect( this, TQT_SIGNAL( user1Clicked() ), this, TQT_SLOT(slotReset()) );

    slotReset();
}

KoTextFormat KoFontDia::newFormat() const
{
    return KoTextFormat( fontTab->getSelection(),
                         layoutTab->getSubSuperScript(),
                         decorationTab->getTextColor(),
                         decorationTab->getBackgroundColor(),
                         highlightingTab->getUnderlineColor(),
                         highlightingTab->getUnderline(),
                         highlightingTab->getUnderlineStyle(),
                         highlightingTab->getStrikethrough(),
                         highlightingTab->getStrikethroughStyle(),
                         highlightingTab->getCapitalisation(),
                         languageTab->getLanguage(),
                         layoutTab->getRelativeTextSize(),
                         layoutTab->getOffsetFromBaseline(),
                         highlightingTab->getWordByWord(),
                         layoutTab->getAutoHyphenation(),
                         decorationTab->getShadowDistanceX(),
                         decorationTab->getShadowDistanceY(),
                         decorationTab->getShadowColor()
			);
}

void KoFontDia::slotApply()
{
    emit applyFont();
}

void KoFontDia::slotOk()
{
    slotApply();
    KDialogBase::slotOk();
}

void KoFontDia::slotReset()
{
    fontTab->setSelection( m_initialFormat.font() );
    highlightingTab->setUnderline( m_initialFormat.underlineType() );
    highlightingTab->setUnderlineStyle( m_initialFormat.underlineStyle() );
    highlightingTab->setUnderlineColor( m_initialFormat.textUnderlineColor() );
    highlightingTab->setStrikethrough( m_initialFormat.strikeOutType() );
    highlightingTab->setStrikethroughStyle( m_initialFormat.strikeOutStyle() );
    highlightingTab->setWordByWord( m_initialFormat.wordByWord() );
    highlightingTab->setCapitalisation( m_initialFormat.attributeFont() );
    decorationTab->setTextColor( m_initialFormat.color() );
    decorationTab->setBackgroundColor( m_initialFormat.textBackgroundColor() );
    decorationTab->setShadow( m_initialFormat.shadowDistanceX(), m_initialFormat.shadowDistanceY(), m_initialFormat.shadowColor() );
    layoutTab->setSubSuperScript( m_initialFormat.vAlign(), m_initialFormat.offsetFromBaseLine(), m_initialFormat.relativeTextSize() );
    layoutTab->setAutoHyphenation( m_initialFormat.hyphenation() );
    languageTab->setLanguage( m_initialFormat.language() );
}

void KoFontDia::slotFontFamilyChanged()
{
    m_changedFlags |= KoTextFormat::Family;
    fontDiaPreview->setFont( fontTab->getSelection() );
}

void KoFontDia::slotFontBoldChanged()
{
    m_changedFlags |= KoTextFormat::Bold;
    fontDiaPreview->setFont( fontTab->getSelection() );
}

void KoFontDia::slotFontItalicChanged()
{
    m_changedFlags |= KoTextFormat::Italic;
    fontDiaPreview->setFont( fontTab->getSelection() );
}

void KoFontDia::slotFontSizeChanged()
{
    m_changedFlags |= KoTextFormat::Size;
    fontDiaPreview->setFont( fontTab->getSelection() );
}

void KoFontDia::slotFontColorChanged( const TQColor& color )
{
    m_changedFlags |= KoTextFormat::Color;
    fontDiaPreview->setFontColor( color );
}

void KoFontDia::slotBackgroundColorChanged( const TQColor& color )
{
    m_changedFlags |= KoTextFormat::TextBackgroundColor;
    fontDiaPreview->setBackgroundColor( color );
}

void KoFontDia::slotCapitalisationChanged( int item )
{
    m_changedFlags |= KoTextFormat::Attribute;
    fontDiaPreview->setCapitalisation( item );
}

void KoFontDia::slotUnderlineChanged( int item )
{
    m_changedFlags |= KoTextFormat::ExtendUnderLine;
    if ( !item ) fontDiaPreview->setUnderlining( item, 0, TQt::black, false );
    else fontDiaPreview->setUnderlining( item, highlightingTab->getUnderlineStyle(), highlightingTab->getUnderlineColor(), highlightingTab->getWordByWord() );
}

void KoFontDia::slotUnderlineStyleChanged( int item )
{
    m_changedFlags |= KoTextFormat::ExtendUnderLine;
    if ( !highlightingTab->getUnderline() ) fontDiaPreview->setUnderlining( 0, 0, TQt::black, false );
    else fontDiaPreview->setUnderlining( highlightingTab->getUnderline(), item, highlightingTab->getUnderlineColor(), highlightingTab->getWordByWord() );
}

void KoFontDia::slotUnderlineColorChanged( const TQColor &color )
{
    m_changedFlags |= KoTextFormat::ExtendUnderLine;
    if ( !highlightingTab->getUnderline() ) fontDiaPreview->setUnderlining( 0, 0, TQt::black, false );
    else fontDiaPreview->setUnderlining( highlightingTab->getUnderline(), highlightingTab->getUnderlineStyle(), color, highlightingTab->getWordByWord() );
}

void KoFontDia::slotWordByWordChanged( bool state )
{
    m_changedFlags |= KoTextFormat::WordByWord;
    fontDiaPreview->setWordByWord( state );
}

void KoFontDia::slotStrikethroughChanged( int item )
{
    m_changedFlags |= KoTextFormat::StrikeOut;
    if ( !item ) fontDiaPreview->setStrikethrough( item, 0, false );
    else fontDiaPreview->setStrikethrough( item, highlightingTab->getStrikethroughStyle(), highlightingTab->getWordByWord() );
}

void KoFontDia::slotStrikethroughStyleChanged( int item )
{
    m_changedFlags |= KoTextFormat::StrikeOut;
    if ( !highlightingTab->getStrikethrough() ) fontDiaPreview->setStrikethrough( 0, 0, false );
    else fontDiaPreview->setStrikethrough( highlightingTab->getStrikethrough(), item, highlightingTab->getWordByWord() );
}

void KoFontDia::slotShadowDistanceChanged( double )
{
    m_changedFlags |= KoTextFormat::ShadowText;
    fontDiaPreview->setShadow( decorationTab->getShadowDistanceX(), decorationTab->getShadowDistanceY(), decorationTab->getShadowColor() );
}

void KoFontDia::slotShadowDirectionChanged( int )
{
    m_changedFlags |= KoTextFormat::ShadowText;
    fontDiaPreview->setShadow( decorationTab->getShadowDistanceX(), decorationTab->getShadowDistanceY(), decorationTab->getShadowColor() );
}

void KoFontDia::slotShadowColorChanged( const TQColor & )
{
    m_changedFlags |= KoTextFormat::ShadowText;
    fontDiaPreview->setShadow( decorationTab->getShadowDistanceX(), decorationTab->getShadowDistanceY(), decorationTab->getShadowColor() );
}

void KoFontDia::slotSubSuperChanged()
{
    m_changedFlags |= KoTextFormat::VAlign;
    fontDiaPreview->setSubSuperscript( layoutTab->getSubSuperScript(), layoutTab->getOffsetFromBaseline(), layoutTab->getRelativeTextSize() );
}

void KoFontDia::slotOffsetChanged( int offset )
{
    m_changedFlags |= KoTextFormat::OffsetFromBaseLine;
    fontDiaPreview->setSubSuperscript( layoutTab->getSubSuperScript(), offset, layoutTab->getRelativeTextSize() );
}

void KoFontDia::slotRelativeSizeChanged( double relativeSize )
{
    m_changedFlags |= KoTextFormat::VAlign;
    fontDiaPreview->setSubSuperscript( layoutTab->getSubSuperScript(), layoutTab->getOffsetFromBaseline(), relativeSize );
}

void KoFontDia::slotHyphenationChanged( bool )
{
    m_changedFlags |= KoTextFormat::Hyphenation;
}

void KoFontDia::slotLanguageChanged()
{
    m_changedFlags |= KoTextFormat::Language;
}

#include "KoFontDia.moc"

