/*
 * Kivio - Visual Modelling and Flowcharting
 * Copyright (C) 2000 theKompany.com
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */
#include "tkaction.h"
#include "tktoolbarbutton.h"
#include "tkcombobox.h"

#include <tqlabel.h>
#include <tqlayout.h>

#include <ktoolbar.h>
#include <kiconloader.h>

#define SET_FOR_ALL_CONTAINER(WIDGET_TYPE,METHOD_NAME,VALUE)             \
  for( int id = 0; id < containerCount(); ++id ) {                       \
    TQWidget* w = container(id);                                          \
    if ( w->inherits("KToolBar") ) {                                     \
      TQWidget* r = static_cast<KToolBar*>(w)->getWidget(itemId(id));     \
      if (qstrcmp(r->name(),"KTToolBarLayout")==0)                       \
        r = (TQWidget*)r->child("widget");                                \
      if ( r && r->inherits(#WIDGET_TYPE) ) {                            \
        WIDGET_TYPE* b = static_cast<WIDGET_TYPE*>(r);                   \
        b->METHOD_NAME(VALUE);                                         \
      }                                                                  \
    }                                                                    \
  }

TKAction::TKAction(TQObject* parent, const char* name)
: KAction( "", 0, parent, name )
{
  m_imode = TK::IconOnly;
}

TKAction::~TKAction()
{
}

int TKAction::plug(TQWidget* widget, int index)
{
  if ( widget->inherits("KToolBar") ) {
    KToolBar* bar = static_cast<KToolBar*>(widget);
    int id_ = KAction::getToolButtonID();
    KInstance *instance;

    if ( parentCollection() )
      instance = parentCollection()->instance();
    else
      instance = KGlobal::instance();

    TKToolBarButton* b = new TKToolBarButton(icon(),plainText(),bar,name(),instance);
    // we don't need clicked() and buttonClicked(), do we?
    // connect(b,TQT_SIGNAL(clicked()),TQT_SLOT(slotActivated()));
    b->setIconMode(m_imode);
    initToolBarButton(b);

    bar->insertWidget( id_, 100, b, index );
    addContainer(bar,id_);
    connect( bar, TQT_SIGNAL(destroyed()), this, TQT_SLOT(slotDestroyed()) );

    return containerCount() - 1;
  }
  return KAction::plug(widget,index);
}

void TKAction::initToolBarButton(TKToolBarButton* button)
{
  connect(button,TQT_SIGNAL(buttonClicked()),TQT_SLOT(slotActivated()));
}

TK::IconMode TKAction::iconMode()
{
  return m_imode;
}

void TKAction::setIconMode(TK::IconMode mode)
{
  m_imode = mode;
  SET_FOR_ALL_CONTAINER(TKToolBarButton,setIconMode,mode)
}

void TKAction::setText(const TQString& text)
{
  KAction::setText(text);
  updateLayout();
}

void TKAction::setIcon(const TQString& icon)
{
  KAction::setIcon(icon);
  updateLayout();
}

void TKAction::updateLayout()
{
  int len = containerCount();
  for( int id = 0; id < len; ++id ) {
    TQWidget* w = container( id );
    if (w->inherits("KToolBar")) {
      TQWidget* r = static_cast<KToolBar*>(w)->getWidget(itemId(id));
      if (qstrcmp(r->name(),"KTToolBarLayout")==0) {
        updateLayout(r);
      }
    }
  }
}

TQWidget* TKAction::createLayout(TQWidget* parent, TQWidget* children)
{
  TQWidget* base = new TQWidget(parent,"KTToolBarLayout");
  TQLabel* textLabel = new TQLabel(base,"text");
  textLabel->setMinimumHeight(1);
  TQLabel* pixLabel = new TQLabel(base,"pixmap");
  children->reparent(base,TQPoint(0,0));
  children->setName("widget");
  TQHBoxLayout* layout = new TQHBoxLayout(base,0,3);
  layout->setResizeMode(TQLayout::Minimum);
  layout->addWidget(textLabel);
  layout->addWidget(pixLabel);
  layout->addWidget(children,1);

  updateLayout(base);
  return base;
}

void TKAction::updateLayout(TQWidget* base)
{
  TQLabel* textLabel = (TQLabel*)base->child("text");
  TQLabel* pixLabel = (TQLabel*)base->child("pixmap");
  TQWidget* w = (TQWidget*)base->child("widget");

  if (!textLabel || !pixLabel || !w)
    return;

  if (!text().isEmpty() && m_imode != TK::IconOnly ) {
    textLabel->setText(text());
    textLabel->show();
  } else
    textLabel->hide();

  TQPixmap pix;
  if (hasIcon())
    pix = iconSet(KIcon::Small).pixmap();

  if (!icon().isEmpty())
    pix = BarIcon(icon());

  if (!pix.isNull() && m_imode != TK::TextOnly) {
    pixLabel->setPixmap(pix);
    pixLabel->show();
  } else
    pixLabel->hide();

  base->setFixedWidth( w->sizeHint().width() +
                       (textLabel->isVisible() ? textLabel->sizeHint().width():0) +
                       (pixLabel->isVisible() ? pixLabel->sizeHint().width():0) );
}
/******************************************************************************/
TKBaseSelectAction::TKBaseSelectAction( TQObject* parent, const char* name )
: TKAction(parent,name)
{
  m_current = 0;
  m_editable = false;
}

TKBaseSelectAction::~TKBaseSelectAction()
{
}

int TKBaseSelectAction::plug(TQWidget* widget, int index)
{
  if ( widget->inherits("KToolBar") )
  {
    KToolBar* bar = static_cast<KToolBar*>( widget );
    int id_ = KAction::getToolButtonID();

    TKComboBox* cb = new TKComboBox(m_editable,bar);
    initComboBox(cb);
    cb->setMinimumWidth( cb->sizeHint().width() );
    TQWidget* base = createLayout(bar,cb);

    bar->insertWidget( id_, 100, base, index );
    addContainer( bar, id_ );

    connect( bar, TQT_SIGNAL( destroyed() ), this, TQT_SLOT( slotDestroyed() ) );

    setCurrentItem(currentItem());

    return containerCount() - 1;
  }
  return -1;
}

int TKBaseSelectAction::currentItem()
{
  return m_current;
}

void TKBaseSelectAction::initComboBox(TKComboBox* cb)
{
  connect(cb,TQT_SIGNAL(activated(int)),TQT_SLOT(slotActivated(int)));
}

void TKBaseSelectAction::setEditable(bool editable)
{
  m_editable = editable;
  SET_FOR_ALL_CONTAINER(TKComboBox,setEditable,editable)
}

bool TKBaseSelectAction::isEditable()
{
  return m_editable;
}

void TKBaseSelectAction::setCurrentItem(int index)
{
  m_current = index;
  SET_FOR_ALL_CONTAINER(TKComboBox,setCurrentItem,index)
}

void TKBaseSelectAction::slotActivated(int id)
{
  if ( m_current == id )
    return;

  m_current = id;
  setCurrentItem(id);
  activate(id);
}

void TKBaseSelectAction::activate(int id)
{
  emit activated(id);
}
/******************************************************************************/
TKSelectAction::TKSelectAction( TQObject* parent, const char* name )
: TKBaseSelectAction(parent,name)
{
}

TKSelectAction::~TKSelectAction()
{
}

void TKSelectAction::initComboBox(TKComboBox* cb)
{
  TKBaseSelectAction::initComboBox(cb);
  connect(cb,TQT_SIGNAL(activated(const TQString&)),TQT_SLOT(slotActivated(const TQString&)));
  cb->insertStringList(items());
}

void TKSelectAction::slotActivated(const TQString& text)
{
  emit activated(text);
}

void TKSelectAction::setItems(const TQStringList& lst )
{
  m_list = lst;
  m_current = -1;

  SET_FOR_ALL_CONTAINER(TKComboBox,clear, )
  SET_FOR_ALL_CONTAINER(TKComboBox,insertStringList,lst)

  // Disable if empty and not editable
  setEnabled ( lst.count() > 0 || m_editable );
}

TQStringList TKSelectAction::items() const
{
  return m_list;
}

void TKSelectAction::clear()
{
  SET_FOR_ALL_CONTAINER(TKComboBox,clear, )
}

void TKSelectAction::setEditText(const TQString& text)
{
  SET_FOR_ALL_CONTAINER(TKComboBox,setEditText,text)
}

#undef SET_FOR_ALL_CONTAINER
#include "tkaction.moc"
